#include <stdio.h>
#include <string.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                             READ_DREC SUBROUTINE                         *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This is the read routine used by all instruments.  The sensor, sweep, *
 *  and calibration data for the data set of interest is retrieved, if      *
 *  applicable, and placed in the data structure that is allocated to hold  *
 *  the values to be returned to the user.  All data is returned in         *
 *  telemetry units.  In this way, one call for the sensor of interest can  *
 *  be made to this routine instead of one call per parameter for the same  *
 *  sensor. The returned data can be converted to other units by calling    *
 *  the routine CONVERT_TO_UNITS(). For this routine, the fwd flag          *
 *  determines when the NEXT time sample is to be processed. For a sweeping *
 *  instrument, N_SAMPLE data elements are returned.  For a scalar          *
 *  instrument, either 1 or all N_SAMPLE elements are returned, depending   *
 *  upon the FULL_SWP flag. If ALL sensors for a given data set are being   *
 *  requested, the user should set the fwd flag to 1 for the last sensor in *
 *  the group, the rest should use a 0 for the fwd flag.  This ensures that *
 *  all data taken at the same time will be processed before advancing to   *
 *  the next time sample.                                                   *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    unsigned long data_key     key which uniquely identifies the data set *
 *                               being processed                            *
 *    char *exten                filename extension for the data to be used *
 *    unsigned short vnum        version number to be associated with this  *
 *                               combination (allows for multiple opens)    *
 *    void *UDF                  memory location ptr for the structure that *
 *                               holds returned data values (read_drec)     *
 *    short SeN                  the sensor or matrix containing the data   *
 *    short CoL                  the matrix column containing the data      *
 *    char fwd                   flag that indicates when to advance to the *
 *                               time sample                                *
 *    char full_swp              request 1 or all values in sensor set be   *
 *                               returned (for scalar parameters only)      *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = read_drec (data_key, exten, vnum, UDF, sen, CoL, fwd, full_swp)   *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    strcpy()                   copies a string to another string variable *
 *    ir_locate_ex()             determines if requested combination has    *
 *                               already been pand points to the            *
 *                               correct structure allocated for the combo  *
 *    ir_next_idf_file()         determines if a new VIDF file needs to be  *
 *                            	 opened and utilized for the data set being *
 *                               processed                                  *
 *    ir_reset_header()          sets data elements that are dependent upon *
 *                               the information in the header file         *
 *    ir_new_data_record ()      retrieves the next data record             *
 *    ir_new_header_record ()    retrieves the next header record           *
 *    ir_read_drec_times ()      fills in the time values returned by this  *
 *                               module                                     *
 *    ir_read_drec_time_advance () advances to the next time sample         *
 *    ir_read_drec_data_values () retrieves data for sensor requested       *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure holding information concerning   *
 *        ginfo                  the experiment that is being processed     *
 *    char ReSetSen              flag indicating if a new header was read,  *
 *                               in which case, new sensors may be returned *
 *    char block_adv             flag indicating if time advancement should *
 *                               be blocked                                 *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa      structure that holds all of the currently  *
 *                               returned data values to be processed       *
 *    struct experiment_info     a pointer to the structure that holds      *
 *          *ex                  specific experiment information            *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all *
 *                               pointers to the header and data for the    *
 *                               experiment of interest                     *
 *    register long  *l1         pointer to long variables                  *
 *    register short *s1         pointer to short variables                 *
 *    register short *s_end      loop termination variable                  *
 *    reg unsigned char *uc1     pointer to process mode values             *
 *    reg uns char *uc_end       loop termination variable                  *
 *    long SenIndex              index value that tells the position of the *
 *                               sensor within the sensors returned         *
 *    long hdr_offset            indicates which header record to access    *
 *    short rV                   holds called routine status flags          *
 *    char found_sensor          flag indicating if the requested sensor is *
 *                               in the sensors returned and is associated  *
 *                               with the time period being processed       *
 *    char reset_called          flag indicating if LOCATE_EX was called    *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

ByTe_2 read_drec (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                  void *UDF, ByTe_2 SeN, ByTe_2 CoL, 
                  ByTe_1 fwd, ByTe_1 full_swp)
{
   extern struct general_info ginfo;
   extern ByTe_1 ReSetSen, block_adv;

   struct idf_data *ExDa;
   struct experiment_info *ex;
   struct ptr_rec *ptr;
   register ByTe_4  *l1;
   register ByTe_2 *s1, *s_end;
   register u_ByTe_1 *uc1, *uc_end;
   ByTe_4 SenIndex, hdr_offset, RelCol;
   ByTe_2 rV;
   ByTe_1 found_sensor, reset_called, more;

  /*************************************************************************/
  /* Check to see if the combination being processed has been processed    */
  /* before.  If not, an error condition - probably didn't call FILE_OPEN. */
  /* Since ir_locate_ex() is called by more than one routine, return an    */
  /* error code that indicates which calling routine resulted in the error.*/
  /* Since a 0 is passed for the last parameter, the only possible error is*/
  /* that the requested combination was not found among processed combos.  */
  /*************************************************************************/

   ExDa = (struct idf_data *) UDF;
   if (ExDa->base_data == NO_MEMORY) {
      rV = ir_check_idf_data_memory (data_key, exten, vnum, UDF);
      if (rV != ALL_OKAY) { return (rV); }
   }

   ExDa->filled_data = 0;
   if (!ginfo.called_locate) {
      rV = ir_locate_ex (data_key, exten, vnum, 0);
      if (rV != ALL_OKAY)
        return (DREC_NOT_FOUND);

      ginfo.called_locate = 1;
      reset_called = 1;
   } else { reset_called = 0; }

  /************************************************************************/
  /* Check file descriptors for the data and header files. If either of   */
  /* the file descriptors is less than 0, the file has not been opened.   */
  /************************************************************************/

   ex  = ginfo.expt;
   ptr = ex->info_ptr;

   if (ex->fdh < 0 || ex->fdd < 0) {
      if (reset_called) { ginfo.called_locate = 0; }
      return (DREC_NO_FILES);
   }

   if (ptr->chg_sen_set) { ExDa->hdr_change = 0; }

  /************************************************************************/
  /*  Set the time values to last time calculated in case EOF's are hit   */
  /*  since fill_data() uses ALL sweeps, even EOF sweeps.                 */
  /************************************************************************/

   ExDa->bmilli = ex->btime_ms % 86400000;
   ExDa->bnano = ex->btime_ns;
   ExDa->emilli = ex->etime_ms % 86400000;
   ExDa->enano = ex->etime_ns;
   ReSetSen = 0;

  /**************************************************************************/
  /* Matrix data can run across data records.  If the data being requested  */
  /* is type matrix and if the column being requesed is greater than the    */
  /* last matrix column in this data record and if this data record does    */
  /* not contain the last column of the matrix then blindly get the next    */
  /* data record.  Note that if the user is jumping around in column in     */
  /* the matrix and triggers an advance then he's out of luck to get data   */
  /* from earlier columns.                                                  */ 
  /**************************************************************************/

   if (ex->smp_id == 3) {
      ExDa->TotCols = *ptr->TOT_COLS;
      if (CoL > *ptr->TOT_COLS) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (DREC_NO_SENSOR);
      }

      RelCol = CoL - *ptr->LEAD_COL; 
      while (RelCol < 0 || RelCol >= *ptr->N_COLS) {
         rV = ir_read_drec_time_advance (UDF, full_swp, 0);
         if (rV == ALL_OKAY) {
            ExDa->TotCols = *ptr->TOT_COLS;
            if (CoL > *ptr->TOT_COLS) {
               if (reset_called) { ginfo.called_locate = 0; }
               return (DREC_NO_SENSOR);
            }
            RelCol = CoL - *ptr->LEAD_COL; 
         } else {
            if (reset_called) { ginfo.called_locate = 0; }
            return (rV);
         }
      }
   } else { ExDa->TotCols = 1; }
     
  /**************************************************************************/
  /* Was there a problem reading from the data file last time?   Send the   */
  /* DREC_EOF flag to ir_new_header_record() to be sure that the same       */
  /* status is used by both routines.  Send no_sensor to new_data_record    */
  /* and new_header_record so return code will be DREC_EOF_NO_SENSOR if     */
  /* data still not there since no data contained in ExDa to be plotted.    */
  /**************************************************************************/

   if (ex->drec_eof != 0) {
      ex->drec_eof = 0;
      if ((rV = ir_new_data_record (0)) != ALL_OKAY) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (rV);
      }
      if ((rV = ir_new_header_record (0, UDF)) != ALL_OKAY) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (rV);
      }
      ReSetSen = 1;
   }

  /************************************************************************/
  /*  Was there a problem reading from the header file last time?         */
  /************************************************************************/

   if (ex->hrec_eof != 0) {
      ex->hrec_eof = 0;
      if ((rV = ir_new_header_record (0, UDF)) != ALL_OKAY) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (rV);
      }
      ReSetSen = 1;
   }

  /************************************************************************/
  /* Find the position of the sensor requested in the sensor index array. */
  /************************************************************************/
 
  found_sensor = 0;
  l1 = &SenIndex;
  s1 = ptr->SENSOR_INDEX;
  s_end = s1 + *ptr->N_SEN;
  if (ex->smp_id == 3)
     RelCol = CoL - *ptr->LEAD_COL; 

  if (SeN < 0)
     SeN = *s1;

  for (*l1 = 0; s1 < s_end; ++(*l1))
    if (*s1++ == SeN)
     {
       found_sensor = 1;
       if (ex->sen_mode < 2 || ex->sen_mode == 4 || ex->sen_mode == 5)
        {
           if (ex->smp_id != 3 && *l1 != ptr->time_col)
              found_sensor = 0;
        }
       break;
     }

  /*************************************************************************/
  /*  Was a new header read on the previous call to this routine?  If so,  */
  /*  certain data elements need to be reset.                              */
  /*************************************************************************/

  if (ptr->reset_hdr)
   {
     /**********************************************************************/
     /*  This check is made in two places to facilitate plotting of data.  */
     /*  (allow the plotting of a partial sweep)                           */
     /**********************************************************************/

     hdr_offset = (*ptr->NSS > 0) ? *(ptr->HDR_OFF + ptr->cur_sen_set) :
                                    *(ptr->HDR_OFF + 0);
     if (hdr_offset == NO_MORE_DATA)
      {
        if (reset_called)
          ginfo.called_locate = 0;
        return (LOS_STATUS);
      }
     else if (hdr_offset == NEXT_FILE)
      {
        if (reset_called)
          ginfo.called_locate = 0;
        return (NEXT_FILE_STATUS);
      }

     if ((rV = ir_reset_header (UDF)) != ALL_OKAY)
      {
        if (reset_called)
          ginfo.called_locate = 0;
        return (rV);
      }
   }

  /**************************************************************************/
  /*  These two fields are updated if a header changes, but if the user is  */
  /*  swapping between various virtuals, or the same virtual, but different */
  /*  time spans, and a header change was not detected for the virtual in   */
  /*  question, user gets wrong YEAR/DAY values being returned.             */
  /**************************************************************************/

  else
   {  
     ExDa->byear = *ptr->YEAR;
     ExDa->bday = *ptr->DAYOFYEAR;
     ExDa->eyear = *ptr->YEAR;
     ExDa->eday = *ptr->DAYOFYEAR;
   }

  /*************************************************************************/
  /* Mode flags and sample sequence data are returned every read.          */
  /* RESET_HEADER() allocates space to hold sample sequence data.          */
  /* Even though changed N_SAMPLE to unsigned short, did not change        */
  /* NUM_SWP_STEPS cause would have forced binning and more importantly,   */
  /* hdr_size to change.  Since hdr_size uses negative numbers, would have */
  /* to change to long, which would cause reprocessing of ALL data sets.   */
  /* Changing N_SAMPLE to unsigned short was needed for a scalar inst.     */
  /*************************************************************************/
 
  ExDa->num_swp_steps = (ex->smp_id == 2) ? 1 : *ptr->N_SAMPLE;

  l1 = ExDa->swp_data;
  s1 = ptr->SAMP_INDEX;
  s_end = s1 + ExDa->num_swp_steps;
  for (; s1 < s_end; )
    *l1++ = *s1++;

  l1 = ExDa->mode;
  uc1 = ptr->MODE_INDEX;
  uc_end = uc1 + *ptr->I_MODE;
  for (; uc1 < uc_end; )
    *l1++ = *uc1++;

  ExDa->mode_len = *(ptr->I_MODE);

  /*************************************************************************/
  /*  Save various data elements associated with this call to the generic  */
  /*  read routine.                                                        */
  /*************************************************************************/
 
  ExDa->data_key = data_key;
  ExDa->version = vnum;
  ExDa->sensor = SeN;
  ExDa->column = CoL;
  ExDa->spin_rate = *(ptr->SPIN);
  ExDa->sun_sen = *(ptr->SUN_SEN);
  strcpy (ExDa->exten, exten);

  /*************************************************************************/
  /*  Set the beginning time and ending time for this time sample.         */
  /*  Adjust the time components for day boundary crossings.  The byear    */
  /*  and eyear elements are the same value BEFORE time adjustment.        */
  /*************************************************************************/
 
  ir_read_drec_times (UDF, full_swp);

  /*************************************************************************/
  /*  If the sensor was found, retrieve the sensor data and any calibration*/
  /*  data for the requested sensor.                                       */
  /*************************************************************************/
 
  if (found_sensor)
   {
     ExDa->d_qual = (ByTe_4) *(ptr->DQUAL + SenIndex);
     rV = ir_read_drec_data_values (UDF, fwd, full_swp, SenIndex);
     if (rV != ALL_OKAY)
      {
        if (reset_called)
          ginfo.called_locate = 0;
        return (rV);
      }
   } else {
      ExDa->d_qual = -1;   /* set to -1 to mean BOGUS value since 0-255 */
   }

  /*************************************************************************/
  /*  If more than 1 IDF file is to be utilized, check the time to see if  */
  /*  the next IDF file should be read.                                    */
  /*************************************************************************/

  if (ex->de_year != -1)
   {
     if ((rV = ir_next_idf_file (UDF)) != ALL_OKAY)
      {
         rV = NewVidf(UDF);
/*
        if (reset_called)
          ginfo.called_locate = 0;
        return (rV);
*/
      }
   }

  /******************************************************************/
  /*  Do we need to advance the pointers to the next time sample?   */
  /******************************************************************/

  if (fwd && (!ReSetSen || !block_adv))
  {
     do 
     {
        rV = ir_read_drec_time_advance (UDF, full_swp, found_sensor);
        if (rV != ALL_OKAY)
         {
           if (reset_called)
             ginfo.called_locate = 0;
           return (rV);
         }
         more = (ex->smp_id != 3 ) ? 0 : (*ptr->LEAD_COL == 0) ? 0 : 1; 
     } while (more);
  }

  /*************************************************************************/
  /*  If the sensor requested was not found, return a different code from  */
  /*  code meaning the sensor was found and data is being returned.        */
  /*************************************************************************/

  if (reset_called)
    ginfo.called_locate = 0;

  /*  Added CAG 12/23/98 - moved mem check into read_drec                 */

   if (ExDa->hdr_change == 1)
   {
      rV = ir_check_idf_data_memory (data_key, exten, vnum, UDF);
      if (rV != ALL_OKAY)
          return (rV);
   }

  if (!found_sensor)
    return (DREC_NO_SENSOR);
  else
    return (ALL_OKAY);
}
