#include "util_str.h"
#include "libbase_udf.h"
#define P_BEFORE_START 1
#define P_EQUAL_START 2
#define P_AFTER_START 3

/*******************************************************************************
 *                                                                             *
 *                         IR_PITCH_START_TIME SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to determine if the current sample starts before, *
 *  at the same time, or after the start time of the delta-t being processed.  *
 *  This information is pertinent to the calculation of the percentage of the  *
 *  data value that is to be included in the current delta-t processing.       *
 *  Unlike the SCF software, do not use the nanoseconds time component; only   *
 *  consider the milliseconds.                                                 *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_times           structure that holds the time period for    *
 *         src_time                the current delta-t being processed         *
 *    short byear                  year value for the start time of the sample *
 *    short bday                   day value for the start time of the sample  *
 *    long bmilli                  time of day value for the start time of the *
 *                                 sample                                      *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_pitch_start_time (src_time, byear, bday, bmilli)                  *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    char use_top                 value returned to calling module            *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_1 ir_pitch_start_time (struct pitch_times src_time, ByTe_2 byear, 
                            ByTe_2 bday, ByTe_4 bmilli)
{
   ByTe_1 use_top;

   /***************************************************************************/
   /*  Check to see if the sample time is prior to the delta-t start time,    */
   /*  equal to the delta-t start time or after the delta-t start time so     */
   /*  the fraction can be calculated with respect to the top edge (start     */
   /*  time) or bottom edge (end time) of the delta-t time period.            */
   /***************************************************************************/

   if (byear > src_time.byear)
     use_top = P_AFTER_START;
   else if (byear < src_time.byear)
     use_top = P_BEFORE_START;
   else
    {
      if (bday > src_time.bday)
        use_top = P_AFTER_START;
      else if (bday < src_time.bday)
        use_top = P_BEFORE_START;
      else
       {
         if (bmilli > src_time.bmilli)
           use_top = P_AFTER_START;
         else if (bmilli == src_time.bmilli)
           use_top = P_EQUAL_START;
         else if (bmilli < src_time.bmilli)
           use_top = P_BEFORE_START;
       }
    }

  return (use_top);
}
