#include "util_str.h"
#include "libbase_udf.h"
#include "user_defs.h"
#define P_BEFORE_START 1
#define P_EQUAL_START 2
#define P_AFTER_START 3

/*******************************************************************************
 *                                                                             *
 *                         IR_PITCH_TIME_FRAC SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to determine what percentage of the data sample   *
 *  should be included for the current delta-t processing.  Based upon where   *
 *  the start time of the sample lies with respect to the start time defined   *
 *  for the delta-t being processed, the fraction is either calculated with    *
 *  respect to the top edge (start time) or bottom edge (end time) of the      *
 *  delta-t time period.  Unlike the SCF software, do not use the nanoseconds  *
 *  time component; only consider the milliseconds.                            *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           *
 *    struct pitch_times           structure that holds the time period for    *
 *         src_time                the current delta-t being processed         *
 *    short which_comp             index specifying which of the 3 components  *
 *                                 is being processed                          *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_pitch_time_frac (&pa_ptr, src_time, which_comp)                   *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_pitch_end_time ()         determines if the current sample terminates *
 *                                 the acquisition for the current component   *
 *    ir_pitch_start_time ()       determines which edge should be used in the *
 *                                 percentage calculation                      *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct idf_data *PA_DATA     structure holding the data for the pitch    *
 *                                 angle component being processed             *
 *    double delta_combo           the delta-t period combined into seconds    *
 *    double diff_combo            the sample duration combined into seconds   *
 *    float frac                   the percentage of the data sample           *
 *    long bmilli                  start time of day for the sample in msec    *
 *    long emilli                  end time of day for the sample in msec      *
 *    long diff_milli              the time of day difference between the      *
 *                                 sample and the edge of the delta-t period   *
 *    long delta_milli             the time of day difference for delta-t      *
 *    short byear                  start year value for the sample             *
 *    short eyear                  end year value for the sample               *
 *    short bday                   start day value for the sample              *
 *    short eday                   end day value for the sample                *
 *    short diff_year              the year difference between the sample and  *
 *                                 the edge of the delta-t period              *
 *    short diff_day               the day difference between the sample and   *
 *                                 the edge of the delta-t period              *
 *    short delta_year             the year difference for the delta-t duration*
 *    short delta_day              the day difference for the delta-t duration *
 *    short days_in_year           the number of day in the year - takes leap  *
 *                                 year into account                           *
 *    char use_top                 indicates if top edge or bottom edge is to  *
 *                                 used in percentage calculation              *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ReaL_4 ir_pitch_time_frac (struct pitch_info *pa_ptr, 
                           struct pitch_times src_time, ByTe_2 which_comp)
{
   struct idf_data *PA_DATA;
   ReaL_8 delta_combo, diff_combo;
   ReaL_4 frac;
   ByTe_4 bmilli, emilli, diff_milli, delta_milli;
   ByTe_2 byear, eyear, bday, eday, diff_year, diff_day, delta_year, delta_day;
   ByTe_2 days_in_yr;
   ByTe_1 use_top;

   /*************************************************************************/
   /*  Get the time of this sample.                                         */
   /*************************************************************************/

   PA_DATA = (struct idf_data *) pa_ptr->idf_data_ptr[which_comp];
   byear = PA_DATA->byear;
   bday = PA_DATA->bday;
   bmilli = PA_DATA->bmilli;

   eyear = PA_DATA->eyear;
   eday = PA_DATA->eday;
   emilli = PA_DATA->emilli;

   ir_pitch_end_time (pa_ptr, src_time, eyear, eday, emilli, which_comp);
   use_top = ir_pitch_start_time (src_time, byear, bday, bmilli);

   /**************************************************************************/
   /*  The following converts the time duration for the delta-t period       */
   /*  being processed into a single value in terms of seconds.              */
   /**************************************************************************/

   if ((src_time.byear % 4 == 0 && src_time.byear % 100 != 0) || 
        src_time.byear % 400 == 0)
     days_in_yr = 366;
   else
     days_in_yr = 365;
   delta_year = src_time.eyear - src_time.byear;
   delta_day = src_time.eday - src_time.bday;
   delta_milli = src_time.emilli - src_time.bmilli;
   delta_combo = (delta_year * days_in_yr * 86400) + delta_day * 86400 +
                 (delta_milli / 1000.0);

   /****************************************************************************/
   /*  The start time of this sample is before or equal to the start time of   */
   /*  the delta-t being processed.                                            */
   /****************************************************************************/

   if (use_top == P_BEFORE_START || use_top == P_EQUAL_START)
    {
       /***********************************************************************/
       /*  The all_done element is set if the end time of the sample is equal */
       /*  to or beyond the end time of the delta-t being processed.          */
       /***********************************************************************/

       if (pa_ptr->all_done[which_comp])
        {
          frac = 1.0;
          return (frac);
        }
       else
        {
          /********************************************************************/
          /*  The time duration of the sample is computed by subtracting the  */
          /*  top edge of the delta-t time period from the end time of the    */
          /*  sample.  The time duration for the sample is converted into a   */
          /*  single value in terms of seconds.                               */
          /********************************************************************/

          diff_year = eyear - src_time.byear;
          diff_day = eday - src_time.bday;
          diff_milli = emilli - src_time.bmilli;
          diff_combo = (diff_year * days_in_yr * 86400) + diff_day * 86400 +
                       (diff_milli / 1000.0);
        }
    }

   /****************************************************************************/
   /*  The start time of this sample is after the start time of the delta-t    */
   /*  being processed.                                                        */
   /****************************************************************************/

   else
    {
      /***********************************************************************/
      /*  If this is true, the time period is WITHIN the delta-t time period.*/
      /***********************************************************************/

      if (pa_ptr->all_done[which_comp])
       {
         frac = 1.0;
         return (frac);
       }

      /*********************************************************************/
      /*  The time duration of the sample is computed by subtracting the   */
      /*  start time of the sample from the bottom edge of the delta-t     */
      /*  time period.  The time duration for the sample is converted into */
      /*  a single value in terms of seconds.                              */
      /*********************************************************************/

      if ((byear % 4 == 0 && byear % 100 != 0) || byear % 400 == 0)
        days_in_yr = 366;
      else
        days_in_yr = 365;

      diff_year = src_time.eyear - byear;
      diff_day = src_time.eday - bday;
      diff_milli = src_time.emilli - bmilli;
      diff_combo = (diff_year * days_in_yr * 86400) + diff_day * 86400 +
                   (diff_milli / 1000.0);
    }

   /*********************************************************************/
   /*  Compute the percentage of the sample that should be included for */
   /*  the current delta-t processing.                                  */
   /*********************************************************************/

   frac = diff_combo / delta_combo;
   return (frac);
}
