#include <stdio.h>
#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                      IR_NUMBER_OF_CAL_ELEMENTS SUBROUTINE                  *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to determine the total number of calibration     *
 *  elements associated with the sample sequence values and/or sensor data.   *
 *  If the total number of calibration values exceeds the size of the memory  *
 *  allocated to return this information to the user, array is resized.       *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *UDF                    memory location ptr for the structure that *
 *                                 holds returned data values (read_drec)     *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_number_of_cal_elements (UDF)                                     *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_check_cal_size ()         makes sure enough space is allocated in    *
 *                                 idf_data structure for cal. data           *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    struct ptr_rec *ptr          a pointer to the structure which holds     *
 *                                 all pointers to the header and data for    *
 *    register short i             looping variable                           *
 *    long mod_val                 a temporary variable used to determine the *
 *                                 number of calibration elements             *
 *    short ret_val                holds value returned by calling module     *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

ByTe_2 ir_number_of_cal_elements (void *UDF)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr; 
  register u_ByTe_2 *uS, *EnD;
  register ByTe_2 *S;
  register ByTe_1 *C;
  ByTe_4 mod_val;
  ByTe_4 TotCol;
  ByTe_2 ret_val;

  /***************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and */
  /*  data information for the experiment currently being processed.         */
  /***************************************************************************/

  ex = ginfo.expt;
  ptr = ex->info_ptr;

  /***************************************************************************/
  /*  If no cal sets then no need to hang around                             */
  /***************************************************************************/

  ptr->NumSenAnc = 0; 
  ptr->NumGenAnc = 0;
  if ( ex->cal_sets == 0 ) { return (ALL_OKAY); }

  EnD = ex->CSET_NUM + ex->cal_sets;
  for (uS = ex->CSET_NUM; uS < EnD; )
    *uS++ = 0;

  /***************************************************************************/
  /*  Determine the number of calibration values n each calibration set.     */
  /***************************************************************************/

  for (uS = ex->CSET_NUM, S = ex->CAL_USE; uS < EnD; ++uS, ++S)
   {
     /**********************************************************************/
     /*  There is one calibration value in this calibration set.           */
     /**********************************************************************/

     if (*S == 0)
       *uS = 1;
     else 
      {
        mod_val = (ByTe_4) *ptr->N_SAMPLE % *S;
        if (mod_val == 0)
          *uS = (ByTe_4) *ptr->N_SAMPLE / *S;
        else
          *uS = ((ByTe_4) *ptr->N_SAMPLE / *S) + 1;
      }
   }

  /***************************************************************************/
  /*  Determine the total number of sensor data anc and sample sequence anc. */
  /*  When data is laid down, there is a calibration value for each sensor.  */
  /***************************************************************************/

  TotCol = (ex->smp_id != 3) ? *ptr->N_SEN : *ptr->N_SEN * *ptr->N_COLS;
  for (uS = ex->CSET_NUM, C = ex->cal_target; uS < EnD; ++uS, ++C)
  {
     /******************************************************************/
     /*  Pick up sweep calibration data for this cal. set.             */
     /******************************************************************/
     
      switch (*C)
      {
          case 0:
             ptr->NumSenAnc += *uS * TotCol;
          break;
          case 2:
             ptr->NumGenAnc += *uS;
          break;
      }
   }

   ret_val = ir_check_cal_size (UDF);
   if (ret_val != ALL_OKAY) { return (ret_val); }

   return (ALL_OKAY);
}
