#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                       NEXT_FILE_START_TIME SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to return the time to be used to retrieve the     *
 *  next data file once the LOS_STATUS or NEXT_FILE_STATUS has been returned   *
 *  by the data acquisition routines for the current data file.                *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long data_key     key which uniquely identifies the data set    *
 *                               being processed                               *
 *    char *exten                filename extension for the data to be used    *
 *    unsigned short vnum        version number to be associated with this     *
 *                               combination (allows for multiple opens)       *
 *    char mode_data             flag indicating if mode times being requested *
 *    short *start_yr            the year for next data file retrieval         *
 *    short *start_day           the day of year for next data file retrieval  *
 *    long *start_sec            the time of day in seconds for next data file *
 *                               retrieval                                     *
 *    long *start_nsec           the time of day residual for next data file   *
 *                               retrieval                                     *
 *  USAGE                                                                      *
 *    x = next_file_start_time (data_key, exten, vnum, mode_data, &start_yr,   *
 *                              &start_day, &start_sec, &start_nsec)           *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_locate_ex()             determines if requested combination has       *
 *                               already been pand points to the               *
 *    adjust_time ()             corrects time elements if too many seconds    *
 *                               (day crossing) or too many days(year crossing)*
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info        structure that holds information concerning   *
 *        ginfo                  the experiment that is being processed        *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info     a pointer to the structure that holds         *
 *          *ex                  specific experiment information               *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all    *
 *                               pointers to the header and data for the       *
 *                               experiment of interest                        *
 *    long bmsec, bnsec          start time of last sample processed           *
 *    long emsec, ensec          stop time of last sample processed            *
 *    short rval                 holds called routine status flags             *
 *    char reset_called          flag indicating if LOCATE_EX was called       *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 next_file_start_time (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                             ByTe_1 mode_data, ByTe_2 *start_yr, 
                             ByTe_2 *start_day, ByTe_4 *start_sec, 
                             ByTe_4 *start_nsec)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_2 rval;
  ByTe_1 reset_called;

  /*************************************************************************/
  /* Check to see if the combination being processed has been processed    */
  /* before.  If not, an error condition - probably didn't call FILE_OPEN. */
  /* Since ir_locate_ex() is called by more than one routine, return an    */
  /* error code that indicates which calling routine resulted in the error.*/
  /* Since a 0 is passed for the last parameter, the only possible error is*/
  /* that the requested combination was not found among processed combos.  */
  /*************************************************************************/
  
  if (!ginfo.called_locate)
   {
     rval = ir_locate_ex (data_key, exten, vnum, 0);
     if (rval != ALL_OKAY)
       return (NEXT_FILE_TIME_NOT_FOUND);

     ginfo.called_locate = 1;
     reset_called = 1;
   }
  else
    reset_called = 0;

  /*************************************************************************/
  /*  Get the beginning time and ending time for the last sample.          */
  /*************************************************************************/
 
  ex = ginfo.expt;
  ptr = ex->info_ptr;
  *start_yr = ex->btime_year;
  *start_day = ex->btime_day;
  *start_sec = ex->btime_sec;
  *start_nsec = ex->btime_nsec;

  adjust_time (start_yr, start_day, start_sec, 1);

  if (reset_called)
    ginfo.called_locate = 0;

  return (ALL_OKAY);
}
