#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                         IR_GET_SENSOR_TABLES SUBROUTINE                    *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to read table information which is utilized by   *
 *  the various sensors for the virtual instrument being processed.  The      *
 *  offset values for the defined tables are needed by the calling routine    *
 *  to determine the number of unique table combinations used by the sensors  *
 *  so that space can be allocated to hold the information.                   *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    char *oInfo              an array of 1 flag per table which indicates   *
 *                             if the table has different offset values       *
 *    char *cInfo              an array of 1 flag per table which indicates   *
 *                             if there are critical action values present    *
 *    long *tOffs              an array holding the table offsets for each    *
 *                             table                                          *
 *    long *crB                an array holding the critical action bytes     *
 *                             for table                                      *
 *  USAGE                                                                     *
 *    x = ir_get_sensor_tables (&iInfo, &cInfo, &tOffs, &crB)                 *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    read_idf()               reads information from the IDF file            *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    long rV                  holds the value returned by the called routine *
 *    long *cur_tbl            pointer to the current sensor table being      *
 *                             processed                                      *
 *    long BaseTO              temporary variable for comparisons purposes    *
 *    long BaseSB              temporary variable for comparisons purposes    *
 *    short i, k               looping variables                              *
 *    short first_sen          the 1st sensor being utilized (not all sensors *
 *                             for a given virtual inst. may be needed)       *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_get_sensor_tables (ByTe_1 *oInfo, ByTe_1 *cInfo, ByTe_4 *tOffs, 
                             ByTe_1 *crB)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register ByTe_4 *l1;
   register ByTe_2 *s1;
   register ByTe_1 *c1, *c2, *c3;

   ByTe_4 rV, BaseTO, BaseSB, nTbl, nSen;
   ByTe_2 i, k, first_sen;

   ex = ginfo.expt;
   nTbl = ex->num_tbls;
   nSen = ex->num_sensor;

  /**************************************************************************/
  /* Loop over tables picking up the needed information                     */
  /**************************************************************************/

   l1 = tOffs;
   c1 = ex->tbl_var;
   c2 = crB;
   c3 = cInfo;
   for (i = 0; i < nTbl; ++i, ++c1, ++c3, c2 += nSen, l1 += nSen) {

  /************************************************************************/
  /*  Do not pick up the offsets for Mode tables since they will not be   */
  /*  considered in the combination determination AND since there may be  */
  /*  more modes than sensors and space was allocated per sensor.         */
  /************************************************************************/
   
      if (*c1 == 4 || *c1 == 5) { continue; }

  /**************************************************************************/
  /*  Retrieve offset values for each table from the VIDF file. tbl_ptrs    */
  /*  holds the address at which to store each table offset values.         */
  /**************************************************************************/

      rV = read_idf (ex->data_key, ex->exten, ex->version, 
                             (ByTe_1 *)l1, _TblOfF, i, 0L, -1);
      if (rV < 0) {
         switch (rV) {
            case IDF_NOT_FOUND:  return (SEN_IDF_NOT_FOUND);
            case IDF_MANY_BYTES: return (SEN_IDF_MANY_BYTES);
            case IDF_TBL_NUM:    return (SEN_IDF_TBL_NUM);
            case IDF_CON_NUM:    return (SEN_IDF_CON_NUM);
            case IDF_NO_ENTRY:   return (SEN_IDF_NO_ENTRY);
            default:             return ((ByTe_2) rV);
         }
      }

  /**************************************************************************/
  /* Get the critical status byte associated with each sensor (provided of  */
  /* course that these are defined).                                        */  
  /**************************************************************************/

      if (*c3 != 0) { 
         rV = read_idf (ex->data_key, ex->exten, ex->version, 
                          (ByTe_1 *)c2, _CritStatuS, i, 0L, -1);
         if (rV < 0) {
            switch (rV) {
               case IDF_NOT_FOUND:  return (SEN_IDF_NOT_FOUND);
               case IDF_MANY_BYTES: return (SEN_IDF_MANY_BYTES);
               case IDF_TBL_NUM:    return (SEN_IDF_TBL_NUM);
               case IDF_CON_NUM:    return (SEN_IDF_CON_NUM);
               case IDF_NO_ENTRY:   return (SEN_IDF_NO_ENTRY);
               default:             return ((ByTe_2) rV);
            }
         }
      }
   }

  /************************************************************************/
  /*  Find the first sensor being utilized.                               */
  /************************************************************************/

   s1 = ex->index_sen_tbl;
   for (i = 0; i < nSen; ++i, ++s1) {
      if (*s1 >= 0) {
         first_sen = i;
         break;
      }
   }

  /**************************************************************************/
  /*  Check the table offsets and if they exist the critical status bytes   */
  /*  for those sensors being utilized in each table. If the values are all */
  /*  the same for a particular table, this table can be eliminated as      */
  /*  being different for different sensors                                 */ 
  /**************************************************************************/

   l1 = tOffs;
   c1 = ex->tbl_var;
   c2 = crB;
   c3 = cInfo;
   for (i = 0; i < nTbl; ++i, ++c1, ++c3, c2 += nSen, l1 += nSen) {

      *(oInfo + i) = 0;

  /************************************************************************/
  /*  Mode tables are not considered in the combination determination.    */
  /************************************************************************/

      if (*c1 == 4 || *c1 == 5) { continue; }
   
      BaseTO = *(l1 + first_sen);

  /************************************************************************/
  /*  If any of the offset values for the sensors being utilized are      */
  /*  different, this table must be included in the comparison of         */
  /*  table offset values.                                                */
  /************************************************************************/

      s1 = ex->index_sen_tbl + first_sen + 1;
      for (k = first_sen + 1; k < nSen; ++k, ++s1 ) {
         if ( (*s1 >= 0) && *(l1 + k) != BaseTO) { *(oInfo + i) = 1; }
      } 

  /**************************************************************************/
  /*  If cr_status values exist, check all values.  If the values are all   */
  /*  the same, this variable can be eliminated from the comparison checks. */
  /**************************************************************************/

      if (*c3 != 0) { 
         *c3 = 0;
         BaseSB = *(c2 + first_sen);

  /************************************************************************/
  /*  If any of the crit_status values for the sensors being utilized are */
  /*  different, this variable must be included in the comparison of      */
  /*  table offset values.                                                */
  /************************************************************************/

         s1 = ex->index_sen_tbl + first_sen + 1;
         for (k = first_sen + 1; k < nSen; ++k, ++s1 ) {
            if ( (*s1 >= 0)  && *(c2 + k) != BaseSB) { *c3 = 1; }
         } 
      }
   }

  return (ALL_OKAY);
}
