#include "util_str.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                          IR_FILL_DATA_ARRAY SUBROUTINE                   *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine fills in the data structure with the data for requested  *
 *  sensor/time period combination.  The algorithms for extracting the data *
 *  are dependent upon the size of the data elements (in bits).  As the     *
 *  program stands, it will handle data up to 32 bits in length.  For       *
 *  larger sizes, code must be added.                                       *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    long SenIndex              gives position of sensors in sensor set    *
 *    char full_swp              flag indicating if 1 or all of the values  *
 *                               in the sensor set are to be retreived (for *
 *                               scalar sensors only)                       *
 *    void *UDF                  memory location ptr for the structure that *
 *                               holds returned data values (read_drec)     *
 *                                                                          *
 *  USAGE                                                                   *
 *    ir_fill_data_array (SenIndex, full_swp, UDF)                          *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    None                                                                  *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure holding information concerning   *
 *        ginfo                  the experiment that is being processed     *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa      structure that holds all of the currently  *
 *                               returned data values to be processed       *
 *    struct experiment_info     a pointer to the structure that holds      *
 *          *ex                  specific experiment information            *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all *
 *                               pointers to the header and data for the    *
 *                               experiment of interest                     *
 *    register long *ret_ptr     ptr to the array where data is returned    *
 *    register uns long *lpt     data is interpreted as an array of longs   *
 *    register uns short *s1     data is interpreted as an array of shorts  *
 *    register uns char *ucpt    data is interpreted as an array of chars   *
 *    register uns char *uc1     pointer to the mask/shift array            *
 *    long num_bytes             number of bytes to bypass in order to get  *
 *                               to the current sensor set                  *
 *    long offset                current bit pointed to within the sensor   *
 *                               set for bit data                           *
 *    long *stop_loop            loop termination variable                  *
 *    unsigned short max_elements the number of data values to be processed *
 *    short result               temporary value                            *
 *    stat short one_bit_mask[]  mask giving the bit of interest            *
 *    stat char two_bit_shift[]  no. of bits to shift to get the two bits   *
 *                               of interest to the last 2 bits in the byte *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

void ir_fill_data_array (ByTe_4 SenIndex, ByTe_1 full_swp, void *UDF)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  register ByTe_4 *ret_ptr;
  register u_ByTe_4 *lpt;
  register u_ByTe_2 *s1;
  register u_ByTe_1 *ucpt;
  register u_ByTe_1 *uc1;
  ByTe_4 num_bytes, offset, *stop_loop, CurCol;
  u_ByTe_2 max_elements;
  ByTe_2 result;
  static u_ByTe_1 one_bit_mask[8] = {128, 64, 32, 16, 8, 4, 2, 1};
  static u_ByTe_1 two_bit_shift[4] = {6, 4, 2, 0};

  /***********************************************************************/
  /* Set a pointer to structure which holds all pointers for header and  */
  /* data information for the experiment currently being processed.      */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  ExDa = (struct idf_data *) UDF;

  /********************************************************************/
  /* For a full swp, max_elements is not just set to N_SAMPLE because */
  /* user could have asked for 1 at a time and changed to a full swp  */
  /* to get remainder of elements. We use time_row since we need to   */
  /* pick up rest of the elements in that column being processed (for */
  /* both SEN_MODE = 0 & 4) and time_row tells how many have already  */
  /* been processed for that sensor.                                  */
  /********************************************************************/

  max_elements = (ex->smp_id == 2 && !full_swp) ? 1 : 
                 *ptr->N_SAMPLE - ptr->time_row;

  /**********************************************************************/
  /*  Determine the number of bits needed to be bypassed to get to the  */
  /*  current sensor set.  Convert that number to bytes (partial bytes  */
  /*  included).                                                        */
  /**********************************************************************/

  num_bytes = ex->accum_ss_sz / 8;
  if (ex->accum_ss_sz % 8 != 0) 
     ++num_bytes;

  /********************************************************************/
  /*  Set pointers to the returned data array and to the position of  */
  /*  the last data value to be returned                              */
  /********************************************************************/ 

   ret_ptr = ExDa->sen_data;
   stop_loop = ret_ptr + max_elements;
 
   CurCol = (ex->smp_id != 3) ? SenIndex : SenIndex * *ptr->N_COLS + 
                                            (ExDa->column - *ptr->LEAD_COL);

  /**********************************************************************/
  /*  The data elements are packed into a byte. (Bit information.       */
  /**********************************************************************/

  if (ex->tdw_len < 8)
   {
     /******************************************************************/ 
     /*  Following switch pulls out data for different bit lengths.    */
     /*  Each case statement does the same                             */
     /*  1) Set offset to data, this includes offsets into the current */
     /*     sensor set, the sensor position, and sensor sample         */
     /*  2) Set base pointer to data                                   */
     /*  3) Loop over all of the elements                              */
     /*     A) Find byte which has sensor data and set pointer to it   */
     /*     B) Mask off data                                           */
     /*     C) If necessary shift data into position                   */
     /*                                                                */
     /*  Note that this data acquisition algorithm assumes that a each */
     /*  sensor set begins as a NEW byte.  That is one byte cannot     */
     /*  contain data from two sensor sets                             */
     /******************************************************************/ 

     switch (ex->tdw_len)
     {
         case 1:
            offset = *ptr->N_SAMPLE * CurCol + ptr->time_row;
            ucpt = (u_ByTe_1 *) (ptr->DATA + num_bytes);
            uc1 = one_bit_mask;
            for (; ret_ptr < stop_loop; ++offset)
            {
               result = *(ucpt + offset/8) & *(uc1 + offset % 8);
               *ret_ptr++ = (result != 0) ? 1 : result;
            }
            break;
         case 2:
            offset = (*ptr->N_SAMPLE * CurCol + ptr->time_row) << 1;
            ucpt = (u_ByTe_1 *) (ptr->DATA + num_bytes);
            uc1 = two_bit_shift;
            for (; ret_ptr < stop_loop; offset += 2)
            {
               result = *(ucpt + offset/8) >> *(uc1 + (offset % 8)/2);
               *ret_ptr++ = result & 3;
            }
            break;
         case 3:
            offset = (*ptr->N_SAMPLE * CurCol + ptr->time_row) << 2;
            ucpt = (u_ByTe_1 *) (ptr->DATA + num_bytes);
            for (; ret_ptr < stop_loop; offset += 4)
            {
               result = (offset % 8 == 0) ? *(ucpt + offset / 8) >> 4 
                                          : *(ucpt + offset / 8);
               *ret_ptr++ = result & 7;
            }
            break;
         case 4:
            offset = (*ptr->N_SAMPLE * CurCol + ptr->time_row) << 2;
            ucpt = (u_ByTe_1 *) (ptr->DATA + num_bytes);
            for (; ret_ptr < stop_loop; offset += 4)
            {
               result = (offset % 8 == 0) ? *(ucpt + offset / 8) >> 4 
                                          : *(ucpt + offset / 8);
               *ret_ptr++ = result & 15;
            }
            break;
         default:
            ucpt = (u_ByTe_1 *) (ptr->DATA + num_bytes + 
                    *ptr->N_SAMPLE * CurCol + ptr->time_row);
            for (; ret_ptr < stop_loop; )
                *ret_ptr++ = *ucpt++;
            break;
      }
   }

  /******************************************************************/
  /*  The data requires one or more bytes per element.              */
  /******************************************************************/
 
  else 
   {
     /*******************************************************************/
     /* Determine the number of bytes per data element.  Data and cal.  */
     /* data are the same size (bytes).  The premise at this time is    */
     /* that all data is <= 32 bits in length.  Data is laid down in 1, */
     /* 2, or 4 bytes.                                                  */
     /*******************************************************************/
 
     if (ex->tdw_len == 8)
     {
        ucpt = ptr->DATA + num_bytes + *ptr->N_SAMPLE * CurCol
                   + ptr->time_row; 
        for (; ret_ptr < stop_loop; )
            *ret_ptr++ = *ucpt++;
     }
     else if (ex->tdw_len <= 16)  /* interpret as shorts */
      {
        s1 = (u_ByTe_2 *) (ptr->DATA + num_bytes + 
              *ptr->N_SAMPLE * 2 * CurCol + 2 * ptr->time_row); 
        for (; ret_ptr < stop_loop; )
           *ret_ptr++ = *s1++;
      }
     else if (ex->tdw_len <= 32)  /* interpret as longs */
      {
        lpt = (u_ByTe_4 *) (ptr->DATA + num_bytes + 
              *ptr->N_SAMPLE * 4 * CurCol + 4 * ptr->time_row); 
        for (; ret_ptr < stop_loop;)
           *ret_ptr++ = *lpt++;
      }
   }

   if (ex->d_type[SenIndex] == 1)
      ir_to_sign (ExDa->sen_data, max_elements, ex->tdw_len);

   ExDa->num_sample = max_elements;
}
