#include "OpSySInD.h"
#include <stdio.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                            FILE_POS SUBROUTINE                             *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This module walks thru the data file to the correct starting position   *
 *  corresponding to the start time selected by the user.  If the user        *
 *  entered a -1 for the start time, the file descriptor for the data file is *
 *  positioned at the beginning of the data file.  If the user entered a -2   *
 *  for the start time, the file descriptor for the data file is positioned   *
 *  at the end of the data file.  In either case, one data record and the     *
 *  corresponding header record is read so that information is ready to be    *
 *  processed when the user calls READ_DREC().  If the user requested a play- *
 *  back scenario, a binary search is performed as a coarse search for the    *
 *  requested start time.  A fine search is then performed in order to get as *
 *  close to the requested start time as possible.                            *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long data_key   key which uniquely identifies the data set     *
 *                             being processed                                *
 *    char *exten              the filename extension for the data to be used *
 *    unsigned short vnum      version number to be associated with this      *
 *                             combination (allows for multiple opens)        *
 *    void *idf_data_ptr       ptr to the memory location for the structure   *
 *                             that holds returned data values (read_drec)    *
 *    short bY                 the start time requested (year component)      *
 *    short bD                 the start time requested (day component)       *
 *    long bS                  the start time requested (seconds component)   *
 *    long bNs                 the start time requested (nanoseconds)         *
 *    short eY                 the stop time requested (year component)       *
 *    short eD                 the stop time requested (day component)        *
 *    long eS                  the stop time requested (seconds component)    *
 *    long eNs                 the stop time requested (nanoseconds)          *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = file_pos (data_key, exten, vnum, idf_data_ptr, bY, bD, bS, bNs,     *
 *                                                       eY, eD, eS, eNs)     *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                the size of the specified object in bytes      *
 *    read()                   reads N bytes from the file associated with    *
 *                             the file descriptor given                      *
 *    lseek()                  moves the file pointer to a location within    *
 *                             the file                                       *
 *    ir_alloc_exp_once()      allocates memory that is needed in order to    *
 *                             utilize the generic routines provided for      *
 *                             retrieving experiment data                     *
 *    ir_locate_ex()           determines if the requested combination has    *
 *                             already been processed and points to the       *
 *                             correct structure allocated for the combo      *
 *    ir_read_header()         reads data from the header file                *
 *    ir_position_real_time()  positions the file pointer for a real-time     *
 *                             scenario                                       *
 *    ir_fine_search ()        positions the file pointer as close to the     *
 *                             requested time as possible (within a record)   *
 *    select_sensor()          specifies which sensors are to be processed    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    struct ptr_rec *ptr      a pointer to the structure which holds all     *
 *                             pointers to the header and data for the        *
 *                             experiment of interest                         *
 *    struct pitch_info        a pointer to the structure that holds pitch    *
 *        *pa_ptr              angle information                              *
 *    int num_bytes            the number of bytes requested from the file    *
 *    int ret_bytes            the number of bytes actually read by READ()    *
 *    int low, high, mid       variables used for the binary search on the    *
 *                             data file to find the requested start time     *
 *    long num_records         the number of data records in the data file    *
 *    long bytes               no. of bytes to seek by for the LSEEK routine  *
 *    long ret_sec             the composite time in seconds                  *
 *    long ret_nsec            nanoseconds component of the composite time    *
 *    long rV                  holds the value returned by the called routine *
 *    long *hdr_offset         header offset value                            *
 *    short R                  holds the value returned by the called routine *
 *    char first_time          flag indicating the first time through a loop  *
 *    char reset_called        flag indicating if LOCATE_EX was called        *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

ByTe_2 file_pos (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                 void *idf_data_ptr, ByTe_2 bY, ByTe_2 bD, ByTe_4 bS, 
		 ByTe_4 bNs, ByTe_2 eY, ByTe_2 eD, ByTe_4 eS, ByTe_4 eNs)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct ptr_rec *ptr;
   struct pitch_info *pa_ptr;
   int num_bytes, ret_bytes, low, high, mid;
   ByTe_4 num_records, bytes, ret_sec, ret_nsec, rV, *hdr_off, FLen;
   ByTe_2 R; 
   ByTe_1 first_time, reset_called;

  /**************************************************************************/
  /*  Check to see if the combination being processed has been processed    */
  /*  before.  If not, an error condition - probably didn't call FILE_OPEN. */
  /*  Since ir_locate_ex() is called by more than one routine, return an    */
  /*  error code that indicates which calling routine resulted in the error.*/
  /*  Since a 0 is passed for the last parameter, the only possible error is*/
  /*  that the requested combination was not found among processed combos.  */
  /**************************************************************************/
  
   if (!ginfo.called_locate) {
      R = ir_locate_ex (data_key, exten, vnum, 0);
      if (R != ALL_OKAY)
         return (POS_NOT_FOUND);
 
      ginfo.called_locate = 1;
      reset_called = 1;
    } else { reset_called = 0; }

  /**************************************************************************/
  /* Set a pointer to the structure which holds all pointers for header and */
  /* data information for the experiment currently being processed.         */
  /**************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;

  /**************************************************************************/
  /* If the file pointers have already been set, just return to the calling */
  /* routine.                                                               */
  /**************************************************************************/

   if (ex->fnext) {
      if (reset_called)
        ginfo.called_locate = 0;
      return (ALL_OKAY);
    }

  /**************************************************************************/
  /*  Set the pointer at the correct location in the data and header file   */
  /*  for the real-time scenarion.                                          */
  /**************************************************************************/

   ex->hrec_eof = 0;
   ex->drec_eof = 0;
   if (bS == -1 || bS == -2) {
      R = ir_position_real_time (bS, idf_data_ptr);
      if (R != ALL_OKAY) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (R);
      }
   } else {

  /***********************************************************************/
  /*  Determine the number of records in the data file.  The last data   */
  /*  record may be a partial record written by the listener, but ignore */
  /*  this record.  Use -2 instead of -1 to exclude trailer record       */
  /*  written by IDFS builder program.                                   */
  /***********************************************************************/
   
      if ((rV = FileLength (ex->data_name, &FLen)) < 0) {
         if (reset_called) { ginfo.called_locate = 0; }
         return (rV - 330); 
      }

      num_records = FLen / ptr->d_size;
      low = 0;
      high = num_records - 2;
      first_time = 1;

  /***********************************************************************/
  /*  Find the record closest to the requested start time using a binary */
  /*  search algorithm.                                                  */
  /***********************************************************************/
   
      while (low <= high) {
         mid = (low + high) / 2;
         bytes = ptr->d_size * mid;
         rV = lseek (ex->fdd, bytes, SEEK_SET);
   
  /*******************************************************************/
  /*  Read the data record to be tested.                             */
  /*******************************************************************/
   
         num_bytes = ptr->d_size;
         ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
   
  /********************************************************************/
  /* Since the data and header files are copied and catalogued as is, */
  /* a check still needs to be made for a short data record (written  */
  /* by listener).                                                    */
  /********************************************************************/

         if (ret_bytes < 0) {
            if (reset_called) { ginfo.called_locate = 0; }
            return (POS_DATA_READ_ERROR);
         }

  /********************************************************************/
  /*  Do any byte swapping should that be required.                   */
  /********************************************************************/

         if (ex->BswaP) { ReOrderUdf (1); }

  /********************************************************************/
  /*  The file closure flags are always in the first hdr_offset field.*/
  /********************************************************************/
   
         if (ret_bytes != num_bytes) {
            if (reset_called) { ginfo.called_locate = 0; }
            hdr_off = (ByTe_4 *) (ex->DATA_MEM + 3 * sizeof (ByTe_4));
            if (*hdr_off == NO_MORE_DATA)
              return (PBACK_LOS);
            else if (*hdr_off == NEXT_FILE)
              return (PBACK_NEXT_FILE);
            else { return (POS_DATA_READ_ERROR); }
         }

  /********************************************************************/
  /*  Get to the start of the header record for this data record.     */
  /********************************************************************/

         rV = lseek (ex->fdh, (ByTe_4)*(ptr->HDR_OFF), SEEK_SET);

         if (first_time) {

   /********************************************************************/
   /*  Allocate memory needed in order to utilize the generic routines */
   /*  provided for retrieving experiment data.  This call will read   */
   /*  the header record that is associated with the data record.      */
   /********************************************************************/
           
            R = ir_alloc_exp_once (idf_data_ptr);
            if (R != ALL_OKAY) {
               if (reset_called) { ginfo.called_locate = 0; }
               if (R == LOS_STATUS)
                 return (PBACK_LOS);
               else if (R == NEXT_FILE_STATUS)
                 return (PBACK_NEXT_FILE);
               else { return (R); }
            }

            first_time = 0;
         } else {

   /*****************************************************************/
   /*  Read the header associated with the data record.             */
   /*****************************************************************/

            R = ir_read_header (idf_data_ptr);
            if (R != ALL_OKAY) {
               if (reset_called) { ginfo.called_locate = 0; }
               if (R == RHDR_READ_ERROR)
                 return (POS_HDR_READ_ERROR);
               else if (R == RHDR_HDR_MALLOC)
                 return (POS_HDR_MALLOC);
               else if (R == RHDR_HDR_REALLOC)
                 return (POS_HDR_REALLOC);
               else if (R == LOS_STATUS)
                 return (PBACK_LOS);
               else if (R == NEXT_FILE_STATUS)
                 return (PBACK_NEXT_FILE);
               else { return (R); }
            }
         }

   /*********************************************************************/
   /*  Dr_time in the data record is in milliseconds.                   */
   /*********************************************************************/
   
         ret_sec = *(ptr->TIME) / 1000;
         ret_nsec = (*(ptr->TIME) % 1000) * 1000000;

   /********************************************************************/
   /*  The requested start time is less than or equal to the time of   */
   /*  the record.                                                     */
   /********************************************************************/
   
         if (bY < *ptr->YEAR || 
             (bY == *ptr->YEAR && bD < *ptr->DAYOFYEAR))
           high = mid - 1;
         else if (bY == *ptr->YEAR && bD == *ptr->DAYOFYEAR && bS < ret_sec)
           high = mid - 1;
         else if (bY == *ptr->YEAR && bD == *ptr->DAYOFYEAR &&
                  bS == ret_sec && bNs <= ret_nsec)
           high = mid - 1;
         else if (bY == *ptr->YEAR && bD == *ptr->DAYOFYEAR &&
                  bS == ret_sec && bNs == ret_nsec)
           high = mid - 1;
   
   /********************************************************************/
   /* The requested start time is greater than the time of the record. */
   /********************************************************************/
   
         else { low = mid + 1; }
      }

    /***********************************************************************/
    /* Always back up one record if possible.  Fine Search will make up    */
    /* for it.  This takes care of the case where the time is between data */
    /* data records and also the case were the sensor set at the end of    */
    /* of one data record has the same time as the beginning time of the   */
    /* next data record.  Mod by Gurgiolo/1998                             */
    /***********************************************************************/
   
      bytes = ptr->d_size * -2;
      rV = lseek (ex->fdd, bytes, SEEK_CUR);
      if ( rV >= 0 ) {
         num_bytes = ptr->d_size;
         ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
   
    /********************************************************************/
    /* Since the data and header files are copied and catalogued as is, */
    /* a check still needs to be made for a short data record (written  */
    /* by listener).                                                    */
    /********************************************************************/

         if (ret_bytes < 0) {
            if (reset_called) { ginfo.called_locate = 0; }
            return (POS_DATA_READ_ERROR);
         }

    /********************************************************************/
    /*  The file closure flags are always in the first hdr_offset field.*/
    /********************************************************************/
   
         if (ex->BswaP) { ReOrderUdf (1); }

         if (ret_bytes != num_bytes) { 
            if (reset_called) { ginfo.called_locate = 0; }
            hdr_off = (ByTe_4 *) (ex->DATA_MEM + 3 * sizeof (ByTe_4));
            if (*hdr_off == NO_MORE_DATA)
              return (PBACK_LOS);
            else if (*hdr_off == NEXT_FILE)
              return (PBACK_NEXT_FILE);
            else { return (POS_DATA_READ_ERROR); }
         }
         rV = lseek (ex->fdh, (ByTe_4)*(ptr->HDR_OFF), SEEK_SET);

    /********************************************************************/
    /* Since ir_read_header() is called by more than 1 routine, return  */
    /* an error code that indicates which calling routine resulted in   */
    /* the error and what the error was.                                */
    /********************************************************************/

         R = ir_read_header (idf_data_ptr);
         if (R != ALL_OKAY) {
            if (reset_called) { ginfo.called_locate = 0; }
            if (R == RHDR_READ_ERROR)
              return (POS_HDR_READ_ERROR);
            else if (R == RHDR_HDR_MALLOC)
              return (POS_HDR_MALLOC);
            else if (R == RHDR_HDR_REALLOC)
              return (POS_HDR_REALLOC);
            else if (R == LOS_STATUS)
              return (PBACK_LOS);
            else if (R == NEXT_FILE_STATUS)
              return (PBACK_NEXT_FILE);
            else { return (R); }
         }
   
         ret_sec = *(ptr->TIME) / 1000;
         ret_nsec = (*(ptr->TIME) % 1000) * 1000000;
      }

    /***********************************************************************/
    /*  Initialize variables pertinent to the ir_fine_search routine.      */
    /***********************************************************************/
 
      ptr->time_row = 0;
      ptr->time_col = 0;
      ptr->cur_sen_set = 0;
      ex->accum_ss_sz = 0;
      ptr->reset_hdr = 1;
      ex->accum_ss_ms = 0;
      ex->accum_ss_ns = 0;
      ex->time_row_ms = 0;
      ex->time_row_ns = 0;
      ex->time_col_ms = 0;
      ex->time_col_ns = 0;
      ptr->chg_sen_set = 0;

    /***********************************************************************/
    /*  If requested time is greater than the start time of the current    */
    /*  record, perform a fine search; otherwise, the requested time may   */
    /*  be before the time of the first record (a data gap) or the time    */
    /*  requested matches the start time of the data record.               */
    /***********************************************************************/
   
      if (bY > *ptr->YEAR || 
         (bY == *ptr->YEAR && bD > *ptr->DAYOFYEAR) ||
         (bY == *ptr->YEAR && bD == *ptr->DAYOFYEAR && bS > ret_sec) ||
         (bY == *ptr->YEAR && bD == *ptr->DAYOFYEAR &&
          bS == ret_sec && bNs > ret_nsec)) {
         R = ir_fine_search (data_key, exten, vnum, idf_data_ptr, bY,
                                bD, bS, bNs);
         if (R != ALL_OKAY) {
            if (reset_called) { ginfo.called_locate = 0; }
            return (R);
         }
      } else { ex->num_sample = (ex->smp_id == 2) ? 1 : *ptr->N_SAMPLE; }

      ex->fnext = 1;
      ptr->reset_hdr = 1;
   }

   if (reset_called) { ginfo.called_locate = 0; }
  
   /********************************************************************/
   /*  Position the data set that is used to compute the pitch angles. */
   /*  After this is done, call LOCATE_EX to point back to original ex */
   /*  structure requested by the user.                                */
   /********************************************************************/

   if (ex->pa_def && ex->bmem.base_pitch_info != NO_MEMORY) {
      pa_ptr = ex->pitch_angles;
      if (pa_ptr->file_status == ALL_OKAY && pa_ptr->ptimes.byear == 0) {
         R = file_pos (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version,
                      pa_ptr->idf_data_ptr[0], bY, bD, bS, 
                      bNs, eY, eD, eS, eNs);
         if (R != ALL_OKAY) { return (FILE_POS_PA); }
      } else if (pa_ptr->file_status != ALL_OKAY) {
         R = ir_locate_ex (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version,
                              0);
         ex = ginfo.expt;
         ex->fnext = 1;
      }
      R = ir_locate_ex (data_key, exten, vnum, 0);
      ex = ginfo.expt;
   }

/*  Added CAG 12/23/98 - moved mem check into file_pos                       */

   R = ir_check_idf_data_memory (data_key, exten, vnum, idf_data_ptr);
   if (R != ALL_OKAY) { return (R); }

   return (ALL_OKAY);
}
