#include "libbase_udf.h"
#include "ret_codes.h"

/*******************************************************************************
 *                                                                             *
 *                        IR_ACQUIRE_PITCH_DATA SUBROUTINE                     *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to acquire data for the three components of the   *
 *  magnetic field used to compute pitch angles.  The data will either need to *
 *  be read or the previous sample will hold for the current delta-t           *
 *  processing.  If a header change is encountered, there is an need to check  *
 *  the sizes of arrays in the idf_data structure since the calls to read_drec *
 *  with the same data set, but different idf_data pointers, will result in    *
 *  the first call getting the idf_data structure updated, but the rest will   *
 *  not.  Data is acquired for one element of the sweep.                       *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *     struct pitch_times          structure that holds the time period for    *
 *         src_time                the current delta-t being processed         *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           *
 *    unsigned short swp_step      current step of the sweep being processed   *
 *    char same_times              flag indicating if the same time range is   *
 *                                 applicable to all steps of the vector       *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_acquire_pitch_data (src_time, pa_ptr, swp_step, same_times)       *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    read_drec()                  the universal read routine that retrieves   *
 *                                 the data for the time sample being processed*
 *    ir_same_pitch_time ()        sets all time values to the time of the     *
 *                                 first sensor processed for the data set     *
 *    ir_check_idf_data_memory ()  makes sure that all allocated arrays in the *
 *                                 idf_data structure are of sufficient size   *
 *    ir_process_pitch_data ()     processes the data for the three components *
 *                                 of the magnetic field                       *
 *    ir_pitch_los_next_file ()    handles the case when an LOS_STATUS or      *
 *                                 NEXT_FILE_STATUS is encountered             *
 *    ir_copy_pitch_angle ()       handles the case where the time for each    *
 *                                 sweep step is the same (all pitch angle     *
 *                                 values should be the same)                  *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct idf_data *PA_DATA     structure holding the data for the pitch    *
 *                                 angle component being processed             *
 *    register double *dptr        pointer to data for the three components    *
 *    register double *dptr_end    loop termination variable                   *
 *    register float *fptr         pointer to the normalization factors        *
 *    register int i               looping variable                            *
 *    short ret_val                the value returned by the called routine    *
 *    short read_code              status code returned from READ_DREC()       *
 *    char more_data               terminate processing of data flag           *
 *    char all_done                the number of components that have data     *
 *                                 ready for pitch angle calculation           *
 *    char full_swp                flag that indicates if 1 value is being     *
 *                                 requested or all values for the record      *
 *    char set_time                flag indicating if time is to be set or     *
 *                                 retrieved                                   *
 *    char data_adv                time advancement flag                       *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_acquire_pitch_data (struct pitch_times src_time, 
               struct pitch_info *pa_ptr, u_ByTe_2 swp_step, ByTe_1 same_times)
{
   struct idf_data *PA_DATA;
   register ReaL_8 *dptr, *dptr_end;
   register ReaL_4 *fptr;
   register ByTe_2 i;
   ByTe_2 ret_val, read_code;
   ByTe_1 more_data = 1, all_done, full_swp = 0, set_time, data_adv;

   /***********************************************************************/
   /*  Retrieve data for all three components so that the current sample  */
   /*  of the sweep can be processed.                                     */
   /***********************************************************************/

   while (more_data)
    {
      all_done = 0;

      /********************************************************************/
      /*  If the times of each step are the same, then simply copy the    */
      /*  value computed for step 0. (due to file pointer manipulation).  */
      /********************************************************************/

      if (same_times && swp_step != 0)
       {
         ir_copy_pitch_angle (pa_ptr, swp_step);
         more_data = 0;
         continue;
       }

      for (i = 0; i < 3; ++i)
       {
         /*********************************************************************/
         /*  Has the entire time delta been processed for this component?     */
         /*********************************************************************/

         PA_DATA = (struct idf_data *) pa_ptr->idf_data_ptr[i];
         data_adv = (i == 2) ? 1 : 0;
         if (!pa_ptr->all_done[i])
          {
            /*******************************************************************/
            /*  Does the next data sample need to be read from the data file?  */
            /*******************************************************************/

            if (pa_ptr->next_data[i])
             {
               read_code = read_drec (pa_ptr->data_key, pa_ptr->exten,
                                      pa_ptr->version, pa_ptr->idf_data_ptr[i],
                                      pa_ptr->sensor[i], 0, data_adv, full_swp);

               /*****************************************************************/
               /*  If an error was encountered, return to the calling routine.  */
               /*****************************************************************/

               if (read_code < 0)
                 return (read_code);

               /*****************************************************************/
               /*  Since the data set could have different TIME_OFF values for  */
               /*  the sensors within a data set, make sure data from the same  */
               /*  data set returns one consistent time value.                  */
               /*****************************************************************/

               set_time = (i == 0) ? 1 : 0;
               ir_same_pitch_time (pa_ptr, set_time, i);

               /*************************************************************/
               /*  Need to check the sizes of arrays in the idf_data        */
               /*  structure since header changed.                          */
               /*************************************************************/

               if (PA_DATA->hdr_change)
                {
                  ret_val = ir_check_idf_data_memory (pa_ptr->data_key,
                                               pa_ptr->exten, pa_ptr->version,
                                               pa_ptr->idf_data_ptr[0]);
                  if (ret_val != ALL_OKAY)
                    return (ret_val);

                  ret_val = ir_check_idf_data_memory (pa_ptr->data_key,
                                               pa_ptr->exten, pa_ptr->version,
                                               pa_ptr->idf_data_ptr[1]);
                  if (ret_val != ALL_OKAY)
                    return (ret_val);

                  ret_val = ir_check_idf_data_memory (pa_ptr->data_key,
                                               pa_ptr->exten, pa_ptr->version,
                                               pa_ptr->idf_data_ptr[2]);
                  if (ret_val != ALL_OKAY)
                    return (ret_val);
                }

               /***************************************************************/
               /*  Process the data just read for the current component.      */
               /***************************************************************/

               ret_val = ir_process_pitch_data (pa_ptr, src_time, i, swp_step);
               if (ret_val != ALL_OKAY)
                 return (ret_val);

               /***************************************************************/
               /*  Handle file crossing condition.                            */
               /***************************************************************/

               if (read_code == LOS_STATUS || read_code == NEXT_FILE_STATUS)
                {
                  ret_val = ir_pitch_los_next_file (pa_ptr, i);
                  if (ret_val != ALL_OKAY)
                    return (ret_val);
                }
             }

            /******************************************************************/
            /*  The existing data is utilized for current delta-t processing. */
            /******************************************************************/

            else
             {
               ret_val = ir_process_pitch_data (pa_ptr, src_time, i, swp_step);
               if (ret_val != ALL_OKAY)
                 return (ret_val);
             }
          }

         all_done += pa_ptr->all_done[i];
       }
   
      /******************************************************************/
      /*  Once the data for all three components has been acquired, the */
      /*  pitch angles can be computed.                                 */
      /******************************************************************/

      if (all_done == 3)
        more_data = 0;
    }

   /*********************************************************************/
   /*  Normalize the data for the three magnetic field components.      */
   /*********************************************************************/

   fptr = pa_ptr->time_frac + (3 * swp_step);
   dptr = pa_ptr->data_val + (3 * swp_step);
   dptr_end = dptr + 3;
   for (; dptr < dptr_end; ++dptr, ++fptr)
    {
      if (*fptr < 0.0)
        continue;

      *dptr /= *fptr;
    }

   /************************************************************************/
   /*  Reset the flag for each component to process the next delta-t.      */
   /************************************************************************/

   for (i = 0; i < 3; ++i)
     pa_ptr->all_done[i] = 0;

   return (ALL_OKAY);
}
