#include <stdlib.h>
#include "util_str.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/******************************************************************************
 *                                                                            *
 *                         IR_GET_TIME_COMPONENTS SUBROUTINE                  *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to get the individual time elements that are     *
 *  used to determine the time associated with a given sample.                *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_get_time_components ()                                               *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    abs()                      returns the absolute value of a number       *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    long milli_value           the value to apply to DATA_ACCUM in order to *
 *                               convert to milliseconds                      *
 *    long nano_value            the value to apply to DATA_ACCUM in order to *
 *                               convert to nanoseconds                       *
 *    unsigned short n_sample    number of samples returned for each sensor   *
 *    short exp_milli            power of 10 to use in order to convert       *
 *                               DATA_ACCUM to milliseconds                   *
 *    short exp_nano             the power of 10 to use in order to convert   *
 *                               DATA_ACCUM to nanoseconds                    *
 *    short i                    looping variable                             *
 *    short temp                 temporary variable used for looping          *
 *    short TotCol               number of data columns in current senor set  *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_get_time_components (void)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_4 milli_value, nano_value, TotCol;
  u_ByTe_2 n_sample;
  ByTe_2 exp_milli, exp_nano, i, temp;

  /****************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and  */
  /*  data information for the experiment currently being processed and point */
  /*  to the sensor structure for the sensor of interest.                     */
  /****************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  n_sample = *ptr->N_SAMPLE;
  TotCol = (ex->smp_id != 3) ? *ptr->N_SEN : *ptr->N_SEN * *ptr->N_COLS;


  /***************************************************************************/
  /*  Take the time elements that are in microseconds and convert that value */
  /*  into millisecond and nanosecond components.  Divide by 1000 converts   */
  /*  to milliseconds (1000 microseconds = 1 millisecond).  Modulus 1000     */
  /*  gets the remaining microseconds and multiplying that number by 1000    */
  /*  converts the number to nanoseconds.  (1000 nanoseconds = 1 microsecond)*/
  /***************************************************************************/

  ex->lat_ms = *ptr->DATA_LAT / 1000;
  ex->lat_ns = (*ptr->DATA_LAT % 1000) * 1000;
  ex->swp_ms = *ptr->SWP_RESET / 1000;
  ex->swp_ns = (*ptr->SWP_RESET % 1000) * 1000;
  ex->sen_ms = *ptr->SEN_RESET / 1000;
  ex->sen_ns = (*ptr->SEN_RESET % 1000) * 1000;

  /**************************************************************************/
  /*  To get the DATA_ACCUM time to milliseconds, find out the answer to    */
  /*  the equation -3 = TIME_UNITS + exponent.  To get the DATA_ACCUM time  */
  /*  to nanoseconds, find out the answer to the equation exponent = 9 +    */
  /*  TIME_UNITS.                                                           */
  /**************************************************************************/

  exp_milli = -1 * *ptr->TIME_UNITS - 3;
  temp = abs (exp_milli);
  for (i = 0, milli_value = 1; i < temp; ++i)
     milli_value *= 10;

  exp_nano = 9 + *ptr->TIME_UNITS;
  temp = abs (exp_nano);
  for (i = 0, nano_value = 1; i < temp; ++i)
     nano_value *= 10;

  /***************************************************************************/
  /*  If the exponent for conversion to milliseconds is 0 or negative, there */
  /*  are no nanosecond components because the time is in units greater than */
  /*  or equal to milliseconds. (seconds, etc)  The value is multiplied by   */
  /*  1.0 to return a floating point number to the user.                     */
  /***************************************************************************/

  if (exp_milli <= 0)
   {
     ex->accum_ms = *ptr->DATA_ACCUM * milli_value;
     ex->accum_ns = 0;
   }
  else 
   {
     /************************************************************************/
     /*  For accum_ns calculation, we assume TIME UNITS is not smaller than  */
     /*  -9, which means we should NOT get negative value for exp_nano so we */
     /*  should always multiply by nano_value.                               */
     /************************************************************************/

     ex->accum_ms = *ptr->DATA_ACCUM / milli_value;
     ex->accum_ns = (*ptr->DATA_ACCUM % milli_value) * nano_value;
   }

  /*************************************************************************/
  /*  Determine the time between acquisition of 2 consecutive data samples.*/
  /*************************************************************************/

  ex->time_sm_ms = ex->accum_ms + ex->lat_ms;
  ex->time_sm_ns = ex->accum_ns + ex->lat_ns;

  /****************************************************************************/
  /*  Determine the time between adjacent data samples within the same column.*/
  /****************************************************************************/

  ex->time_col_ms = ex->TCP * (ex->TRC * ex->time_sm_ms + (1 - ex->TRC) *
           ((TotCol * ex->TRP + (1 - ex->TRP)) * ex->time_sm_ms + ex->swp_ms));
  ex->time_col_ns = ex->TCP * (ex->TRC * ex->time_sm_ns + (1 - ex->TRC) *
           ((TotCol * ex->TRP + (1 - ex->TRP)) * ex->time_sm_ns + ex->swp_ns));

  /****************************************************************************/
  /*  Determine the time between adjacent data samples within the same row.   */
  /****************************************************************************/

  ex->time_row_ms = ex->TRP * ((1 - ex->TRC) * ex->time_sm_ms + ex->TRC * 
          ((n_sample * ex->TCP + (1 - ex->TCP)) * ex->time_sm_ms + ex->swp_ms));
  ex->time_row_ns = ex->TRP * ((1 - ex->TRC) * ex->time_sm_ns + ex->TRC * 
          ((n_sample * ex->TCP + (1 - ex->TCP)) * ex->time_sm_ns + ex->swp_ns));
}
