#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                          IR_RESET_HEADER SUBROUTINE                         *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called whenever a header change has been detected.  Some *
 *  of the elements that are returned to the user come from the header record. *
 *  In addition, the arrays for the sensor data, sample sequence values and    *
 *  azimuthal angles are allocated based on information returned in the header *
 *  record (N_SAMPLE).  All three arrays will end up being maxed out by the    *
 *  virtual instrument returning the most number of samples.                   *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    void *idf_data_ptr         ptr to the memory location for the structure  *
 *                               that holds returned data values (read_drec)   *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_reset_header (idf_data_ptr)                                       *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_check_critical_status() checks each sensor's critical status byte     *
 *                               value and takes appropriate action            *
 *    ir_get_time_components ()  retrieves time elements that are used to      *
 *                               calculate time associated with a sample       *
 *    ir_check_dependent_on_header() makes sure that memory dependent upon     *
 *                               header information is of sufficient size      *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info        structure that holds information concerning   *
 *        ginfo                  the experiment that is being processed        *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct idf_data *EXP_DATA  structure that holds all of the currently     *
 *                               returned data values to be processed          *
 *    struct experiment_info     a pointer to the structure that holds         *
 *          *ex                  specific experiment information               *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all    *
 *                               pointers to the header and data for the       *
 *                               experiment of interest                        *
 *    short ret_val              holds value returned by called routine        *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_reset_header (void *idf_data_ptr)
{
  extern struct general_info ginfo;

  struct idf_data *EXP_DATA;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_2 ret_val;

  /***********************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header */
  /*  and data information for the experiment currently being processed. */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  EXP_DATA = (struct idf_data *) idf_data_ptr;

  ret_val = ir_check_dependent_on_header (idf_data_ptr);
  if (ret_val != ALL_OKAY)
    return (ret_val);

  /**********************************************************************/
  /*  Return a header change flag to the user for their use, along with */
  /*  the year and day.  Since there is only one year/day value in the  */
  /*  header, return same value for both start/end time values.         */
  /**********************************************************************/

  EXP_DATA->hdr_change = 1;
  EXP_DATA->byear = *ptr->YEAR;
  EXP_DATA->bday = *ptr->DAYOFYEAR;
  EXP_DATA->eyear = *ptr->YEAR;
  EXP_DATA->eday = *ptr->DAYOFYEAR;
  ir_get_time_components ();

  /**********************************************************************/
  /*  Reset header information flags.                                   */
  /**********************************************************************/

  ptr->prev_hd = (*ptr->NSS > 0) ? *(ptr->HDR_OFF + ptr->cur_sen_set) :
                                   *ptr->HDR_OFF;
  ptr->reset_hdr = 0;

  /**********************************************************************/
  /*  Check the critical status byte for any change.                    */
  /**********************************************************************/

  ret_val = ir_check_critical_status ();
  if (ret_val != ALL_OKAY)
    return (ret_val);

  return (ALL_OKAY);
}
