#include "OpSySInD.h"
#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "gen_defs.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_READ_HEADER SUBROUTINE                        *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called anytime a new header record needs to be read and *
 *  includes all the code needed for realtime acquisition of data.            * 
 *                                                                            *
 *  Memory is allocated and pointers are set when the first header read is    *
 *  made. After that a check is made to see if the size of the header record  *
 *  has increased, and if so, memory is increased and the pointers are reset. *
 *  If the size hasn't increased, the pointers that need to be redefined      *
 *  based on the values read from the header are processed. If an EOF code is *
 *  encountered during a read or if a  partial record is read, an internal    *
 *  flag is set so that the next time read_drec is called, another attempt    *
 *  is made to read the header record from the file. This additional attempt  *
 *  is made for realtime processing only. A partial record is considered a    *
 *  fatal error for playback processing.                                      *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *idf_data_ptr           ptr to memory location for structure that  *
 *                                 holds returned data values (read_drec)     *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_read_header (idf_data_ptr)                                       *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    lseek()                      moves the file pointer to a location       *
 *                                 within the file                            *
 *    read()                       reads N bytes from the file associated     *
 *                                 with the file descriptor given             *
 *    malloc()                     allocates memory                           *
 *    free ()                      frees previously allocated memory          *
 *    ir_assign_pointers()         assigns the pointers for the header, data, *
 *                                 and calibration data elements              *
 *    ir_fix_pointers ()           assigns the pointers for those elements    *
 *                                 whose size may vary from header to header  *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure that holds information concern-  *
 *                                 ing the experiment that is being processed *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    struct ptr_rec *ptr          a pointer to the structure which holds     *
 *                                 all pointers to the header and data for    *
 *                                 the experiment of interest                 *
 *    size_t B                     the number of bytes to allocate            *
 *    int nB                       the no. of bytes requested from the file   *
 *    int retB                     the no. of bytes actually read by READ()   *
 *    long rS                      the value returned by the seek routine     *
 *    short hSize                  the size of the header record              *
 *    short rval                   the value returned by the called routine   *
 *    char incHdr                  flag that indicates if a larger header     *
 *                                 was read                                   *
 *    void *tPtr                   pointer which holds address passed back by *
 *                                 the call to the MALLOC routine             *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_read_header (void *idf_data_ptr)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct ptr_rec *ptr; 
   size_t B;

   int nB, retB;
   ByTe_4 rS;
   ByTe_2 hSize, rval;
   ByTe_1 incHdr;
   void *tPtr;

  /*************************************************************************/
  /*  Set pointer to the structure holding all pointers for the header and */
  /*  data information for the experiment currently being processed.       */
  /*************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;

  /*************************************************************************/
  /*  Determine the size of the header. This is given in the first 2 bytes */
  /*  of the header record.                                                */
  /*************************************************************************/

   incHdr = 0;
   nB = sizeof(ByTe_2);
   retB = read (ex->fdh, (ByTe_1 *) &hSize, nB);
   if (ex->BswaP) { ReOrderBytes (2, (u_ByTe_1 *)&hSize, 1); }

  /*************************************************************************/
  /*  There are three return conditions.  If the read returned a value < 0 */
  /*  then this is a hard error.  If the read did not return the number of */
  /*  requested bytes then this is an error for playback processing but    */
  /*  not for realtime processing where we just need to wait for the data  */
  /*  to show up.                                                          */
  /*************************************************************************/

  /*************************************************************************/
  /*  A hard read error was encountered on the file.                       */
  /*************************************************************************/

   if (retB < 0) { return (RHDR_READ_ERROR); } 

  /*************************************************************************/
  /*  Not the correct number of bytes returned from read.  If this is a    */
  /*  a realtime read then back off the number of bytes read.  If this is  */
  /*  a playback read then we have an error.                               */
  /*************************************************************************/

   if (retB != nB) {
      ex->hrec_eof = 1;
      if (ex->btime_sec >= 0) { return (RHDR_READ_ERROR); }
      if ( retB > 0 ) { rS = lseek (ex->fdh, (ByTe_4)(-1 * retB), 1); }
      return (EOF_STATUS);
   }

  /*************************************************************************/
  /*  The header is a fill header (header not yet received on LAN by the   */
  /*  listener).  Back up the bytes read to get the header size in the     */
  /*  anticipation of future read attempts.                                */
  /*************************************************************************/

   if (hSize == 0 || hSize == NO_MORE_DATA || hSize == NEXT_FILE) {
      ex->hrec_eof = 1;
      retB = sizeof(ByTe_2);
      rS = lseek (ex->fdh, (ByTe_4) (-1 * retB), 1);

  /*************************************************************************/
  /*  The listener has encountered an LOS or EOF while waiting on the      */
  /*  redelivery of the header record in question.                         */
  /*************************************************************************/
   
      if (hSize == NO_MORE_DATA) {
         return (LOS_STATUS);
      } else if (hSize == NEXT_FILE) {
         return (NEXT_FILE_STATUS);
      } else if (ex->btime_sec >= 0) {
        return (FILL_HEADER);
      } else { return (EOF_STATUS); }
   }

  /*************************************************************************/
  /*  If space has been previously allocated, free old memory before new   */
  /*  space is allocated.                                                  */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc           */
  /*************************************************************************/

   if (hSize > ptr->h_size) {
      B = hSize;
      if (B <= 0) { return (RHDR_HDR_MALLOC); }

      if (ex->bmem.base_hdr != NO_MEMORY) {
         free (ex->bmem.base_hdr);
         ex->bmem.base_hdr = NO_MEMORY;
      }
      if ((tPtr = malloc (B)) == NO_MEMORY) { return (RHDR_HDR_MALLOC); }
      ex->bmem.base_hdr = tPtr;
      ex->HEADER_MEM = (ByTe_1 *) ex->bmem.base_hdr;
      incHdr = 1;
   } 

  /*************************************************************************/
  /*  Read in the new header information.                                  */
  /*************************************************************************/

   nB = hSize - sizeof(ByTe_2);
   retB = read (ex->fdh, ex->HEADER_MEM + sizeof (ByTe_2), nB);

  /*************************************************************************/
  /*  A read error was encountered on the file.                            */
  /*************************************************************************/

   if (retB < 0) { return (RHDR_READ_ERROR); } 

  /*************************************************************************/
  /*  Not the correct number of bytes returned from read.  If this is a    */
  /*  a realtime read then back off the number of bytes read.  If this is  */
  /*  a playback read then we have an error.                               */
  /*************************************************************************/

   if (retB != nB) {
      ex->hrec_eof = 1;
      if (ex->btime_sec >= 0) { return (RHDR_READ_ERROR); }

      nB = retB + sizeof (ByTe_2);
      rS = lseek (ex->fdh, (ByTe_4) (-1 * retB), 1);
      return (EOF_STATUS);
   } 

  /*************************************************************************/
  /*  Assign/adjust the pointers to the header, data, and cal. elements.   */
  /*************************************************************************/

   if (ex->BswaP) { ReOrderUdf (0); }

   ptr->h_size = hSize;
   if (incHdr) {
      rval = ir_assign_pointers (1, 0, idf_data_ptr);
      if (rval != ALL_OKAY) { return (rval); }
   } else {
      rval = ir_fix_pointers (idf_data_ptr);
      if (rval != ALL_OKAY) { return (rval); }
   }

   return (ALL_OKAY);
}
