#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                       IR_POSITION_REAL_TIME SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to position the data and header file pointers     *
 *  for the real time data files.  If the user entered a -1 for the start      *
 *  time, the file descriptor for the data file is positioned at the beginning *
 *  of the data file.  If the user entered a -2 for the start time, the file   *
 *  descriptor for the data file is positioned one record from the end of the  *
 *  data file, with the reason being so that the EOF or LOS record can be      *
 *  processed correctly.  This check is made to ensure that the user would not *
 *  be forever stuck waiting for another data record to be placed in this data *
 *  file.  One data and the corresponding header record is read so that        *
 *  information is ready to be processed when the user calls READ_DREC().      *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    long btime               the start time requested                        *
 *    void *idf_data_ptr       ptr to the memory location for the structure    *
 *                             that holds returned data values (read_drec)     *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_position_real_time (btime, idf_data_ptr)                         *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    sizeof ()                the size of the specified object in bytes       *
 *    read()                   reads N bytes from the file associated with     *
 *                             the file descriptor given                       *
 *    lseek()                  moves the file pointer to a location within     *
 *                             the file                                        *
 *    ir_alloc_exp_once()      allocates memory that is needed in order to     *
 *                             utilize the generic routines provided for       *
 *                             retrieving experiment data                      *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info      structure that holds information concerning     *
 *        ginfo                the experiment that is being processed          *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info   a pointer to the structure that holds specific  *
 *          *ex                experiment information                          *
 *    struct ptr_rec *ptr      a pointer to the structure which holds all      *
 *                             pointers to the header and data for the         *
 *                             experiment of interest                          *
 *    int num_bytes            the number of bytes requested from the file     *
 *    int ret_bytes            the number of bytes actually read by READ()     *
 *    long num_records         the number of data records in the data file     *
 *    long bytes               no. of bytes to seek by for the LSEEK routine   *
 *    long ret_val             holds the value returned by the called routine  *
 *    long *hdr_off            the offset into the header file (in bytes)      *
 *    short rval               holds the value returned by the called routine  *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_position_real_time (ByTe_4 btime, void *idf_data_ptr)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct ptr_rec *ptr;
   int num_bytes, ret_bytes;
   ByTe_4 num_records, bytes, ret_val, *hdr_off;
   ByTe_2 rval;

   /***************************************************************************/
   /*  Set a pointer to the structure which holds all pointers for header and */
   /*  data information for the experiment currently being processed.         */
   /***************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;

   /**************************************************************************/
   /*  If start time is -1, set the pointer at the beginning of the file.    */
   /*  If start time is -2, set the pointer at the end of the file.          */
   /**************************************************************************/

   if (btime == -1)
     ret_val = lseek (ex->fdd, 0L, 0);
   else 
    {
      ret_val = lseek (ex->fdd, 0L, 2);

      /******************************************************************/
      /*  Make sure that the pointer is on a record boundary.  Back up  */
      /*  one record from the end of the file.                          */
      /******************************************************************/

      num_records = ret_val / ptr->d_size;
      bytes = (num_records - 1) * ptr->d_size;                
      if (bytes < 0)
        bytes = 0;
      ret_val = lseek (ex->fdd, bytes, 0);
    }

   /***********************************************************************/
   /*  Read the data record and the appropriate header record.            */
   /***********************************************************************/

   num_bytes = ptr->d_size;
   ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
   if (ret_bytes < 0)
     return (POS_DATA_READ_ERROR);
   else if (ret_bytes == 0)
    {
      ex->drec_eof = 1;
      return (EOF_STATUS);
    }

   /********************************************************************/
   /*  Either we read before the listener could finish writing, or the */
   /*  listener never received the file closure packet and wrote down  */
   /*  20 bytes of -1 or -2 values.                                    */
   /********************************************************************/

   if (ex->BswaP)
      ReOrderUdf (1);
   if (ret_bytes != num_bytes)
    {
      /********************************************************************/
      /*  The file closure flags are always in the first hdr_offset field.*/
      /*  There are three long data values in the data structure before   */
      /*  the header offsets.                                             */
      /********************************************************************/

      hdr_off = (ByTe_4 *) (ex->DATA_MEM + 3 * sizeof (ByTe_4));

      /*******************************************************************/
      /*  We read before the listener could finish writing so back up    */
      /*  the partial no. of bytes read for future read attempts.        */
      /*******************************************************************/

      ex->drec_eof = 1;
      ret_val = lseek (ex->fdd, (ByTe_4) (-1 * ret_bytes), 1);
      if (*hdr_off == NO_MORE_DATA)
        return (LOS_STATUS);
      else if (*hdr_off == NEXT_FILE)
        return (NEXT_FILE_STATUS);
      else
        return (EOF_STATUS);
    }
   ret_val = lseek (ex->fdh, (ByTe_4)*(ptr->HDR_OFF), 0);

   /************************************************************************/
   /*  Allocate memory that is needed in order to utilize the generic      */
   /*  routines provided for retrieving experiment data.  This call will   */
   /*  read the header record that is associated with the data record.     */
   /************************************************************************/
 
   rval = ir_alloc_exp_once (idf_data_ptr);
   if (rval != ALL_OKAY)
      return (rval);

   /************************************************************************/
   /*  Initialize variables pertinent to the READ_DREC routine and return  */
   /*  to the calling routine.                                             */
   /************************************************************************/

   ptr->time_row = 0;
   ptr->time_col = 0;
   ptr->cur_sen_set = 0;
   ex->accum_ss_sz = 0;
   ex->accum_ss_ms = 0;
   ex->accum_ss_ns = 0;
   ptr->chg_sen_set = 0;

   ptr->reset_hdr = 1;
   ex->fnext = 1;
   ex->num_sample = (ex->smp_id == 2) ? 1 : *ptr->N_SAMPLE;

   return (ALL_OKAY);
}
