#include <math.h>
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                           IR_POLY_EXPAND SUBROUTINE                         *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine will calculate data values using a set of polynomial        *
 *  coefficients that are stored in the table that is passed to this routine.  *
 *  if the order of the polynomial is 0 the routine will return the value in   *
 *  the table at the position of the input variable.  Note that there is a     *
 *  tacit assumption made that no table will have more elements in it than a   *
 *  the largest positive signed long and hence both unsigned and signed longs  *
 *  are equivalent in this case and so signed type is used.  Passing a float   *
 *  with a table size of 0 will have unexpected consequences.                  *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    float DaTaIn               the data value to be applied in the equation  *
 *    float *tbl                 the table that holds the coefficients         *
 *    long tbl_sz                the number of coefficients in the table       *
 *    char T                     the input type                                *
 *                                  <0 = converted float;                      *
 *                                   0 = unsigned;                             *
 *                                   1 = signed                                *
 *                                  >1 = idfs float                            *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_poly_expand (inp_data, tbl, tbl_sz)                               *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    ir_intfp2fp ()       converts idfs float to system float                 *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    register float *C          pointer to the coefficients                   *
 *    register float V           input data cast as float                      *
 *    register float *stop_loop  loop termination variable                     *
 *    register double x          the result of x**0, x**1, x**2, etc.          *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ReaL_8 ir_poly_expand (ByTe_4 *DaTaIn, ReaL_4 *tbl, ByTe_4 tbl_sz, ByTe_1 T)
{
   register ReaL_4 *C, *StoP;
   register ReaL_8 x;
   register ByTe_4 Low, High, Mid;
   ReaL_8 V, rV, Tval, Dif1, Dif2;
   ByTe_4 Max;
   ByTe_2 Op;
   ByTe_1 InvOK;

   Op = (tbl_sz == 0) ? 0 : (tbl_sz < 0) ? 1 : 2;    /* operation            */

   if (T < 0)                                        /* input is float?      */
      V = *(ReaL_8 *)DaTaIn;                         /* cast it here         */
   else                                              /* an IDFS type value   */
   {                                                 /* BEGIN IDFS VALUE     */
      if (T < 2)                                     /* some kind of integer */
      {                                              /* BEGIN SET INTEGER    */
         if (T == 0)                                 /* unsigned?            */
            V = (u_ByTe_4)*DaTaIn;                   /* set it here          */
         else                                        /* must be signed       */
            V = *DaTaIn;                             /* so set it here       */
      }                                              /* END SET INTEGER      */
      else                                           /* its a float!         */
         V = ir_intfp2fp ((u_ByTe_4)*DaTaIn, T);     /* convert it here      */
   }                                                 /* END IDFS VALUE       */

   switch (Op)                                       /* operation switch     */
   {                                                 /* BEG OPERATION SWITCH */
      case 0:                                        /* START LOOKUP         */
         rV = (ReaL_8) *(tbl + (ByTe_4)(V + .001));  /* spit out value       */
      break;                                         /* STOP LOOKUP          */
      case 1:                                        /* START INVERSE        */
         if (tbl_sz < -3)                            /* can we invert        */
         {                                           /* BEG INVERT CHECK     */
            Dif1 = *tbl - *(tbl + 1);                /* diff first two vals  */
            Dif2 = *(tbl + 1) - *(tbl + 2);          /* diff next two vals   */
            InvOK = (Dif1 * Dif2 < 0) ? 0 : 1;       /* set Inv OK flag      */
         }                                           /* END INVERT CHECK     */
         else                                        /* no check possible    */
           InvOK = 1;                                /* always OK            */

         if (tbl_sz == -1)                           /* only one element     */
            rV = 0;                                  /* it's always closest  */
         else                                        /* normal case          */
         {                                           /* BEG NORMALACY        */
            if (InvOK)                               /* Ok to invert         */
            {                                        /* BEG INVERT           */
               C = tbl;                              /* table values         */
               Low = 0;                              /* minimum position     */
               High = -tbl_sz - 1;                   /* maximum position     */
               Max = -tbl_sz - 1;                    /* maximum position     */
               if (*(C + 1) > *C)                    /* ascending table      */
               {                                     /* BEG GOING UP         */
                  while (Low <= High)                /* binary search loop   */
                  {                                  /* BEG BINARY SEARCH    */
                     Mid = (Low + High) / 2;         /* middle position      */
                     Tval = *(C + Mid);              /* mid value            */
                     if (V <= Tval)                  /* less than mid        */
                        High = Mid - 1;              /* lower the top        */
                     else                            /* other way            */
                        Low = Mid + 1;               /* raise the bottom     */
                  }                                  /* END BINARY SEARCH    */
               }                                     /* END GOING UP         */
               else                                  /* other way            */
               {                                     /* BEG GOING DOWN       */
                  while (Low <= High)                /* binary search loop   */
                  {                                  /* BEG BINARY SEARCH    */
                     Mid = (Low + High) / 2;         /* middle position      */
                     Tval = *(C + Mid);              /* mid value            */
                     if (V <= Tval)                  /* less than mid        */
                        Low = Mid + 1;               /* raise the bottom     */
                     else                            /* other way            */
                        High = Mid - 1;              /* lower the top        */
                  }                                  /* END BINARY SEARCH    */
               }                                     /* END GOING DOWN       */
               if ((Low < 0 || Low > Max) || 
                           (High < 0 || High > Max)) /* out of limits        */
                  rV = Mid;                          /* index is midpoint    */
               else                                  /* get index            */
               {                                     /* BEG CLOSEST INDEX    */
                  Dif1 = (V > *(C + High)) ? V - *(C + High) : *(C + High) - V;
                  Dif2 = (V > *(C + Low)) ? V - *(C + Low) : *(C + Low) - V;
                  rV = (Dif1 >= Dif2) ? Low : High;  /* choose this index    */
               }                                     /* END CLOSEST INDEX    */
            }                                        /* BEG INVERT           */
            else                                     /* no invert            */
              rV = V;                                /* return value         */
         }                                           /* END NORMALACY        */
      break;                                         /* STOP INVERSE         */
      case 2:                                        /* START POLYNOMIAL     */
         C = tbl;                                    /* polynomial coef      */
         StoP = C + tbl_sz;                          /* last coefficient     */
         x = 1.0;                                    /* first multiplier     */
         rV = *C++;                                  /* intial value         */

         for (; C < StoP; )                          /* coefficient loop     */
         {                                           /* BEGIN POLY LOOP      */
            x *= V;                                  /* update multiplier    */
            rV += *C++ * x;                          /* add next coef        */
         }                                           /* END POLY LOOP        */

      break;                                         /* STOP POLYNOMIAL      */
   }                                                 /* END OPERATION SWITCH */

   return (rV);                                      /* send back value      */
}

