#include "util_str.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                          IR_PITCH_END_TIME SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to determine if the current sample is sufficient  *
 *  to terminate data acquisition for the current component and if the current *
 *  data is to be used for the next delta-t processing.  The end time of the   *
 *  sample is compared against the end time of the delta-t being processed.    *
 *  If the sample end time is beyond the delta-t end time, keep the data for   *
 *  the next delta-t processing (no read) and this delta-t processing is all   *
 *  done.  If the sample end time is equal to the delta-t time, get the next   *
 *  data set and this delta-t processing is all done.   If the sample end time *
 *  is before the delta-t end time, get the next data set and this delta-t     *
 *  processing needs to continue.  Unlike the SCF software, do not use the     *
 *  nanoseconds time component; only consider the milliseconds.                *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           * 
 *    struct pitch_times           structure that holds the time period for    *
 *        src_time                 the current delta-t being processed         *
 *    short eyear                  year value for the end time of the sample   *
 *    short eday                   day value for the end time of the sample    *
 *    long emilli                  time of day value for the end time of the   *
 *                                 sample                                      *
 *    short comp                   index specifying which of the 3 components  *
 *                                 is being processed                          *
 *                                                                             *
 *  USAGE                                                                      *
 *    ir_pitch_end_time (&pa_ptr, src_time, eyear, eday, emilli, comp)         *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

void ir_pitch_end_time (struct pitch_info *pa_ptr, struct pitch_times src_time,
                        ByTe_2 eyear, ByTe_2 eday, ByTe_4 emilli, ByTe_2 comp)
{
   /***********************************************************************/
   /*  Check the year time component.                                     */
   /***********************************************************************/

   if (eyear > src_time.eyear)
    {
      pa_ptr->next_data[comp] = 0;
      pa_ptr->all_done[comp] = 1;
    }

   /***********************************************************************/
   /*  Check other time components besides year.                          */
   /***********************************************************************/

   else if (eyear == src_time.eyear)
    {
      if (eday > src_time.eday)
       {
         pa_ptr->next_data[comp] = 0;
         pa_ptr->all_done[comp] = 1;
       }

      /********************************************************************/
      /*  Check other time components besides day.                        */
      /********************************************************************/

      else if (eday == src_time.eday)
       {
         if (emilli > src_time.emilli)
          {
            pa_ptr->next_data[comp] = 0;
            pa_ptr->all_done[comp] = 1;
          }

         /********************************************************************/
         /*  Check time of day residual.                                     */
         /********************************************************************/

         else if (emilli == src_time.emilli)
          {
            pa_ptr->next_data[comp] = 1;
            pa_ptr->all_done[comp] = 1;
          }
         else
           pa_ptr->next_data[comp] = 1;
       }
      else
        pa_ptr->next_data[comp] = 1;
    }
   else
      pa_ptr->next_data[comp] = 1;
}
