#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"
#include "Server.h"
#include "libCfg.h"
#include "OpSySInD.h"

/*******************************************************************************
 *                                                                             *
 *                      IR_PLAYBACK_DATA_FILES SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This module is called to determine if the playback data files for the    *
 *  requested time period and data set are available for usage.                *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long data_key       key which uniquely identifies the data set  *
 *                                 being processed                             *
 *    short btime_yr               the start time requested (year component)   *
 *    short btime_day              the start time requested (day component)    *
 *    long btime_sec               the start time requested (seconds component)*
 *    long btime_nsec              the start time requested (nanoseconds)      *
 *    short etime_yr               the stop time requested (year component)    *
 *    short etime_day              the stop time requested (day component)     *
 *    long etime_sec               the stop time requested (seconds component) *
 *    long etime_nsec              the stop time requested (nanoseconds)       *
 *    char *exten                  filename extension for the data to be used  *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_playback_data_files (data_key, btime_yr, btime_day, btime_sec,    *
 *                             btime_nsec, etime_yr, etime_day, etime_sec      *
 *                             etime_nsec, exten)                              *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    strcpy()                     copies a string to another string variable  *
 *    open()                       opens the file requested                    *
 *    pbr()                        the playback data request routine           *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info          structure that holds information concerning *
 *         ginfo                   the experiment that is being processed      *
 *    char plot_package            flag which indicates if the plotting or menu*
 *                                 program made the call (will not be changed  *
 *                                 from the default of plotting for the generic*
 *                                 user)                                       *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    int rval1, rval2, rval3      holds value returned by the routine PBR()   *
 *    short ret_code               return code                                 *
 *    char hname[]                 the full path name of the header file       *
 *    char dname[]                 the full path name of the data file         *
 *    char vname[]                 the full path name of the VIDF file         *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_playback_data_files (u_ByTe_4 data_key, ByTe_2 btime_yr, 
                               ByTe_2 btime_day, ByTe_4 btime_sec, 
                               ByTe_4 btime_nsec, ByTe_2 etime_yr, 
                               ByTe_2 etime_day, ByTe_4 etime_sec, 
                               ByTe_4 etime_nsec, ByTe_1 *exten)
{
   extern struct general_info ginfo;
   extern ByTe_1 plot_package;
   extern LinkList Projects;

   StrHier DSrc;
   Time_t btm, etm;
   struct experiment_info *ex;
   int rval1, rval2, rval3;
   ByTe_4 dTm[3], vTm[3];
   ByTe_2 rV1, rV2, rV3;
   ByTe_2 ret_code, params[5];
   ByTe_1 hname[NAME_SIZE], dname[NAME_SIZE], vname[NAME_SIZE];

   /***************************************************************************/
   /*  If the data files have already been opened, no need to open them again.*/
   /***************************************************************************/

   ex = ginfo.expt;
   if (ex->fdh != -1) { return (ALL_OKAY); }

   key_to_fields (data_key, params);
   DSrc = SourceByNum (Projects, (ByTe_1 *) params[0], (ByTe_1 *) params[1],
                       (ByTe_1 *) params[2], (ByTe_1 *) params[3], 
                       (ByTe_1 *) params[4], (ByTe_1 *) 0);
   if (DSrc == NULL) { return (PBACK_BAD_SRC); }

   btm.yr = btime_yr;
   btm.day = btime_day;
   btm.msec = (btime_sec * 1000) + (btime_nsec / 1000000);

   etm.yr = etime_yr;
   etm.day = etime_day;
   etm.msec = (etime_sec * 1000) + (etime_nsec / 1000000);

   /**************************************************************************/
   /*  When plotting the data, all three files must be online and ready to   */
   /*  go.  For now, the database only works in terms of seconds, so the     */
   /*  nanoseconds parameter is ignored.  This may change later.             */
   /**************************************************************************/
    
   if (plot_package) {

      rval1 = pbr (btm, etm, DSrc, _HEADER, exten, hname, dTm);
      rval2 = pbr (btm, etm, DSrc, _DATA, exten, dname, dTm);
      rval3 = pbr (btm, etm, DSrc, _VIDF, exten, vname, vTm);

      if (rval3 == ALL_OKAY) {
         strcpy (ex->idf_name, vname);
         if ((ex->fdi = open (vname, O_RDONLY)) >= 0 ) {
#ifdef _CyGnUsC
            setmode (ex->fdi, O_BINARY);
#endif
#ifdef WIN32
            _setmode (ex->fdi, O_BINARY);
#endif
         }
      }

      rV1 = ((rval1 < 0) || (rval1 == NO_DATA)) ? -1 : 0; 
      rV2 = ((rval2 < 0) || (rval2 == NO_DATA)) ? -3 : 0; 
      rV3 = ((rval3 < 0) || (rval3 == NO_DATA)) ? -5 : 0; 

      ret_code = rV1 + rV2 + rV3;

      if ( ret_code < 0 )
         return (-400 + ret_code);
      
      /***********************************************************************/
      /*  If any of the requests result in no data for the time frame        */
      /*  specified, return this status.                                     */
      /***********************************************************************/

      else if (rval1 == NO_DATA_AOS || rval2 == NO_DATA_AOS || 
               rval3 == NO_DATA_AOS)
         return (NO_DATA_AOS);

      /************************************************************************/
      /*  If any of the requests result in data retrieval, return this status.*/
      /************************************************************************/

      else if (rval1 == RETRIEVING || rval2 == RETRIEVING || rval3 ==RETRIEVING)
         return (RETRIEVING);

      if (rval1 == RETRIEVING_DATA || rval2 == RETRIEVING_DATA || 
	       rval3 == RETRIEVING_DATA)
         ret_code = RETRIEVING_DATA;
      else {

         strcpy (ex->header_name, hname);
         if ((ex->fdh = open (hname, O_RDONLY)) < 0)
            return (PBACK_NO_HEADER);

#ifdef _CyGnUsC
         setmode (ex->fdh, O_BINARY);
#endif
#ifdef WIN32
         _setmode (ex->fdh, O_BINARY);
#endif

         strcpy (ex->data_name, dname);
         if ((ex->fdd = open (dname, O_RDONLY)) < 0)
            return (PBACK_NO_DATA);

#ifdef _CyGnUsC
         setmode (ex->fdd, O_BINARY);
#endif
#ifdef WIN32
         _setmode (ex->fdd, O_BINARY);
#endif

         ex->btime_year = dTm[0];
         ex->btime_day  = dTm[1];
         ex->btime_sec  = (ByTe_4)dTm[2]/1000;
         ex->btime_nsec = ((dTm[2] % 1000) + 1) * 1000000; 
         ret_code = ALL_OKAY;
      }
   } else {

   /**************************************************************************/
   /*  For the menus, only the VIDF file needs to be online.                 */
   /**************************************************************************/
    
      rval1 = pbr (btm, etm, DSrc, _VIDF, exten, vname, vTm);
      if (rval1 != ALL_OKAY) {
         return ((ByTe_2) rval1);
      } else { 
         strcpy (ex->idf_name, vname);
         if ((ex->fdi = open (vname, O_RDONLY)) < 0 ) {
             return (PBACK_NO_VIDF);
         } else { 
#ifdef _CyGnUsC
            setmode (ex->fdi, O_BINARY);
#endif
#ifdef WIN32
            _setmode (ex->fdi, O_BINARY);
#endif
         }
         ret_code = ALL_OKAY;
      }
   }

   return (ret_code);
}
