#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                          IR_NEXT_IDF_FILE SUBROUTINE                       *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called in order to determine if a new VIDF file is      *
 *  needed in order to continue processing the data.  The VIDF file is time   *
 *  stamped in a way that indicates the time range for which the VIDF file is *
 *  applicable.  If the current time has exceeded the end time for the VIDF   *
 *  file, the existing file must be closed, a new file opened and memory      *
 *  obtained accordingly.                                                     *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *idf_data_ptr         ptr to the memory location for the structure *
 *                               that holds returned data values (read_drec)  *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_next_idf_file (idf_data_ptr)                                     *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *EXP_DATA  structure that holds all of the currently    *
 *                               returned data values to be processed         *
 *    struct experiment_info     pointer to the structure that holds specific *
 *          *ex                  experiment information                       *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    long cur_time              the current time sample being processed      *
 *    long max_time              the maximum time sample associated with the  *
 *                               current VIDF file                            *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_next_idf_file (void *idf_data_ptr)
{
   extern struct general_info ginfo;

   struct idf_data *EXP_DATA;
   struct experiment_info *ex;
   struct ptr_rec *ptr;
   ByTe_4 cur_time;

   /**************************************************************************/
   /*  Set pointer to the structure which holds all pointers for header and  */
   /*  data information for the experiment currently being processed.        */
   /***************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;
   EXP_DATA = (struct idf_data *) idf_data_ptr;

   cur_time = EXP_DATA->bmilli + EXP_DATA->bnano / 1000000;

   /************************************************************************/
   /*  The time has exceeded the current VIDF time frame.                  */
   /************************************************************************/

   if (*ptr->YEAR > ex->de_year || *ptr->YEAR < ex->ds_year)
   {
      return (NEED_NEW_VIDF);
   }
   else if ((*ptr->YEAR == ex->de_year && *ptr->DAYOFYEAR > ex->de_day) ||
            (*ptr->YEAR == ex->ds_year && *ptr->DAYOFYEAR < ex->de_day)) 
   {
      return (NEED_NEW_VIDF);
   }
   else if ((*ptr->YEAR == ex->de_year && *ptr->DAYOFYEAR == ex->de_day &&
            cur_time > ex->de_msec) ||
            (*ptr->YEAR == ex->ds_year && *ptr->DAYOFYEAR == ex->ds_day &&
            cur_time < ex->ds_msec)) 
   {
      return (NEED_NEW_VIDF);
   }

   return (ALL_OKAY);
}
