#include <stdlib.h>
#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_NEW_HEADER_RECORD SUBROUTINE                  *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called by the read_drec routine whenever a new header   *
 *  record needs to be read, which is determined by the data record being     *
 *  processed.  If upon record retrieval, an EOF code is encountered or if a  *
 *  partial record is read, an internal flag is set so that the next time     *
 *  read_drec is called, another attempt is made to read from the header file.*
 *  This additional attempt is made only for real-time processing.  For       *
 *  playback, a partial read is a fatal error.                                *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    char found_sensor          flag which tells if the requested sensor is  *
 *                               among the sensors returned and is associated *
 *                               with the time period being processed         *
 *    void *idf_data_ptr         ptr to the memory location for the structure *
 *                               that holds returned data values (read_drec)  *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_new_header_record (found_sensor, idf_data_ptr)                   *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    lseek()                    moves the file pointer to a location within  *
 *                               the file                                     *
 *    ir_read_header()           reads data from the header file              *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    long ret_val               value returned by a called routine           *
 *    long hdr_offset            indicates which header record to access      *
 *    long num_nss               the number of sensor sets within the record  *
 *    long num_items             the number of elements in the data array     *
 *    long i                     looping variable                             *
 *    short rval                 value returned by a called routine           *
 *    short old_day              day-of-year before header is read            *
 *    short new_day              day-of-year after header is read             *
 *    short day_diff             determines if a day changed                  *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_new_header_record (ByTe_1 found_sensor, void *idf_data_ptr)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_4 ret_val, hdr_offset;
  ByTe_2 rval, old_day, new_day, day_diff;

  /***********************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header */
  /*  and data information for the experiment currently being processed. */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;

  /********************************************************************/
  /*  Do we need to read another header record ?  If NSS is negative, */
  /*  there is only one header to be utilized by all sensor sets.     */
  /********************************************************************/

  hdr_offset = (*ptr->NSS > 0) ? *(ptr->HDR_OFF + ptr->cur_sen_set) : 
                                 *(ptr->HDR_OFF + 0);
  if (hdr_offset != ptr->prev_hd)
   {
      old_day = *ptr->DAYOFYEAR;
      ptr->reset_hdr = 1;
      ex->hrec_eof = 0;

      /***************************************************************/
      /*  Check for LOS and/or next file signals.  The check is made */
      /*  here to ensure that the reset flag is set.                 */
      /***************************************************************/

      if (hdr_offset == NO_MORE_DATA)
        return (LOS_STATUS);
      else if (hdr_offset == NEXT_FILE)
        return (NEXT_FILE_STATUS);

      /**********************************************************************/
      /*  Find the position in header file, seek to it, and read.           */
      /*  Since ir_read_header() is called by more than one routine, return */
      /*  an error code that indicates which calling routine resulted in    */
      /*  the error and what the error was.                                 */
      /**********************************************************************/

      ret_val = lseek (ex->fdh, hdr_offset, 0);
      rval = ir_read_header (idf_data_ptr);
      if (rval != ALL_OKAY)
       {
         if (rval == RHDR_READ_ERROR)
           return (DREC_HDR_READ_ERROR);
         else if (rval == RHDR_HDR_MALLOC)
           return (DREC_HDR_MALLOC);
         else if (rval == RHDR_HDR_REALLOC)
           return (DREC_HDR_REALLOC);

         /********************************************************************/
         /*  Timing is done AFTER sensor data is acquired so return the code */
         /*  to enable plotting of data if retrieved before time advancement.*/
         /********************************************************************/

         else if (rval == EOF_STATUS && !found_sensor)
          {
            ex->hrec_eof = 1;
            return (DREC_EOF_NO_SENSOR);
          }
         else if (rval == EOF_STATUS && found_sensor)
          {
            ex->hrec_eof = 2;
            return (DREC_EOF_SENSOR);
          }
         else
           return (rval);
       }

      new_day = *ptr->DAYOFYEAR;
      day_diff = new_day - old_day;
      if (day_diff < 0)
       day_diff += old_day;

   /**************************************************************************/
   /*  If the day changed, adjust accum_ss values.  Right now, this is done  */
   /*  no matter what, but I feel there must be some additional condition    */
   /*  placed on whether or not this is done, besides having the day change. */
   /**************************************************************************/

      if (day_diff != 0 && ptr->cur_sen_set != 0)
        ex->accum_ss_ms -= (day_diff * 86400000);
    }

  if (ex->BswaP)
     ReOrderUdf(2);

  ex->hrec_eof = 0;
  return (ALL_OKAY);
}
