#include <stdio.h>
#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_NEW_DATA_RECORD SUBROUTINE                    *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called by the read_drec routine whenever a new data     *
 *  record needs to be read.  If upon record retrieval, an EOF code is        *
 *  encountered or if a partial record is read, an internal flag is set so    *
 *  that the next time read_drec is called, another attempt is made to read   *
 *  from the data file.  This additional attempt is made only for real-time   *
 *  processing.  For playback, a partial read is a fatal error.               *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    char found_sensor          flag which tells if the requested sensor is  *
 *                               is in the sensors returned and is associated *
 *                               with the time period being processed         *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_new_data_record (found_sensor)                                   *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    lseek()                    moves the file pointer to a location within  *
 *                               the file                                     *
 *    read()                     reads N bytes from the file associated with  *
 *                               the file descriptor given                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    int num_bytes              the number of bytes requested from the file  *
 *    int ret_bytes              the number of bytes actually read by READ()  *
 *    long ret_val               holds value returned by the called routine   *
 *    long hdr_offset            indicates which header record to access      *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_new_data_record (ByTe_1 found_sensor)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  int num_bytes, ret_bytes;
  ByTe_4 ret_val, hdr_offset;

  /***********************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header */
  /*  and data information for the experiment currently being processed. */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;

  /*************************************************************************/
  /*  Read the next data record in the data file.                          */
  /*************************************************************************/

  num_bytes = ptr->d_size;
  ret_bytes = read (ex->fdd, ex->DATA_MEM, num_bytes);
  if (ex->BswaP)
      ReOrderUdf(1);
  if (ret_bytes <= 0 || ret_bytes != num_bytes)
   {
     /**********************************************************************/
     /*  An error was encountered when trying to read from the file.       */
     /**********************************************************************/

     if (ret_bytes < 0)
       return (DREC_READ_ERROR);

     /**********************************************************************/
     /*  An EOF was encountered and the sensor being returned is not the   */
     /*  sensor requested (sensor requested does not appear in the current */
     /*  time frame being processed).  If a non-real time file is being    */
     /*  processed, an error was encountered.                              */
     /**********************************************************************/

     else if (ret_bytes == 0 && !found_sensor)
      {
        ex->drec_eof = 1;
        if (ex->btime_sec >= 0)
          return (DREC_READ_ERROR);
        else
          return (DREC_EOF_NO_SENSOR);
      }

     /**********************************************************************/
     /*  An EOF was encountered for the sensor being requested.  If a      */
     /*  non-real time file is being processed, an error was encountered.  */
     /**********************************************************************/

     else if (ret_bytes == 0 && found_sensor)
      {
        ex->drec_eof = 2;
        if (ex->btime_sec >= 0)
          return (DREC_READ_ERROR);
        else
          return (DREC_EOF_SENSOR);
      }

     /********************************************************************/
     /*  Either we read before the listener could finish writing, or the */
     /*  listener never received the file closure packet and wrote down  */
     /*  20 bytes of -1 or -2 values.                                    */
     /********************************************************************/

     else if (ret_bytes != num_bytes)
      {
        /*******************************************************************/
        /*  We read before the listener could finish writing so back up    */
        /*  the partial no. of bytes read for future read attempts.        */
        /*******************************************************************/

        hdr_offset = *(ptr->HDR_OFF + 0);
        ex->drec_eof = (!found_sensor) ? 1 : 2;
        ret_val = lseek (ex->fdd, (ByTe_4) (-1 * ret_bytes), 1);

        /********************************************************************/
        /*  Check for LOS and/or next file signals.  The file closure flags */
        /*  are always in the first hdr_offset field.                       */
        /********************************************************************/

        if (hdr_offset == NO_MORE_DATA)
          return (LOS_STATUS);
        else if (hdr_offset == NEXT_FILE)
          return (NEXT_FILE_STATUS);
        else if (ex->btime_sec >= 0)
          return (PARTIAL_READ);
        else if (found_sensor)
          return (DREC_EOF_SENSOR);
        else
          return (DREC_EOF_NO_SENSOR);
      }
   }

  ex->drec_eof = 0;
  return (ALL_OKAY);
}
