#include <stdio.h>
#include <stdlib.h>
#include "gen_defs.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                         IR_LOCATE_EX SUBROUTINE                            *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine checks to see if the requested data set has already been   *
 *  initialized.  If this combination is the first combination to be          *
 *  processed and the calling routine specified that memory be allocated,     *
 *  space is allocated for the experiment_info definition structure and this  *
 *  memory is initialized to reflect the current requested combination.       *
 *  Otherwise, if the calling routine is only verifying that the combination  *
 *  exists within the already defined combination, an error code is returned. *
 *  If this combination is not the first combination to be processed, the     *
 *  combination was not found amongst the already defined combinations, and   *
 *  the calling routine specified that memory be allocated, the memory for    *
 *  the experiment_info structures are re-allocated in order to add an        *
 *  experiment_info structure for the new combination being processed.        *
 *  Otherwise, if the combination was not found, an error code is returned to *
 *  the calling routine.  If the combination is found amongst the already     *
 *  defined combinations, the pointers to the correct experiment_info         *
 *  structure are set.  The version number allows multiple opens to the same  *
 *  IDFS data set, in which case, a unique experiment_info structure will be  *
 *  allocated for each version; that is, it is treated as a new combo to be   *
 *  processed.                                                                *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long data_key       key which uniquely identifies the data set *
 *                                 being processed                            *
 *    char *exten                  the filename extension for the data file   *
 *                                 to be utilized                             *
 *    unsigned short vnum          version number to be associated with this  *
 *                                 combination (allows for multiple opens)    *
 *    char alloc_mem               flag which indicates if memory should be   *
 *                                 allocated if the requested combination is  *
 *                                 not found within the existing combinations *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_locate_ex (data_key, exten, vnum, alloc_mem)                     *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    malloc()                     allocates memory                           *
 *    realloc()                    reallocates previously allocated memory    *
 *    ir_init_experiment_info()    initializes a newly allocated experiment_  *
 *                                 info structure                             *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure that holds information concern-  *
 *                                 ing the experiment that is being processed *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    register short pos           indicates which definition (0, 1, etc.)    *
 *                                 matches the combination requested          *
 *    unsigned int bytes           the number of bytes to allocate            *
 *                                 the call to the MALLOC routine             *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

ByTe_2 ir_locate_ex (u_ByTe_4  data_key, ByTe_1  *exten, u_ByTe_2  vnum,
                     ByTe_1  alloc_mem)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register ByTe_2 pos;
   ByTe_2 free_pos = -1, init_pos;
   size_t bytes;

  /************************************************************************/
  /*  Is this combination the first combination to be processed?          */
  /*  ginfo.nds is the number of allocated experiment_info structures.    */
  /************************************************************************/

   if (ginfo.nds == 0) {

  /************************************************************************/
  /*  Allocate space for 1 experiment_info structure.                     */
  /************************************************************************/

      ginfo.nds = 1;
      bytes = sizeof(struct experiment_info);
      ginfo.sexper = ginfo.nds * bytes;
      if (ginfo.sexper <= 0) { return (LOCATE_EX_MALLOC); }
   
      if ((ginfo.mexper = malloc(ginfo.sexper)) == NO_MEMORY)
         return (LOCATE_EX_MALLOC);
      ginfo.exbase = (struct experiment_info *) ginfo.mexper;

  /************************************************************************/
  /*  Initialize the new experiment_info structure                        */
  /************************************************************************/

      ir_init_experiment_info (0);
   }

  /************************************************************************/
  /*  Determine if the requested combination has already been processed.  */
  /************************************************************************/

   ex = ginfo.exbase;
   for (pos = 0; pos < ginfo.nds; ++ex, ++pos) {

  /************************************************************************/
  /*  If the experiment_info structure for this combination has been      */
  /*  allocated, set the pointers to point to this structure and return.  */
  /************************************************************************/

      if (ex->data_key == data_key && ex->version == vnum && 
          *(exten) == ex->exten[0] && 
          (ex->exten[0] == '\0' || *(exten + 1) == ex->exten[1]))
      {
          ginfo.pos_ex = pos;
          ginfo.expt   = ex;
          return (ALL_OKAY);
      }
      if (ex->data_key == 0)  { free_pos = pos; }
   }

  /************************************************************************/
  /*  If alloc_mem is set to 1 then we are not just searching for an      */
  /*  existing experiment_info definition but will create a new one if    */
  /*  the one we are looking for doesn't exist.                           */
  /************************************************************************/

   if (alloc_mem) {

  /************************************************************************/
  /*  If there wasn't a free slot in the experiment_info array then we    */
  /*  we need to re-alloc memory in order to define a new experiment_info */
  /************************************************************************/

      if ( free_pos < 0 ) {
         bytes = sizeof(struct experiment_info);
         ginfo.sexper = (ginfo.nds + 1) * bytes;
         if (ginfo.sexper <= 0) { return (LOCATE_EX_REALLOC); }
   
         if ((ginfo.mexper = realloc(ginfo.mexper, ginfo.sexper)) == NO_MEMORY)
           return (LOCATE_EX_REALLOC);

         ginfo.exbase = (struct experiment_info *) ginfo.mexper;
 	 init_pos = ginfo.nds++;
      } else { init_pos = free_pos; }
 
  /************************************************************************/
  /*  Initialize the new experiment_info structure                        */
  /************************************************************************/
 
      ir_init_experiment_info (init_pos);
      ex = ginfo.exbase + init_pos;
      ex->data_key = data_key;
      ex->exten[0] = *exten;
      ex->exten[1] = *(exten + 1);
      ex->exten[2] = '\0';
      ex->version = vnum;
      ginfo.pos_ex = init_pos;
      ginfo.expt = ex;
 
  /************************************************************************/
  /*  Memory needs to be allocated for the ptr_rec structure, which is    */
  /*  included in each experiment_info structure.                         */
  /************************************************************************/
 
      bytes = sizeof(struct ptr_rec);
      if (bytes <= 0) { return (LOCATE_PTR_MALLOC); }

      if ((ex->bmem.base_info_ptr = malloc (bytes)) == NO_MEMORY)
        return (LOCATE_PTR_MALLOC);
      ex->info_ptr = (struct ptr_rec *) ex->bmem.base_info_ptr;
      ex->info_ptr->prev_hd = -1;
      ex->info_ptr->cur_sen_set = -1;
      ex->info_ptr->h_size = -1;
      ex->info_ptr->chg_sen_set = 0;
      return (ALL_OKAY);
   } else { return (LOCATE_NOT_FOUND); }
}
