#include <math.h>
#include "libbase_udf.h"
#include "util_str.h"

/*******************************************************************************
 *                                                                             *
 *                           IR_IDF_POWER SUBROUTINE                           *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is used in conjunction with the definition of the tables    *
 *  in the VIDF file.  This module will be used when the TBL_OPER value        *
 *  specifies the raise to power operation.                                    *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    float *DataOut             pointer to the array that holds the values    *
 *                               returned by the module                        *
 *    float *tbl                 pointer to the table values being utilized    *
 *    long  DaTaOff              index value indicating where to start in the  *
 *                               input data array                              *
 *    long  TblSz                the number of elements in the table           *
 *    long  Ne                   the number of elements being returned         *
 *    long  Hold                 the number of successive data values to be    *
 *                               modified by a single input data value (used   *
 *                               for calibration data; otherwise, set to 1)    *
 *    char  LookUp               flag that indicates if the table has been     *
 *                               expanded and serves as a lookup table or if   *
 *                               the value or if the values need to be         *
 *                               calculated using the poly. coefficients       *
 *    char  *DaTaIn              pointer to the array that holds the input     *
 *                               data values                                   *
 *    char  dtype                the format of the data - uns int, int, float  *
 *    char  FnOfRaw              flag indicating if the table is a function of *
 *                               raw or processed data                         *
 *    char  TblType              the TBL_TYPE value for the table of interest  *
 *    void *idf_data_ptr         ptr to memory location for structure that     *
 *                               holds returned data values (read_drec)        *
 *    short SwpLen               the maximum number of sample sequence values  *
 *                                                                             *
 *  USAGE                                                                      *
 *    ir_idf_multiply (&DataOut, &DaTaIn, DaTaOff, &tbl, TblSz, Ne, lookup,    *
 *                     dtype, Hold, FnOfRaw, TblType, SwpData, SwpLen) *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_poly_expand()           calculates values using a set of polynomial   *
 *                               coefficients                                  *
 *    TimeTblVal()               computes input for time based table           *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct idf_data *IDFS      structure holding all of the currently        *
 *                               returned data values to be processed          *
 *    register float *f1, *f2    pointers to float values                      *
 *    register long *l1          pointer to rad IDFS data values               *
 *    register long I, J, K      looping variables                             *
 *    register long StoP         the stopping value for the loop               *
 *    float *f1                  offset ot beginning of table                  *
 *    float *f3                  offset ot beginning of table                  *
 *    float V                    time input for type 3 and 4 tables            *
 *    float X                    power to raise value to                       *
 *    long nC                    number of polynomial coefficients             *
 *    long adv                   distance to advance in table if TblType is 2  *
 *    char dF;                   input data format                             *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

void ir_idf_power (ReaL_8 *DataOut, ByTe_1 *DaTaIn, ByTe_4 DaTaOff, 
                   ReaL_4 *tbl, ByTe_4 TblSz, ByTe_4 Ne, ByTe_1 LookUp, 
                   ByTe_1 dtype, ByTe_4 Hold, ByTe_1 FnOfRaw, 
                   ByTe_1 TblType, void *idf_data_ptr, ByTe_2 SwpLen)
{
   struct idf_data *IDFS;
   register ReaL_8 *d1;
   register ByTe_4 *l1, *l2, I, J, K, StoP;
   ReaL_8 X;
   ReaL_8 V;
   ReaL_4 *f1, *f3;
   ByTe_4 adv, nC, Inc;
   ByTe_1 dF;

   IDFS = (struct idf_data *)idf_data_ptr;             /* IDFS read data     */
  
   adv = (LookUp) ? SwpLen : TblSz;                    /* scan advance       */
   nC = (LookUp) ? 0 : TblSz;                          /* # of poly coefs    */
   dF  = (FnOfRaw) ? dtype : -1;                       /* input data format  */
   Inc  = (FnOfRaw) ? 1 : 2;                           /* data inc value     */
   d1 = DataOut;                                       /* return data here   */
      
   switch (TblType)                                    /* switch on tbl_type */
   {                                                   /* BEGIN TTYPE SWITCH */
      case 0:                                          /* START LOOKUP TBLS  */
      case 5:
         l1 = (ByTe_4 *) DaTaIn + DaTaOff;             /* input data         */
         for (I = 0, J = Ne; I < Ne; J -= Hold, l1 += Inc)  /* element loop  */
         {                                             /* BEGIN ELEMENT LOOP */
            StoP = (J < Hold) ? J : Hold;              /* end for held data  */
            for (K = 0; K < StoP; ++K, ++d1, ++I)      /* loop over data     */
            {                                          /* BEGIN DATA LOOP    */
               X = ir_poly_expand (l1, tbl, nC, dF);   /* power              */
               V = RawToFloat(l1, dF) + .05;           /* input data valued  */
               *d1 = pow ((ReaL_8)V, X);               /* V ** X             */
            }                                          /* BEGIN DATA LOOP    */
         }                                             /* END ELEMENT LOOP   */
      break;                                           /* END LOOKUP TBLS    */
      case 2:                                          /* START LOOKUP TBLS  */
         l1 = (ByTe_4 *) DaTaIn + DaTaOff;             /* input data         */
         l2 = IDFS->swp_data;                          /* scan data is here  */
         for (I = 0, J = Ne; I < Ne; J -= Hold, l1 += Inc)  /* element loop  */
         {                                             /* BEGIN ELEMENT LOOP */
            StoP = (J < Hold) ? J : Hold;              /* end for held data  */
            for (K = 0; K < StoP; ++K, ++d1, ++I)      /* loop over data     */
            {                                          /* BEGIN HOLD LOOP    */
               f3 = tbl + *l2++ * adv;                 /* offset into table  */
               X = ir_poly_expand (l1, f3, nC, dF);    /* power              */
               V = RawToFloat(l1, dF) + .05;           /* input data valued  */
               *d1 = pow ((ReaL_8)V, X);               /* V ** X             */
            }                                          /* END HOLD LOOP      */
         }                                             /* END ELEMENT LOOP   */
      break;                                           /* END LOOKUP TBLS    */
      case 3:                                          /* START TIME TBLS    */
         V = TimeTblVal(tbl, idf_data_ptr);            /* input data         */
         f1 = tbl + 5;                                 /* fwd past base inps */
         V = ir_poly_expand ((ByTe_4*)&V, f1, nC, -1); /* time variable      */
         for (I = 0, J = Ne; I < Ne; J -= Hold)        /* loop over elements */
         {                                             /* BEGIN ELEMENT LOOP */
            StoP = (J < Hold) ? J : Hold;              /* end for held data  */
            for (K = 0; K < StoP; ++K, ++d1, ++I)      /* loop over data     */
               *d1 = pow ((ReaL_8)*d1, (ReaL_8)V);     /* *d1 ** V           */
         }                                             /* END ELEMENT LOOP   */
      break;                                           /* STOP TIME/SWP TBLS */
      case 4:                                          /* START TIME TBLS    */
         V = TimeTblVal(tbl, idf_data_ptr);            /* input data         */
         f1 = tbl + 5;                                 /* fwd past base inps */
         l2 = IDFS->swp_data;                          /* scan data is here  */
         for (I = 0, J = Ne; I < Ne; J -= Hold)        /* loop over elements */
         {                                             /* BEGIN ELEMENT LOOP */
            StoP = (J < Hold) ? J : Hold;              /* end for held data  */
            for (K = 0; K < StoP; ++K, ++d1, ++I)      /* loop over data     */
            {                                          /* BEGIN HOLD LOOP    */
               f3 = f1 + *l2++ * adv;                  /* offset into table  */
               X = ir_poly_expand ((ByTe_4*)&V, f3, nC, -1);  /* power       */
               *d1 = pow ((ReaL_8)*d1, X);             /* *d1 ** X           */
            }                                          /* END HOLD LOOP      */
         }                                             /* END ELEMENT LOOP   */
      break;                                           /* STOP TIME/SWP TBLS */
   }                                                   /* END TTYPE SWITCH   */
}
