#include "util_str.h"
#include "idf_defs.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                          IR_GET_PITCH_INFO SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to retrieve information from the VIDF file        *
 *  concerning pitch angle calculations.  It is only called if the IDFS source *
 *  has the pa_defined VIDF flag set to one.                                   *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    short btime_yr               the start time requested (year component)   *
 *    short btime_day              the start time requested (day component)    *
 *    long btime_sec               the start time requested (seconds component)*
 *    long btime_nsec              the start time requested (nanoseconds)      *
 *    short etime_yr               the stop time requested (year component)    *
 *    short etime_day              the stop time requested (day component)     *
 *    long etime_sec               the stop time requested (seconds component) *
 *    long etime_nsec              the stop time requested (nanoseconds)       *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_get_pitch_info (btime_yr, btime_day, btime_sec, btime_nsec,       *
 *                           etime_yr, etime_day, etime_sec, etime_nsec)       *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    get_version_number ()        returns the next available version number   *
 *    read_idf()                   reads information from the VIDF file        *
 *    select_sensor()              specifies which sensors are to be processed *
 *    create_idf_data_structure()  creates the idf_data structure that will    *
 *                                 hold data from the pitch angle IDFS source  *
 *    ir_file_setup()              opens the data, header and vidf file and    *
 *                                 retrieves information from the VIDF file    *
 *    ir_get_pinfo_rval ()         returns an error code unique to this module *
 *                                 for each possible READ_IDF() error code     *
 *    ir_alloc_pitch_info ()       allocates the structure that holds the      *
 *                                 pitch angle information                     *
 *    ir_get_unit_normals ()       retrieves the unit normal to the aperture   *
 *                                 and computes the magnitude of the vector    *
 *    ir_pitch_angle_data_src ()   determines IDFS data source for pitch angles*
 *    ir_pitch_angle_tbl_info ()   retrieves unit information for pitch angles *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *    char inside_reset_exp        flag indicating in the RESET_EXPERIMENT_INFO*
 *                                 module (needed for FILE_OPEN logic)         *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    register struct pitch_info   a pointer to the structure that holds pitch *
 *         *pa_ptr                 angle information                           *
 *    register short i             looping variable                            *
 *    long ret_val                 holds value returned by called module       *
 *    short rval                   holds value returned by called module       *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_get_pitch_info (ByTe_2 btime_yr, ByTe_2 btime_day, ByTe_4 btime_sec, 
                          ByTe_4 btime_nsec, ByTe_2 etime_yr, ByTe_2 etime_day,
                          ByTe_4 etime_sec, ByTe_4 etime_nsec)
{
   extern struct general_info ginfo;
   extern ByTe_1 inside_reset_exp;

   struct experiment_info *ex;
   register struct pitch_info *pa_ptr;
   register ByTe_2 i;
   ByTe_4 ret_val;
   ByTe_2 rval;

   /**************************************************************************/
   /*  Allocate the structure that holds the pitch angle information.        */
   /**************************************************************************/

   ex = ginfo.expt;
   if (ex->bmem.base_pitch_info == NO_MEMORY)
    {
      rval = ir_alloc_pitch_info (etime_yr, etime_day, etime_sec, etime_nsec);
      if (rval != ALL_OKAY)
        return (rval);
    }
   pa_ptr = ex->pitch_angles;

   /**************************************************************************/
   /*  Retrieve pertinent information from the VIDF file for pitch angle     */
   /*  processing.                                                           */
   /**************************************************************************/

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) &pa_ptr->pa_format, _PaFormaT, 0, 0L, 1);
   if (ret_val < 0)
    {
      pa_ptr->file_status = ir_get_pinfo_rval (ret_val);
      return (ALL_OKAY);
    }

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) pa_ptr->sensor, _PaBxbybZ, 0, 0L, -1);
   if (ret_val < 0)
    {
       pa_ptr->file_status = ir_get_pinfo_rval (ret_val);
       return (ALL_OKAY);
    }

   rval = ir_pitch_angle_data_src ();
   if (rval != ALL_OKAY)
    {
      pa_ptr->file_status = rval;
      return (ALL_OKAY);
    }

   rval = ir_pitch_angle_tbl_info ();
   if (rval != ALL_OKAY)
    {
      pa_ptr->file_status = rval;
      return (ALL_OKAY);
    }

   /***********************************************************************/
   /*  Retrieve the unit normal to the aperture.                          */
   /***********************************************************************/

   ret_val = ir_get_unit_normals ();
   if (ret_val < 0)
    {
       pa_ptr->file_status = ir_get_pinfo_rval (ret_val);
       return (ALL_OKAY);
    }

   /************************************************************************/
   /*  Try to open the necessary files using a new version number in order */
   /*  to get separate file descriptor.  If reset_experiment_info invoked  */
   /*  this routine, the ex structure for the pitch angle data has been    */
   /*  allocated so point back to this structure.                          */
   /************************************************************************/

   if (!inside_reset_exp)
    {
      get_version_number (&pa_ptr->version);
      ex->pa_version = pa_ptr->version;
    }
   else
     pa_ptr->version = ex->pa_version;

   pa_ptr->file_status = ir_file_setup (pa_ptr->data_key, pa_ptr->exten,
                                     pa_ptr->version, btime_yr, btime_day,
                                     btime_sec, btime_nsec, etime_yr, etime_day,
                                     etime_sec, etime_nsec);
   if (pa_ptr->file_status != ALL_OKAY)
     return (ALL_OKAY);

   /***********************************************************************/
   /*  Use only those sensors needed to compute pitch angles.             */
   /***********************************************************************/

   for (i = 0; i < 3; ++i)
    {
      rval = select_sensor (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version, 
                            pa_ptr->sensor[i]);
      if (rval != ALL_OKAY)
       {
         pa_ptr->file_status = rval;
         return (ALL_OKAY);
       }

      rval = create_idf_data_structure (&pa_ptr->idf_data_ptr[i]);
      if (rval != ALL_OKAY)
       {
         pa_ptr->file_status = rval;
         return (ALL_OKAY);
       }
    }

   /***********************************************************************/
   /*  Make sure pitch angle IDFS source is a scalar instrument.          */
   /***********************************************************************/

   ex = ginfo.expt;
   if (ex->smp_id != 2)
    {
      pa_ptr->file_status = PA_BAD_SRC;
      return (ALL_OKAY);
    }

   if (inside_reset_exp)
     ex->fnext = 0;

   return (ALL_OKAY);
}

/*******************************************************************************
 *                                                                             *
 *                          IR_GET_PINFO_RVAL SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine returns an error code unique to the module GET_PITCH_INFO() *
 *  based upon the error code returned by the call to READ_IDF().              *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    long ret_val                the error code returned by READ_IDF()        *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_get_pinfo_rval (ret_val)                                          *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_get_pinfo_rval (ByTe_4 ret_val)
{
   switch (ret_val)
    {
       case IDF_NOT_FOUND:
         return (PINFO_IDF_NOT_FOUND);
       case IDF_MANY_BYTES:
         return (PINFO_IDF_MANY_BYTES);
       case IDF_TBL_NUM:
         return (PINFO_IDF_TBL_NUM);
       case IDF_CON_NUM:
         return (PINFO_IDF_CON_NUM);
       case IDF_NO_ENTRY:
         return (PINFO_IDF_NO_ENTRY);
       default:
         return ((ByTe_2) ret_val);
    }
}
