#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                        IR_GET_NEW_TBL  SUBROUTINE                         *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine will retrieve the table values for the specified table.   *
 *  The values will either be read in (one-to-one mapping) from the VIDF     *
 *  file, constructed from the polynomial coefficients, or the table will    *
 *  be left as coefficients, but the values will be scaled and saved.        *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    short sen                the sensor being processed                    *
 *    short which_tbl          the number of the table being processed       *
 *    char expand              flag indicating if the table needs expanding  *
 *    char tFmt                table format                                  *
 *    float *tbl               pointer to the table                          *
 *    long tbl_size            the maximum number of values associated with  *
 *                             the given sensor                              *
 *    long tbl_off             variable which holds the offset into table    *
 *                             (-1 if need to read this info. from VIDF file)* 
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_get_new_tbl (sen, which_tbl, expand, tFmt, tbl, tbl_size,       *
 *                     tbl_off)                                              *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                the size of the specified object in bytes     *
 *    read_idf()               reads information from the IDF file           *
 *    ir_tento()               returns the power of 10 value to multiply by  *
 *                             to get values to the correct base (scaling)   *
 *    ir_get_new_rval ()       returns an error code unique to this module   *
 *                             for each possible READ_IDF() error code       *
 *    ir_read_in_tbl ()        retrieves the table values from the VIDF file *
 *                             (no need for construction)                    *
 *    ir_create_tbl ()         constructs the table values using information *
 *                             from the VIDF file                            *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info      structure that holds information concerning   *
 *        ginfo                the experiment that is being processed        *
 *    unsigned int Tmp_Bytes   number of bytes allocated for scratch space   *
 *    void *Tmp_Space          scratch space for various uses                *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct experiment_info   a pointer to the structure that holds         *
 *          *ex                specific experiment information               *
 *    register float *f1       pointer to data values (float values)         *
 *    register float power     the scaling to be applied to the table values *
 *    register long *l1        pointer to the values read                    *
 *    register char *c1        pointer to the scaling values                 *
 *    unsigned int bytes       the number of bytes to allocate space for     *
 *    long rV                  holds the value returned by called routine    *
 *    long sca_size            the number of scaling factors                 *
 *    short nV                 number of table values to process             *
 *    char scale               scaling factor to be applied to all values    *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *****************************************************************************/

ByTe_2 ir_get_new_tbl (ByTe_2 sen, ByTe_2 which_tbl, ByTe_1 expand, 
                       ByTe_1 tFmt, ReaL_4 *tbl, ByTe_4 tbl_size, 
                       ByTe_4 tbl_off)
{
   extern struct general_info ginfo;
   extern size_t Tmp_Bytes;
   extern void *Tmp_Space;

   struct experiment_info *ex;
   register ReaL_8  power;
   register ReaL_4 *f1, *fEnD;
   register ByTe_4 *l1;
   register ByTe_1 *c1;
   size_t bytes;
   ByTe_4 rV, sca_size, OffSet, NumSca;
   ByTe_2 nV;
   ByTe_1 scale;

   ex = ginfo.expt;
   
  /**********************************************************************/
  /*  The table just needs to be read in (one-to-one mapping).          */
  /**********************************************************************/

   if (tFmt == 0) {
      rV = ir_read_in_tbl (sen, tbl, tbl_off, tbl_size, which_tbl);
      if (rV != ALL_OKAY) { return ((ByTe_2)rV); }
   } else if (expand && tFmt > 0) {

  /***********************************************************************/
  /*  The table needs to be constructed from the polynomial coefficients.*/
  /***********************************************************************/

      rV = ir_create_tbl (sen, tbl, tbl_size, tFmt, which_tbl, 1, tbl_off);
      if (rV != ALL_OKAY) { return ((ByTe_2)rV); }
   } else if (!expand && tFmt > 0) {

  /**************************************************************************/
  /*  The polynomial coefficients should be scaled and saved (no expansion).*/
  /**************************************************************************/

  /***********************************************************************/
  /*  The TBL_EXPAND flag is hard-coded to 0 for tbl_type 2 & 4 tables.  */
  /*  CHANGE FOR TBL_TYPE 4 by CAG 4/5/98                                */
  /***********************************************************************/

      switch (*(ex->tbl_type + which_tbl) ) {
          case 0:
          case 5:
             OffSet = 0;
             nV = tFmt;
             NumSca = nV;
          break;
          case 2:
             OffSet = 0;
             nV = tFmt * ex->swp_len;
             NumSca = nV;
          break;
          case 3:
             OffSet = 1;
             nV = tFmt + 5;
             NumSca = tFmt;
          break;
          case 4:
             OffSet = 1;
             nV = tFmt * ex->swp_len + 5;
             NumSca = tFmt * ex->swp_len;
          break;
      }

  /***********************************************************************/
  /*  A table offset of -1 indicates that the correct value needs to be  */
  /*  read from the VIDF file.                                           */
  /***********************************************************************/

      if (tbl_off == -1) {
         rV = read_idf (ex->data_key, ex->exten, ex->version,
                   (ByTe_1 *) &tbl_off, _TblOfF, which_tbl, (ByTe_4) sen, 1);
         if (rV < 0) { return (ir_get_new_rval (rV)); }
      }

      rV = read_idf (ex->data_key, ex->exten, ex->version,
                        (ByTe_1 *) &sca_size, _TblScaSZ, which_tbl, 0L, 1);
      if (rV < 0) { return (ir_get_new_rval (rV)); }
   
      rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) tbl,
                          _TbL, which_tbl, tbl_off, nV);
      if (rV < 0) { return (ir_get_new_rval (rV)); }

  /**********************************************************************/
  /*  There is one scaling value per value.                             */
  /**********************************************************************/
   
      if (sca_size > 0) {

  /******************************************************************/
  /*  Allocate space to hold the scaling values.                    */
  /*  CAG: 12/16/08 - removed realloc in favor of a free/malloc     */
  /******************************************************************/
   
         bytes = nV * sizeof (ByTe_1);
         if (bytes > Tmp_Bytes) {
            if (bytes <= 0) { return (NEW_SCALE_MALLOC); }
	    free(Tmp_Space);

            if ((Tmp_Space = malloc (bytes)) == NO_MEMORY)
               return (NEW_SCALE_MALLOC);
            Tmp_Bytes = bytes;
         }

         rV = read_idf (ex->data_key, ex->exten, ex->version,
                           (ByTe_1 *) Tmp_Space, _TblScA, which_tbl, tbl_off,
                            NumSca);
         if (rV < 0) { return (ir_get_new_rval (rV)); }
   
  /******************************************************************/
  /*  Apply the scaling factors to the table values.                */
  /******************************************************************/
   
         l1 = (ByTe_4 *) tbl;
         f1 = tbl;
         if (OffSet == 1 ) {
            *f1++ = *l1++;
            *f1++ = *l1++;
            *f1++ = *l1++;
            *f1++ = *l1++;
            *f1++ = *l1++;
            fEnD = f1 + NumSca;
            for (c1 = (ByTe_1 *) Tmp_Space; f1 < fEnD;)
               *f1++ = *l1++ * ir_tento (*c1++);
         } else {
            fEnD = f1 + nV;
            for (c1 = (ByTe_1 *) Tmp_Space; f1 < fEnD;)
               *f1++ = *l1++ * ir_tento (*c1++);
         }
      } else if (sca_size < 0) {
   
  /**********************************************************************/
  /*  There is a single scaling value for each sensor.                  */
  /**********************************************************************/
   
        rV = read_idf (ex->data_key, ex->exten, ex->version,
                      (ByTe_1 *) &scale, _TblScA, which_tbl, (ByTe_4) sen, 1);
        if (rV < 0) { return (ir_get_new_rval (rV)); }
  
  /******************************************************************/
  /*  Apply the scaling factor to all the table values.             */
  /******************************************************************/

        l1 = (ByTe_4 *) tbl;
        f1 = tbl;
        if (OffSet == 1 ) {
           *f1++ = *l1++;
           *f1++ = *l1++;
           *f1++ = *l1++;
           *f1++ = *l1++;
           *f1++ = *l1++;
           fEnD = f1 + NumSca;
         } else { fEnD = f1 + nV; }

         power = ir_tento (scale);
         for (f1 = tbl; f1 < fEnD;) { *f1++ = *l1++ * power; }
      }
   }

   return (ALL_OKAY);
}

/*****************************************************************************
 *                                                                           *
 *                             IR_GET_NEW_RVAL SUBROUTINE                    *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine returns an error code unique to the module GET_NEW_TBL()  *
 *  based upon the error code returned by the call to READ_IDF().            *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    long rV                the error code returned by READ_IDF()           *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_get_new_rval (rV)                                               *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    None                                                                   *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    None                                                                   *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    None                                                                   *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 ****************************************************************************/

ByTe_2 ir_get_new_rval (ByTe_4 rV)
{
   switch (rV) {
      case IDF_NOT_FOUND:  return (NEW_IDF_NOT_FOUND);
      case IDF_MANY_BYTES: return (NEW_IDF_MANY_BYTES);
      case IDF_TBL_NUM:    return (NEW_IDF_TBL_NUM);
      case IDF_CON_NUM:    return (NEW_IDF_CON_NUM);
      case IDF_NO_ENTRY:   return (NEW_IDF_NO_ENTRY);
      default:             return ((ByTe_2) rV);
   }
}
