#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <memory.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                            IR_GET_MODE_INFO SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to allocate space for the structure(s) that hold  *
 *  the information for the mode tables.  One structure is allocated per mode, *
 *  but if no tables are defined, no tables get loaded.                        *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    None                                                                     *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_get_mode_info ()                                                  *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    sizeof ()                the size of the specified object in bytes       *
 *    malloc()                 allocates memory                                *
 *    free()                   frees allocated memory                          *
 *    ir_get_mode_tbl ()       retrieves the various table(s) information for  *
 *                             the mode being processed                        *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info      structure that holds information concerning     *
 *        ginfo                the experiment that is being processed          *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info   a pointer to the structure that holds specific  *
 *          *ex                experiment information                          *
 *    unsigned int bytes       the number of bytes to allocate                 *
 *    short i                  looping variable                                *
 *    short rval               holds the value returned by the called routine  *
 *    void *tmp_ptr            pointer which holds address passed back by      *
 *                             the call to the MALLOC routine                  *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_get_mode_info (void)
{
  extern struct general_info ginfo;

   struct experiment_info *ex;
   size_t bytes;
   ByTe_2 i, rval;
   void *tmp_ptr;

  /*************************************************************************/
  /*  Malloc the space to hold the mode tables, one structure per mode,    */
  /*  and initialize the structure(s).                                     */
  /*************************************************************************/

  ex = ginfo.expt;
  if (ex->bmem.base_mode_ptr != NO_MEMORY) {
     free (ex->bmem.base_mode_ptr);
     ex->bmem.base_mode_ptr = NO_MEMORY;
   }

  /*************************************************************************/
  /*  If no modes are defined, simply return.                              */
  /*************************************************************************/

  if (ex->num_modes == 0) { return (ALL_OKAY); }

  bytes = sizeof (struct mode_tables) * ex->num_modes;
  if (bytes <= 0)
    return (MODE_PTR_MALLOC);

  if ((tmp_ptr = malloc (bytes)) == NO_MEMORY)
    return (MODE_PTR_MALLOC);
  ex->bmem.base_mode_ptr = tmp_ptr;
  ex->mode_tbl_ptr = (struct mode_tables *) ex->bmem.base_mode_ptr;

  /************************************************************************/
  /*  Loop over each mode defined.                                        */
  /************************************************************************/

  for (i = 0; i < ex->num_modes; ++i)
   {
     /************************************************************************/
     /*  Get the various table(s) information for this sensor.               */
     /************************************************************************/

     rval = ir_get_mode_tbl (i);
     if (rval != ALL_OKAY)
       return (rval);
   }

  return (ALL_OKAY);
}
