#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_GET_CONSTANTS  SUBROUTINE                     *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This module is called in order to allocate the space to hold the angle  * 
 *  offset values that are used to compute the returned azimuthal angles and  *
 *  to retrieve all angle constants for the virtual instrument in question.   *
 *  The angle offset values are initialized to zero.  If one of the defined   *
 *  constants is an azimuthal angle offset constant, the values are replaced  *
 *  by the values read from the VIDF file.                                    *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_get_constants ()                                                 *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                the size of the specified object in bytes      *
 *    free ()                  frees allocated memory                         *
 *    malloc()                 allocates memory                               *
 *    read_idf()               reads information from the IDF file            *
 *    ir_tento()               returns the power of 10 value to multiply by   *
 *                             to get values to the correct base (scaling)    *
 *    ir_get_const_rval ()     returns an error code unique to this module    *
 *                             for each possible READ_IDF() error code        *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *    unsigned int Tmp_Bytes   number of bytes allocated for scratch space    *
 *    void *Tmp_Space          scratch space for various uses                 *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds          *
 *          *ex                specific experiment information                *
 *    register float *f1       pointer to floating point values               *
 *    reg float *fEnD          loop termination variable                      *
 *    register long *l1        pointer to long values                         *
 *    register short i         looping variables                              *
 *    unsigned int bytes       the number of bytes to allocate space for      *
 *    long rV                  holds the value returned by called routine     *
 *    char cId                 the constant identification category           *
 *    char num_constants       the number of constants to allocate space for  *
 *    char *const_sca          memory that holds the scaling values           *
 *    char *cptr               pointer to character values                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_get_constants (void)
{
   extern struct general_info ginfo;
   extern size_t Tmp_Bytes;
   extern void   *Tmp_Space;

   struct experiment_info *ex;
   register ReaL_4 *f1, *fEnD;
   register ByTe_4 *l1;
   register ByTe_2 I;
   size_t   bytes;
   ByTe_4 rV, Loc, Loc1;
   ByTe_2 Id[2], iD;
   ByTe_1 cId, num_constants, EdGe = 2;
   ByTe_1 *c1, *cEnD;

  /**************************************************************************/
  /*  Allocate the space to hold the angle constant values for each sensor  */
  /*  and initialize.                                                       */
  /**************************************************************************/

   ex = ginfo.expt;
   num_constants = 7;
   bytes = (num_constants * ex->num_sensor * sizeof (ReaL_4));
   if (bytes <= 0) { return (CONST_ANG_MALLOC); }

   if (ex->bmem.base_constant != NO_MEMORY) {
      free (ex->bmem.base_constant);
      ex->bmem.base_constant = NO_MEMORY;
   }
   if ((ex->bmem.base_constant = malloc (bytes)) == NO_MEMORY)
      return (CONST_ANG_MALLOC);
   ex->constants = (ReaL_4 *) ex->bmem.base_constant;

   f1 = ex->constants;
   fEnD = ex->constants + ex->num_sensor;
   for (; f1 < fEnD;) { *f1++ = 0.0; }

   f1 = ex->constants + ex->num_sensor;
   fEnD = ex->constants + num_constants * ex->num_sensor;
   for (; f1 < fEnD;) { *f1++ = -500.0; }

  /************************************************************************/
  /*  Allocate temporary space to hold the constant ids (1 per constant). */
  /*  CAG: 12/16/08 - removed realloc in favor of a free/malloc           */
  /************************************************************************/

   ex = ginfo.expt;
   if (ex->num_consts != 0) {
      bytes = ex->num_sensor * sizeof (ByTe_1);
      if (bytes > Tmp_Bytes) {
         if (bytes <= 0) { return (CONST_TEMP_MALLOC); }
	 free(Tmp_Space);
         if ((Tmp_Space = malloc(bytes)) == NO_MEMORY)
            return (CONST_TEMP_MALLOC);
         Tmp_Bytes = bytes;
      }
  
      Id[0] = -1;
      Id[1] = -1;
      for (I = 0; I < ex->num_consts; ++I) {
         rV = read_idf (ex->data_key, ex->exten, ex->version, &cId,
                                                     _ConstID, I, 0L, 1); 
         if (rV < 0) { return (ir_get_const_rval (rV)); }

         Loc = -1;
         switch (cId) {
            case 1:
            case 4:
            case 5:
            case 9:
               if (EdGe < 4) {
                  Id[EdGe - 2] = cId; 
                  Loc = EdGe++;
               } 
            break;
            case 2:
            case 3:
              Loc = cId - 2;
            break;
            case 29:
            case 30:
            case 31:
              Loc = cId - 25;
            break;
         }

         if (Loc >= 0) {
            c1 = (ByTe_1 *)Tmp_Space;
            f1 = ex->constants + Loc * ex->num_sensor;
            l1 = (ByTe_4 *)(ex->constants + Loc * ex->num_sensor);

            rV=read_idf (ex->data_key, ex->exten, ex->version,
                             (ByTe_1 *) l1, _ConsT, I, 0L, ex->num_sensor); 
            if (rV < 0) { return (ir_get_const_rval (rV)); }

            rV = read_idf (ex->data_key, ex->exten, ex->version, c1, 
                                         _ConstScA, I, 0L, ex->num_sensor);
            if (rV < 0) { return (ir_get_const_rval (rV)); }

            cEnD = c1 + ex->num_sensor;
            for ( ; c1 < cEnD;) { *f1++ = *l1++ * ir_tento (*c1++); }
         }
      }
   }

   if (EdGe > 3) {
      switch (Id[0]) {
         case 1:
            Loc1 = 2;
            Loc = 3;
            iD = (Id[1] == 4) ? 0 : (Id[1] == 5) ? 1 : 2; 
         break;
         case 4:
            Loc = 2;
            Loc1 = 3;
            iD = (Id[1] == 1) ? 0 : (Id[1] == 5) ? 3 : 4; 
         break;
         case 5:
            if (Id[1] == 4) {
               Loc = 3;
               Loc1 = 2;
               iD = 3;
            } else {
               Loc = 2;
               Loc1 = 3;
               iD = (Id[1] == 1) ? 1 : 5; 
            }
         break;
         case 9:
            if (Id[1] == 1) {
               Loc = 2;
               Loc1 = 3;
               iD = 3;
            } else {
               Loc = 3;
               Loc1 = 2;
               iD = (Id[1] == 4) ? 4 : 5; 
            }
         break;
      }

      BuildBins (Loc, Loc1, 2, 3, iD);
   } else {
      if (EdGe == 3) { BuildBins (2, -1, 2, 3, 6); }
   }

   return (ALL_OKAY);
}

/******************************************************************************
 *                                                                            *
 *                             IR_GET_CONST_RVAL SUBROUTINE                   *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine returns an error code unique to the module GET_CONSTANTS() *
 *  based upon the error code returned by the call to READ_IDF().             *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long rV                     the error code returned by READ_IDF()       *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_get_const_rval (rV)                                              *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_get_const_rval (ByTe_4 rV)
{
   switch (rV) {
       case IDF_NOT_FOUND:  return (CONST_IDF_NOT_FOUND);
       case IDF_MANY_BYTES: return (CONST_IDF_MANY_BYTES);
       case IDF_TBL_NUM:    return (CONST_IDF_TBL_NUM);
       case IDF_CON_NUM:    return (CONST_IDF_CON_NUM);
       case IDF_NO_ENTRY:   return (CONST_IDF_NO_ENTRY);
       default:             return ((ByTe_2) rV);
    }
}
