#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"
#ifndef TRUE
#define TRUE 1
#endif

/******************************************************************************
 *                                                                            *
 *                           IR_FINE_SEARCH SUBROUTINE                        *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This module walks thru the data file to the correct starting position   *
 *  corresponding to the playback start time selected by the user once the    *
 *  coarse search has been performed.  A fine search is performed by calling  *
 *  READ_DREC() to return the next time sample.  One sample is picked up with *
 *  each call for scalar instruments.  This call to READ_DREC() is continued  *
 *  until the sample closest to the requested start time has been found.      *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long data_key   key which uniquely identifies the data set     *
 *                             being processed                                *
 *    char *exten              the filename extension for the data to be used *
 *    unsigned short vnum      version number to be associated with this      *
 *                             combination (allows for multiple opens)        *
 *    void *UDF                ptr to the memory location for the structure   *
 *                             that holds returned data values (read_drec)    *
 *    short btime_yr           the start time requested (year component)      *
 *    short btime_day          the start time requested (day component)       *
 *    long btime_sec           the start time requested (seconds component)   *
 *    long btime_nsec          the start time requested (nanoseconds)         *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_fine_search (data_key, exten, vnum, UDF, btime_yr,      *
 *                     btime_day, btime_sec, btime_nsec)                      *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    read_drec()              the universal read routine that retrieves the  *
 *                             data for the time sample being processed       *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data          structure that holds all of the currently      *
 *       *ExDa                 returned data values to be processed           *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    struct ptr_rec *ptr      a pointer to the structure which holds all     *
 *                             pointers to the header and data for the        *
 *                             experiment of interest                         *
 *    long rec_bgn_sec         the starting time of the sweep being processed *
 *    long rec_bgn_nsec        the starting time of the sweep (nanoseconds)   *
 *    long rec_end_sec         the ending time of the sweep being processed   *
 *    long rec_end_nsec        the ending time of the sweep (nanoseconds)     *
 *    short rV                 holds the value returned by the called routine *
 *    char FswP                flag that indicates if 1 value is being        *
 *                             requested or all values for the record (for    *
 *                             a scalar parameter only)                       *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

ByTe_2 ir_fine_search (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                       void *UDF, ByTe_2 bY, ByTe_2 bD, ByTe_4 bS, ByTe_4 bNs)
{
   extern struct general_info ginfo;

   struct idf_data *ExDa;
   struct experiment_info *ex;
   struct ptr_rec *ptr;
   ByTe_4 rBs, rBns, rBt, bT;
   ByTe_4 rEs, rEns, rEt;
   ByTe_4 dY;
   ByTe_2 cC = 0, cCnT = 0, rV, rBy, rBd;
   ByTe_2 rEy, rEd;
   ByTe_1 FswP = 0, FwD;

  /**************************************************************************/
  /*  Set a pointer to the structure which holds pointers for header and    */
  /*  data information for the experiment currently being processed.        */
  /**************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;
   ExDa = (struct idf_data *) UDF;

  /**************************************************************************/
  /* reduce beginning year, day, sec to sec from year                       */
  /**************************************************************************/

   bT = 86400 * bD + bS;

  /**************************************************************************/
  /* Begin the fine search -- first read the data record which contains     */
  /* the user requested time (use no advance to test time value).  It does  */
  /* not matter which sensor we request since we are interested in the time.*/
  /**************************************************************************/

   rV = read_drec (data_key, exten, vnum, UDF, 0, cC, 0, FswP);
   if (rV < 0 || rV % 3 == 0) {
      if (rV == LOS_STATUS)
        return (PBACK_LOS);
      else if (rV == NEXT_FILE_STATUS)
        return (PBACK_NEXT_FILE);
      else { return (rV); }
   }

   rBy = ExDa->byear;
   rBd = ExDa->bday;
   rBs = (ExDa->bmilli + (ExDa->bnano / 1000000)) / 1000;
   rBns = (ExDa->bmilli % 1000) * 1000000 + ExDa->bnano;
   rEy = ExDa->eyear;
   rEd = ExDa->eday;
   rEs = (ExDa->emilli + (ExDa->enano / 1000000)) / 1000;
   rEns = (ExDa->emilli % 1000) * 1000000 + ExDa->enano;

   dY = rBy - bY;
   if (dY >= 0) {
      rBt = (365 * dY + rBd) * 86400 + rBs;
      if (dY == 1) {
         if ((bY % 4 == 0) && ((bY % 100 != 0) || (bY % 400 == 0)))
            rBt += 86400;
      }
   } else { rBt = -1; }

   dY = rEy - bY;
   if (dY >= 0) {
      rEt = (365 * dY + rEd) * 86400 + rEs;
      if (dY == 1) {
         if ((bY % 4 == 0) && ((bY % 100 != 0) || (bY % 400 == 0)))
            rBt += 86400;
      }
   } else { rEt = -1; }

   /**************************************************************************/
   /* Read the data record sweep by sweep and find the sweep which contains  */
   /* the user requested time.                                               */
   /**************************************************************************/

   do {

   /***********************************************************************/
   /*  If the time asked for results in a gap between the end of a record */
   /*  and the start of the next, terminate the search.                   */
   /***********************************************************************/

      if ( ((rBt > bT) || (rBt == bT && rBns >= bNs)) || 
           (((bT > rBt) || (rBt == bT && bNs >= rBns )) && 
           ((bT < rEt) || (rEt == bT && bNs < rBns))) ) {
         ex->next_offset = ptr->cur_sen_set;
         break;
      } else {

   /********************************************************************/
   /*  Advance to the next sweep since the current sweep does not hold */
   /*  the requested start time.                                       */
   /********************************************************************/

         if (ExDa->TotCols > 1) {
            if (++cCnT >= *ptr->N_COLS) {
               cCnT = 0;
               cC = 0;
               FwD = 1;
            } else {
               cC = *ptr->LEAD_COL + cCnT; 
               FwD = 0;
            }
         } else { 
            cC = 0;
            FwD = 1;
         }

         rV = read_drec (data_key, exten, vnum, UDF, 0, cC, FwD, FswP);
         if (rV < 0 || rV % 3 == 0) {
            if (rV == LOS_STATUS)
              return (PBACK_LOS);
            else if (rV == NEXT_FILE_STATUS)
              return (PBACK_NEXT_FILE);
            else { return (rV); }
         }

   /*********************************************************************/
   /*  Read the sweep. (use no advance to test time value).             */
   /*********************************************************************/

         if (FwD) {
            rV = read_drec (data_key, exten, vnum, UDF, 0, cC, 0, FswP);
            if (rV < 0 || rV % 3 == 0) {
               if (rV == LOS_STATUS)
                 return (PBACK_LOS);
               else if (rV == NEXT_FILE_STATUS)
                  return (PBACK_NEXT_FILE);
               else { return (rV); }
            }
         }

   /*********************************************************************/
   /* Compute the time of the data at this time period.                 */
   /*********************************************************************/

         rBy = ExDa->byear;
         rBd = ExDa->bday;
         rBs = (ExDa->bmilli + (ExDa->bnano / 1000000)) / 1000;
         rBns = (ExDa->bmilli % 1000) * 1000000 + ExDa->bnano;
         rEy = ExDa->eyear;
         rEd = ExDa->eday;
         rEs = (ExDa->emilli + (ExDa->enano / 1000000)) / 1000;
         rEns = (ExDa->emilli % 1000) * 1000000 + ExDa->enano;

         dY = rBy - bY;
         if (dY >= 0) {
            rBt = (365 * dY + rBd) * 86400 + rBs;
            if (dY == 1) {
               if ((bY % 4 == 0) && ((bY % 100 != 0) || (bY % 400 == 0)))
                  rBt += 86400;
            }
         } else { rBt = -1; }

         dY = rEy - bY;
         if (dY >= 0) {
            rEt = (365 * dY + rEd) * 86400 + rEs;
            if (dY == 1) {
               if ((bY % 4 == 0) && ((bY % 100 != 0) || (bY % 400 == 0)))
                  rBt += 86400;
            }
         } else { rEt = -1; }
      }
   } while (TRUE);

   return (ALL_OKAY);
}
