#include "util_str.h"
#include "gen_defs.h"
#include "libbase_udf.h"
#include "ret_codes.h"

/******************************************************************************
 *                                                                            *
 *                         FILE_OPEN SUBROUTINE                               *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine retrieves and opens the necessary header, data, and VIDF   *
 *  files for the time period and data set requested.  For the playback       *
 *  scenario if a filename extension is not given the default data files will *
 *  be retrieved and opened; otherwise the data files for the time period     *
 *  and filename extension requested will be retrieved and opened.  In        *
 *  addition, information that is needed from the VIDF file is retrieved, if  *
 *  necessary. In addition, the user must specify if mode processing is to be *
 *  done for the data set in question since modes are NOT tied to             *
 *  sensor-specific data and need their own ex structure.                     *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long Key            key which uniquely identifies the data set *
 *                                 being processed                            *
 *    char *exten                  filename extension for the data to be used *
 *    unsigned short vN            version number to be associated with this  *
 *                                 combination (allows for multiple opens)    *
 *    short bY                     the start time requested (year)            *
 *    short bD                     the start time requested (day)             *
 *    long bS                      the start time requested (seconds)         *
 *    long bNs                     the start time requested (nanoseconds)     *
 *    short eY                     the stop time requested (year)             *
 *    short eD                     the stop time requested (day)              *
 *    long eS                      the stop time requested (seconds)          *
 *    long eNs                     the stop time requested (nanoseconds)      *
 *    char mode_data               flag indicating that mode data is to be    *
 *                                 processed                                  *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = file_open (Key, exten, vN, bY, bD, bS, bNs, eY, eD, eS, eNs,        *
 *                   mode_data)                                               *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_locate_ex()               determines if the requested UDF definition *
 *                                 already been processed and points to the   *
 *                                 correct structure allocated for the combo  *
 *    ir_file_setup()              opens the data, header and vidf file and   *
 *                                 retrieves information from the VIDF file   *
 *    get_version_number ()        returns the next available version number  *
 *    ir_get_pitch_info()          retrieves pitch angle info. from VIDF file *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info          structure holding information concerning   *
 *         ginfo                   the experiment that is being processed     *
 *    char inside_reset_exp        flag indicating that  we are in the        *
 *                                 RESET_EXPERIMENT_INFO routine.  This is    *
 *                                 needed for FILE_OPEN logic                 *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    reg struct experiment_info   a pointer to the structure that holds      *
 *            *ex                  specific experiment information            *
 *    short rV                     the value returned by the called routine   *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 file_open (u_ByTe_4 Key, ByTe_1 *exten, u_ByTe_2 vN, 
                  ByTe_2 bY, ByTe_2 bD, ByTe_4 bS, ByTe_4 bNs, ByTe_2 eY, 
		  ByTe_2 eD, ByTe_4 eS, ByTe_4 eNs, ByTe_1 mode_data)
{
   extern struct general_info ginfo;

   register struct experiment_info *ex;
   ByTe_2 rV;

  /***********************************************************************/
  /*  Open the data, header and vidf files and retrieve information      */
  /*  from the VIDF file.                                                */
  /***********************************************************************/

   rV = ir_file_setup(Key, exten, vN, bY, bD, bS, bNs, eY, eD, eS, eNs);
   if (rV != ALL_OKAY) { return (rV); }

  /************************************************************************/
  /*  If the user specified that mode data is to be processed, a copy of  */
  /*  the ex structure is needed so call file_open with same parameters,  */
  /*  except for a new version number, to get another ex structure.       */
  /*  After this is done, call LOCATE_EX to point back to original ex     */
  /*  structure requested by user. If version number matches mode version */
  /*  number, it is implied that no mode data is to be processed.         */
  /************************************************************************/

   ex = ginfo.expt;

   /********************************************************************/
   /*  Retrieve pitch angle information if we haven't already done so. */
   /*  After this is done, call LOCATE_EX to point back to original ex */
   /*  structure requested by the user.                                */
   /********************************************************************/

   if (ex->pa_def && ex->bmem.base_pitch_info == NO_MEMORY)
    {
      rV = ir_get_pitch_info (bY, bD, bS, bNs, eY, eD, eS, eNs);
      if (rV != ALL_OKAY) 
        return (rV);
      rV = ir_locate_ex (Key, exten, vN, 0);
      ex = ginfo.expt;
    }

   return (ALL_OKAY);
}
