#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                      IR_CREATE_TBL SUBROUTINE                             *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to construct the table values using information *
 *  from the VIDF file.  In some instances, the only action to be taken is   *
 *  the scaling of the coefficients; that is, the table does not need to be  *
 *  expanded every time this routine is called.  The action is determined    *
 *  based upon the value of the parameter "expand_tbl".                      *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    short sen                the sensor being processed                    *
 *    short which_tbl          the number of the table being processed       *
 *    float *tbl               pointer to the table                          *
 *    long tbl_size            the maximum number of values associated with  *
 *                             the given sensor                              *
 *    long tbl_off             variable which holds the offset into table    *
 *                             (-1 if need to read this info. from VIDF file)* 
 *    char tbl_fmt             table format                                  *
 *    char expand_tbl          flag indicating if the table is to be expanded*
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_create_tbl (sen, &tbl, tbl_size, tbl_fmt, which_tbl, expand_tbl,*
 *                    tbl_off)                                               *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                the size of the specified object in bytes     *
 *    read_idf()               reads information from the IDF file           *
 *    ir_tento()               returns the power of 10 value to multiply by  *
 *                             to get values to the correct base (scaling)   *
 *    ir_poly_expand()         calculates values using a set of polynomial   *
 *                             coefficients                                  *
 *    ir_create_rval ()        returns an error code unique to this module   *
 *                             for each possible READ_IDF() error code       *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info      structure that holds information concerning   *
 *        ginfo                the experiment that is being processed        *
 *    unsigned int Tmp_Bytes   number of bytes allocated for scratch space   *
 *    void *Tmp_Space          scratch space for various uses                *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct experiment_info   a pointer to the structure that holds         *
 *          *ex                specific experiment information               *
 *    register float *f1,*f2   pointers to data values (float values)        *
 *    register float *stop_loop      loop termination variable               *
 *    register long *l1        pointer to the values read                    *
 *    register char *c1        pointer to the scaling values                 *
 *    unsigned int bytes       the number of bytes to allocate space for     *
 *    float x                  looping variable to construct the table       *
 *    float *fwk               the values read from the VIDF file once the   *
 *                             scaling factor has been applied               *
 *    float power              the scaling to be applied to the table values *
 *    long *lwk                memory that holds the values                  *
 *    long ret_val             holds the value returned by called routine    *
 *    long offset              sensor offset into table                      *
 *    long sca_size            the number of scaling factors                 *
 *    long base_off            indexing variable                             *
 *    char *cwk2               memory that holds the scaling values          *
 *    char scale               scaling factor to be applied to all values    *
 *    char tbl_var             flag indicating if the table is modifying     *
 *                             sensor or sweep step data                     *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *****************************************************************************/

ByTe_2 ir_create_tbl (ByTe_2 sen, ReaL_4 *tbl, ByTe_4 tbl_size, ByTe_1 tbl_fmt,
                      ByTe_2 which_tbl, ByTe_1 expand_tbl, ByTe_4 tbl_off)
{
   extern struct general_info ginfo;
   extern size_t Tmp_Bytes;
   extern void   *Tmp_Space;

   struct experiment_info *ex;
   register ReaL_4 *f1, *f2, *stop_loop;
   register ByTe_4 *l1;
   register ByTe_1 *c1;
   size_t bytes;
   ReaL_8 power;
   ReaL_4 x, *fwk;
   ByTe_4 *lwk, ret_val, offset, sca_size, base_off;
   ByTe_1 *cwk2, scale, tbl_var;

   /************************************************************************/
   /*  Allocate space to hold the values to be read from the VIDF file.    */
   /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc          */
   /************************************************************************/

   ex = ginfo.expt;
   bytes = (sizeof (ByTe_4) + sizeof (ReaL_4) + sizeof (ByTe_1)) * tbl_fmt;

   if (bytes > Tmp_Bytes) {
      if (bytes <= 0) { return (CREATE_TBL_MALLOC); }

      free(Tmp_Space);
      if ((Tmp_Space = malloc(bytes)) == NO_MEMORY)
        return (CREATE_TBL_MALLOC);
      Tmp_Bytes = bytes;
    }

   lwk  = (ByTe_4 *) Tmp_Space;
   base_off = sizeof (ByTe_4) * tbl_fmt;
   fwk  = (ReaL_4 *) ((ByTe_1 *)Tmp_Space + base_off);
   base_off += sizeof (ReaL_4) * tbl_fmt;
   cwk2 = (ByTe_1 *) ((ByTe_1 *)Tmp_Space + base_off);
   tbl_var = *(ex->tbl_var + which_tbl);
      
   /*************************************************************************/
   /*  Read the offset value into the table for this sensor, the values and */
   /*  the size of the scaling table.  If the offset value passed into this */
   /*  routine is -1, read the offset value into the table from the VIDF    */
   /*  file; otherwise, use the offset value passed into this routine.      */
   /*************************************************************************/

   if (tbl_off == -1)
    {
      ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                    (ByTe_1 *) &offset, _TblOfF, which_tbl, (ByTe_4) sen, 1);
      if (ret_val < 0)
        return (ir_create_rval (ret_val));
    }
   else
     offset = tbl_off;

   ret_val = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) lwk, 
                       _TbL, which_tbl, offset, (ByTe_2) tbl_fmt);
   if (ret_val < 0)
     return (ir_create_rval (ret_val));

   ret_val = read_idf (ex->data_key, ex->exten, ex->version, 
                       (ByTe_1 *) &sca_size, _TblScaSZ, which_tbl, 0L, 1);
   if (ret_val < 0)
     return (ir_create_rval (ret_val));

   /**********************************************************************/
   /*  There is one scaling value per value.                             */
   /**********************************************************************/

   if (sca_size > 0)
    {
       ret_val = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *)cwk2,
                           _TblScA, which_tbl, offset, (ByTe_2) tbl_fmt);
       if (ret_val < 0)
         return (ir_create_rval (ret_val));

       stop_loop = fwk + tbl_fmt;
       for (f1 = fwk, l1 = lwk, c1 = cwk2; f1 < stop_loop;)
           *f1++ = *l1++ * ir_tento (*c1++);
    }

   /**********************************************************************/
   /*  There is a single scaling value for each sensor.                  */
   /**********************************************************************/

   else if (sca_size < 0)
    {
      ret_val = read_idf (ex->data_key, ex->exten, ex->version, 
                     (ByTe_1 *) &scale, _TblScA, which_tbl, (ByTe_4) sen, 1);
      if (ret_val < 0)
        return (ir_create_rval (ret_val));

      power = ir_tento (scale);
      stop_loop = fwk + tbl_fmt;
      for (f1 = fwk, l1 = lwk; f1 < stop_loop;)
         *f1++ = *l1++ * power;
    }
  
   /**********************************************************************/
   /*  The stored values are considered already scaled.                  */
   /**********************************************************************/

   else if (sca_size == 0)
    {
      stop_loop = fwk + tbl_fmt;
      for (f1 = fwk, l1 = lwk; f1 < stop_loop;)
         *f1++ = *l1++;
    }

   /**********************************************************************/
   /*  If the table is to be expanded and if the table modifies sensor   */
   /*  data and if injested data is signed binary data, special table    */
   /*  expansion is necessary due to the signed nature.                  */
   /**********************************************************************/

   if (expand_tbl && tbl_var == 0 && *(ex->d_type + sen) == 1)
    {
      /********************************************************************/
      /*  Once the scaling factor(s) has been applied to the values, the  */
      /*  table should be formed using the set of polynomial coefficients.*/
      /*  TBL_SIZE is the number of elements in the expanded table.       */
      /*  FWK holds the scaled coefficients.                              */
      /********************************************************************/

      stop_loop = tbl + tbl_size;
      for (f1 = tbl, x = -tbl_size / 2.0; f1 < stop_loop; x += 1.0)
        *f1++ = ir_poly_expand ((ByTe_4 *)&x, fwk, (ByTe_4) tbl_fmt, -1);
    }
   else if (expand_tbl)
    {
      /********************************************************************/
      /*  Once the scaling factor(s) has been applied to the values, the  */
      /*  table should be formed using the set of polynomial coefficients.*/
      /*  TBL_SIZE is the number of elements in the expanded table.       */
      /*  FWK holds the scaled coefficients.                              */
      /********************************************************************/

      stop_loop = tbl + tbl_size;
      for (f1 = tbl, x = 0.0; f1 < stop_loop; x += 1.0)
         *f1++ = ir_poly_expand ((ByTe_4 *)&x, fwk, (ByTe_4) tbl_fmt, -1);
    }

   /*************************************************************************/
   /*  Copy the scaled coefficients into the table; don't expand the table. */
   /*************************************************************************/

   else
    {
      stop_loop = tbl + tbl_fmt;
      for (f1 = tbl, f2 = fwk; f1 < stop_loop;)
        *f1++ = *f2++;
    }

   return (ALL_OKAY);
}

/*******************************************************************************
 *                                                                             *
 *                          IR_CREATE_RVAL SUBROUTINE                          *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine returns an error code unique to the module CREATE_TBL()     *
 *  based upon the error code returned by the call to READ_IDF().              *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    long ret_val                the error code returned by READ_IDF()        *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_create_rval (ret_val)                                             *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_create_rval (ByTe_4  ret_val)
{
   switch (ret_val)
    {
       case IDF_NOT_FOUND:
         return (CREATE_IDF_NOT_FOUND);
       case IDF_MANY_BYTES:
         return (CREATE_IDF_MANY_BYTES);
       case IDF_TBL_NUM:
         return (CREATE_IDF_TBL_NUM);
       case IDF_CON_NUM:
         return (CREATE_IDF_CON_NUM);
       case IDF_NO_ENTRY:
         return (CREATE_IDF_NO_ENTRY);
       default:
         return ((ByTe_2) ret_val);
    }
}
