#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "user_defs.h"
#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                        IR_CONVERT_CAL_DATA SUBROUTINE                     *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to convert the ancillary data for the requested *
 *  sensor into units by applying the specified tables and table operations  *
 *  in the given order to achieve the desired output. The order is implied   *
 *  by the table specification.  If the chk_fill flag is set to 1, the data  *
 *  values will be compared against the specified FILL value.  If the data   *
 *  value is identified as a fill value, the value OUTSIDE_MIN will be       *
 *  returned and it is up to the user to decide what is to be done with this *
 *  value.  The FILL value must be specified in terms of raw telemetry.      *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    char cal_set                 calibration set number                    *
 *    char num_tbls                number of tables to apply                 *
 *    char *tbls_to_apply          array of tables to be applied             *
 *    short *tbl_oper              operations applied with the tables        *
 *    char chk_fill                flag to see if the data is to be checked  *
 *                                 for FILL values                           *
 *    float *DataOut               array in which data is returned in the    *
 *                                 converted units                           *
 *    long fill_value              the value identified as the FILL value    *
 *    void *UDF                    ptr to memory location for the structure  *
 *                                 holding returned data values (read_drec)  *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_convert_cal_data (cal_set, num_tbls, tbls_to_apply, tbl_oper    *
 *                             &DataOut, chk_fill, fill_value, UDF)          *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    ir_transfer_data ()          converts the raw integer telemetry to the *
 *                                 raw units (integer or float)              *
 *    abs()                        returns the absolute value of a number    *
 *    ir_idf_ex_func ()            performs higher order table operations    *
 *    IDF_OPER[]._math_drv ()      performs the arithmetic operation defined *
 *                                 for the table being processed             *
 *    CombineOp                    performs the arithmetic operation defined *
 *                                 to combine the permanent and temporary    *
 *                                 accumulators                              * 
 *    sizeof ()                    the size of the specified object in bytes *
 *    malloc()                     allocates memory                          *
 *    free ()                      frees previously allocated memory         *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info ginfo    structure that holds information in thel  *
 *                                 experiment being processed                *
 *    struct def_oper IDF_OPER[]   array of pointers to routines that do the *
 *                                 arithmetic operations defined for the     *
 *                                 sensor, sweep, and anc data tables        *
 *    void *Tmp_Space              scratch space for various uses            *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct idf_data *ExDa        structure that holds all of the currently *
 *                                 returned data values to be processed      *
 *    struct experiment_info *ex   a pointer to the structure that holds     *
 *                                 specific experiment information           *
 *    struct sensor_tables *sptr   a pointer to the structure which holding  *
 *                                 various info for the tables utilized by   *
 *                                 sensor-table combination being processed  *
 *    register unsigned short *s1  pointer to number in each cal set info.   *
 *    register char i              looping variable                          *
 *    unsigned int bytes           the number of bytes to allocate           *
 *    float *RetD                  pointer to the temporary array that holds *
 *                                 the cal. data as it is being converted    *
 *    long data_start              offset into the data at which to start    *
 *                                 picking up data values (used for cal.)    *
 *    long num_data                the number of data values being returned  *
 *    long HoldVal                 number of successive data values within a *
 *                                 data set that one calibration value is to *
 *                                 be applied to                             *
 *    long num_ele_in_tbl          no. of elements in table being applied    *
 *    long stop_ind                the stopping value for the loop           *
 *    unsigned short *sEnD         loop termination variable                 *
 *    ByTe_2 BasOp                 basic table operation                     *
 *    ByTe_2 ExOp                  extended (after table) operation          *
 *    ByTe_2 BufOp                 buffer operation modifier                 * 
 *    ByTe_2 PosOp                 positive operator                         * 
 *    char *sen                    pointer to sensor data from READ_DREC()   *
 *    char *swp                    pointer to sweep data from READ_DREC()    *
 *    char *cal                    pointer to the cal. data from READ_DREC() *
 *    char dtype                   format of the data - uns int, int, float  *
 *    char fn_of_raw               flag indicating if table is a function    *
 *                                 of raw data or processed data             *
 *    char *DFrom                  ptr to the data values which the table    *
 *                                 values are to be applied to               *
 *    float *DataFrom              this is the input data array              *
 *    float *DataTo                this is the output data array             *
 *    char which_cal               the calibration set being processed       *
 *    char *mode                   pointer to the mode data from READ_DREC() *
 *    void *tmp_ptr                ptr which holds address passed back by    *
 *                                 the call to the MALLOC routine            *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 ****************************************************************************/

ByTe_2 ir_convert_cal_data (ByTe_1 cal_set, ByTe_1 num_tbls, 
                            ByTe_1 *tbls_to_apply, ByTe_2 *tbl_oper, 
                            ReaL_4 *DataOut, ByTe_1 IFmt, ByTe_1 chk_fill, 
                            ByTe_4 fill_value, void *UDF)
{
  extern struct general_info ginfo;
  extern struct def_oper IDF_OPER[];
  extern void   *Tmp_Space;
  extern size_t Tmp_Bytes;
  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct sensor_tables *sptr;
  register ReaL_8 *d1;
  register u_ByTe_2 *s1;
  register ByTe_2 *OP;
  register ByTe_1 *TbL, *Tend;
  size_t bytes;
  ReaL_8 *RetD, *RetB, *dEnD, *dStoP, *dLasT; 
  ReaL_8 *DataFrom, *DataTo; 
  ReaL_8 SenNum[2];
  ReaL_4 *TblVals, *f1;

  ByTe_4 *l1, data_start, num_data, HoldVal, num_ele_in_tbl;
  ByTe_4 T, F;
  ByTe_4 cRow;
  u_ByTe_2 *sEnD;
  ByTe_2 BasOp, ExOp, BufOp, PosOp;
  ByTe_2 A, C, R;
  ByTe_1 *sen, *swp, *cal, target, dtype, fn_of_raw; 
  ByTe_1 *DFrom, which_cal, *mode, DType;
  ByTe_1 tV;
  void *Scratch;

  /**********************************************************************/
  /*  Set pointers to sensor, sweep step, calibration and status (mode) */
  /*  data returned by the READ_DREC() routine.                         */
  /**********************************************************************/
 
  ExDa = (struct idf_data *) UDF;
  sen = (ByTe_1 *) ExDa->sen_data;
  swp = (ByTe_1 *) ExDa->swp_data;
  cal = (ByTe_1 *) ExDa->cal_data;
  mode = (ByTe_1 *) ExDa->mode;

  target = *(ginfo.expt->cal_target + cal_set);

  /**********************************************************************/
  /* If this ancillary value is attached to a scalar sensor then we     */
  /* need to know which one(s) in the column are being picked up        */
  /**********************************************************************/

  if ((target == 0) && (ginfo.expt->smp_id == 2) ) {
     cRow = ginfo.expt->info_ptr->time_row ;
  } else { cRow = 0; }

  /**********************************************************************/
  /*  Allocate space to temporarily hold cal. values since data needs   */
  /*  to be expanded out to vector length (one cal per data returned).  */
  /**********************************************************************/

  bytes = sizeof(ReaL_8) * (ExDa->num_sample + cRow);
  if (bytes <= 0) { return (CONV_CAL_MALLOC); }
  if ((Scratch = malloc (bytes)) == NO_MEMORY) { return (CONV_CAL_MALLOC); }
  RetD = (ReaL_8 *) Scratch;

  if (bytes > Tmp_Bytes) {
     free(Tmp_Space);
     if ((Tmp_Space = malloc(bytes)) == NO_MEMORY) {
        free (Scratch);
        return (CONV_CAL_MALLOC);
     }
     Tmp_Bytes = bytes;
  }
  RetB = (ReaL_8 *) Tmp_Space;

  /**********************************************************************/
  /*  Set pointers to the current sensor being processed.               */
  /**********************************************************************/

  ex = ginfo.expt;
  sptr = ex->sen_tbl_ptr + *(ex->index_sen_tbl + ExDa->sensor);

  /***********************************************************************/
  /*  Determine the index into the calibration data array at which this  */
  /*  calibration set starts.  Transfer the integer raw values into      */
  /*  the data array to be returned to the user.  There is no D_TYPE for */
  /*  calibration data.  It is always an unsigned long.                  */
  /***********************************************************************/

  s1 = ExDa->cset_num;
  sEnD = s1 + cal_set;
  for (data_start = 0; s1 < sEnD; )
    data_start += *s1++;

  num_data = *(ExDa->cset_num + cal_set);
  DType = (*(ex->cal_wlen + cal_set) < 0) ? 1 : 0;
  
  ir_transfer_data (RetB, cal, num_data, DType, data_start);

  /*********************************************************************/
  /* Expand the data to cover a full sweep.  Moved this code from the  */
  /* bottom to the top.  Turns out that all anc sets expand to the     */
  /* target set.  If it is not the largest then problems occur when    */
  /* they are combined.  So this expands the target set to the max     */
  /*                                                                   */
  /* CHANGE made here to fully expand scalar ancillary sets to the     */
  /*   the number of measurements per sensor in the sensor set         */
  /*********************************************************************/

  HoldVal = (*(ex->CAL_USE + cal_set) == 0) ? ExDa->num_sample :
                                              *(ex->CAL_USE + cal_set);

  dEnD = RetB + num_data;
  d1 = (ReaL_8 *)RetD;
  dLasT = d1 + ExDa->num_sample;
  if (HoldVal == 1) {
     for ( ; d1 < dLasT; ) { *d1++ = *RetB++; }
  } else { 
     for ( ; RetB < dEnD; ++RetB) {
        dStoP = (d1 + HoldVal > dLasT) ? dLasT : d1 + HoldVal;
        for ( ; d1 < dStoP; ) { *d1++ = *RetB; }
     }
  }

  RetB = (ReaL_8 *)Tmp_Space;
  RetD += cRow;
  DataTo = RetD;

  /*********************************************************************/
  /*  If tables are to be applied, convert to the data accordingly.    */
  /*********************************************************************/

  OP = tbl_oper;
  TbL = tbls_to_apply;
  Tend = TbL + num_tbls;
  for (  ; TbL < Tend; ++OP, ++TbL)
   {
     /*******************************************************************/
     /*  Check for combination of accumulators since operations are NOT */
     /*  table specific.                                                */
     /*******************************************************************/

     tV = *(sptr->tbl_var + *TbL);                  /* table variable        */
     PosOp = (*OP < 0 ) ? -(*OP): *OP;              /* positive operator     */
     BufOp = PosOp / 1000;                          /* buffer specific op    */
     ExOp = (PosOp % 1000) / 10;                    /* extended operation    */
     BasOp = (*OP % 10);                            /* basic operation       */
     if (BasOp < 0)                                 /* negative base oper?   */
        BasOp = 9 - BasOp;                          /* make it positive      */
     
     if (BufOp > 1 && BufOp < 20)                   /* buf-buf operation     */
     {                                              /* BEG BUFFER OPERATION  */
        R = BufOp - 2;                              /* reduced buffer OP     */
        A = R % 4;                                  /* seq ops 0-3,0-3,0-3   */
        C = R / 4;                                  /* oper block            */
        if (C < 3)                                  /* 2-way buf operations  */
        {                                           /* BEGIN 2-WAY OPS       */
           T = A;                                   /* finding TO buffer     */
           F = (A == 0) ? C + 1 : (A == 3) ? C :    /* finding FROM buffer   */
                        (C < 2) ? R/2 : 3*(A % 2);  /* offset                */
        }                                           /* END 2-WAY OPS         */
        else                                        /* 1-way buf operations  */
        {                                           /* BEGIN 1-WAY OPS       */
           T = (R % 3) + 4;                         /* finding TO buffer     */
           F = (R > 16) ? (R % 16) : 0;             /* finding FROM buffer   */
        }                                           /* END 1-WAY OPS         */

        F = (F - 1) * ex->swp_len;                  /* from byte offset      */
        T = (T - 1) * ex->swp_len;                  /* to byte offset        */

        DataTo = (T < 0) ? RetD : RetB + T;         /* set the DataTo buffer */
        DataFrom = (F < 0) ? RetD : RetB + F;       /* set DataFrom buffer   */

        CombineOp (DataTo, DataFrom, ExDa->num_sample, BasOp); /* basic ops  */

        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/

        if (ExOp > 0)
          ir_idf_ex_func (ExOp, DataTo, ExDa->num_sample, UDF);
        continue;
      }

     /********************************************************************/
     /*  A check is made upfront for ASCII tables and tables associated  */
     /*  with status mode information.                                   */
     /********************************************************************/

     if (*(ex->tbl_type + *TbL) == 1)
       continue;
     else if (*(ex->tbl_var + *TbL) == 4 || *(ex->tbl_var + *TbL) == 5)
       continue;

     /********************************************************************/
     /*  Is this table to be applied for the conversion of the data?     */
     /*  A size of 0 indicates that no table is defined (tbl_fmt < 0);   */
     /*  therefore, this table is bypassed for data conversion.          */
     /********************************************************************/

     if (tV != 0)
      {
        fn_of_raw = (tV == 1 || tV >= 7) ? 0 : 1;
        T = ((BufOp % 18) - 1) * ex->swp_len;
        DataTo = (T < 0) ? RetD : RetB + T;

        /*****************************************************************/
        /* This table is a function of a calibration set.  Determine the */
        /* index value to get to the calibration values in question.     */
        /*****************************************************************/

        data_start = 0;
        if (tV < 0)
         {
           which_cal = abs (tV) - 1;
           num_data = *(ExDa->cset_num + which_cal);
           s1 = ExDa->cset_num;
           sEnD = s1 + which_cal;
           for ( ; s1 < sEnD; )
              data_start += *s1++;

           /***************************************************************/
           /*  Determine the number of data elements each cal. value is   */
           /*  to be applied to.  A single cal. value is to be applied to */
           /*  each data element if CAL_USE = 0.                          */
           /***************************************************************/

           if (*(ex->CAL_USE + which_cal) == 0)
             HoldVal = ExDa->num_sample;
           else
             HoldVal = *(ex->CAL_USE + which_cal);
         }

        /***************************************************************/
        /* Start at the beginning of the data array and there is a one-*/
        /* to-one mapping between table elements and data elements.    */
        /***************************************************************/

        else
           HoldVal = (tV != 7) ? 1 : ex->num_sample;

        /************************************************************/
        /*  Get the data type and input data array.                 */
        /************************************************************/

        switch (tV) 
        {
            case 0:                                /* RAW SENSOR DATA */
            case 2:                                /* RAW SCAN DATA   */
            case 4:                                /* RAW MODE DATA   */
            case 5:                                /* PROC MODE DATA  */
            case 6:                                /* RAW QUAL DATA   */
              free (Scratch);
              return (CONV_CAL_VECTOR_MISMATCH);
            break;
            case 1:                                /* PROCESSED DATA */
               DFrom = (ByTe_1 *)DataTo;
               dtype = 2;
            break;
            case 7:                                /* SENSOR NUMBER  */
            case 8:                                /* MATRIX COL NUM */
               SenNum[0] = (tV == 7) ? ExDa->sensor : ExDa->column;
               DFrom = (ByTe_1 *)SenNum;
               dtype = 2;
            break;
            default:                               /* RAW ANC DATA   */
              DFrom = cal;
              dtype = (*(ex->cal_wlen + which_cal) < 0) ? 1 : 0;
            break;
        }

        /****************************************************************/
        /*  Perform the operation defined for the table.                */
        /****************************************************************/


        if (*(ex->tbl_type + *TbL) == 2 || *(ex->tbl_type + *TbL) == 4)
           num_ele_in_tbl = *(sptr->tbl_size + *TbL) / ex->swp_len;
        else
           num_ele_in_tbl = *(sptr->tbl_size + *TbL);

        TblVals = *(sptr->tbl_ptr + *TbL);
        if (*(ex->tbl_type + *TbL) == 5) 
           TblVals += ExDa->column * ex->swp_len;

        IDF_OPER[BasOp]._math_drv (DataTo, DFrom, data_start, TblVals,  
                    num_ele_in_tbl, ExDa->num_sample, 
                    *(sptr->tbl_expand + *TbL), dtype, HoldVal, 
                    fn_of_raw, *(ex->tbl_type + *TbL), UDF, ex->swp_len);
  
        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/

        if (ExOp > 0)
           ir_idf_ex_func (ExOp, DataTo, ExDa->num_sample, UDF);
     }
   }

  /************************************************************************/
  /*  Check the data for FILL values.  If the data is found to be fill    */
  /*  data, the value returned is set to OUTSIDE_MIN.                     */
  /************************************************************************/

   dEnD = RetD + ExDa->num_sample;
   if (chk_fill)
   {
      l1 = ExDa->cal_data + data_start;
      for ( ; RetD < dEnD; ++RetD)
      {
         if (*l1++ == fill_value)
            *RetD = OUTSIDE_MIN;
      }
   }

  /*************************************************************************/
  /*  Transfer the output to the correct floating point format             */
  /*************************************************************************/

   RetD = DataTo;
   if (IFmt == 1) {
      d1 = (ReaL_8 *)DataOut;
      for ( ; RetD < dEnD; )
           *d1++ = *RetD++;
   } else {
      f1 = DataOut;
      for ( ; RetD < dEnD; )
           *f1++ = *RetD++;
   }

   free (Scratch);
   return (ALL_OKAY);
}
