#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                        IR_CHECK_DEPENDENT_ON_HEADER SUBROUTINE             *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to determine if the arrays for the sensor data,  *
 *  mode data, sample sequence values, azimuthal angles and pitch angles are  *
 *  large enough to accomodate the data based upon the header sizes.  If not, *
 *  the space is enlarged to the newly defined size.                          *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *idf_data_ptr         ptr to the memory location for the structure *
 *                               that holds returned data values (read_drec)  *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_check_dependent_on_header (idf_data_ptr)                         *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    malloc()                   allocates memory                             *
 *    sizeof ()                  the size of the specified object in bytes    *
 *    ir_alloc_swp_data()        allocates space for sample sequence values   *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *ExDa  structure that holds all of the currently        *
 *                               returned data values to be processed         *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    size_t B                   the number of bytes to allocate              *
 *    unsigned short num_ele     number of sample sequence values             *
 *    short ret_val              holds value returned by called routine       *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_check_dependent_on_header (void *idf_data_ptr)
{
   extern struct general_info ginfo;

   struct idf_data *ExDa;
   struct experiment_info *ex;
   struct ptr_rec *ptr;
   size_t B;
   u_ByTe_2 num_ele;
   ByTe_2 ret_val;

  /************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header  */
  /*  and data information for the experiment currently being processed.  */
  /************************************************************************/

   ex  = ginfo.expt;
   ptr = ex->info_ptr;
   ExDa = (struct idf_data *) idf_data_ptr;

  /*************************************************************************/
  /*  Allocate space to hold the sensor data that will be returned to the  */
  /*  user if no memory has been allocated yet.                            */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc           */
  /*************************************************************************/

   B = *(ptr->N_SAMPLE) * sizeof (ByTe_4);
   if (B <= 0) { return (RESET_DATA_MALLOC); }
   if ((B > ExDa->data_size) && (ExDa->base_data != NO_MEMORY)) {
      free(ExDa->base_data);
      ExDa->base_data = NO_MEMORY;
      ExDa->data_size = 0;
   }

   if (ExDa->base_data == NO_MEMORY) {
      if ((ExDa->base_data = malloc (B)) == NO_MEMORY)
         return (RESET_DATA_MALLOC);
      ExDa->sen_data = (ByTe_4 *)ExDa->base_data;
      ExDa->data_size = B;
   } 

  /**************************************************************************/
  /*  Allocate space to hold the start and stop aziumthal angles that are   */
  /*  to be returned to the user if no memory has been allocated yet.       */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc           */
  /**************************************************************************/

   B = 4 * (*ptr->N_SAMPLE * sizeof (ReaL_4));  
   if (B <= 0) { return (RESET_ANGLE_MALLOC); }
   if ((B > ExDa->angle_size) && (ExDa->base_angle != NO_MEMORY)) {
      free(ExDa->base_angle);
      ExDa->base_angle = NO_MEMORY;
      ExDa->angle_size = 0;
   }

   if (ExDa->base_angle == NO_MEMORY) {
      if ((ExDa->base_angle = malloc (B)) == NO_MEMORY)
         return (RESET_ANGLE_MALLOC);

      ExDa->start_az = (ReaL_4 *) ExDa->base_angle;
      ExDa->stop_az = ExDa->start_az + *ptr->N_SAMPLE;
      ExDa->start_theta = ExDa->stop_az + *ptr->N_SAMPLE;
      ExDa->stop_theta = ExDa->start_theta + *ptr->N_SAMPLE;
      ExDa->angle_size = B;
   } 

  /**************************************************************************/
  /*  Allocate space to hold the pitch angles that are to be returned to    */
  /*  the user if no memory has been allocated yet.                         */
  /**************************************************************************/

  /**************************************************************************/
  /*  CAG - we only need this memory if you can compute pitch angles so     */
  /*  check that before allocating! 11/19/2008                              */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc           */
  /**************************************************************************/

   if ( ex->pa_def ) {
      B = *ptr->N_SAMPLE * sizeof (ReaL_4);
      if (B <= 0) { return (RESET_PITCH_MALLOC); }
      if ((B > ExDa->pitch_size) && (ExDa->base_pitch != NO_MEMORY)) {
         free(ExDa->base_pitch);
         ExDa->base_pitch = NO_MEMORY;
         ExDa->pitch_size = 0;
      }

      if (ExDa->base_pitch == NO_MEMORY) {
         if ((ExDa->base_pitch = malloc (B)) == NO_MEMORY)
            return (RESET_PITCH_MALLOC);

         ExDa->pitch_angles = (ReaL_4 *) ExDa->base_pitch;
         ExDa->pitch_size = B;
      } 
   }

  /************************************************************************/
  /*  Allocate space to hold the mode flags that are to be returned to    */
  /*  the user, retured as longs, if modes are defined for this data set. */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc           */
  /************************************************************************/

   if (*ptr->I_MODE != 0) {
      B = *(ptr->I_MODE) * sizeof (ByTe_4);
      ExDa->mode_len = *(ptr->I_MODE);

      if ((B > ExDa->mode_size) && (ExDa->base_mode != NO_MEMORY)) {
         free(ExDa->base_mode);
         ExDa->base_mode = NO_MEMORY;
         ExDa->mode_size = 0;
      }

      if (ExDa->base_mode == NO_MEMORY) {
         if ((ExDa->base_mode = malloc (B + 16)) == NO_MEMORY)
            return (RESET_MODE_MALLOC);
         ExDa->mode = (ByTe_4 *) ExDa->base_mode;
         ExDa->mode_size = B + 16;
      }
   }

  /*************************************************************************/
  /*  Allocate space to hold the sample sequence values.                   */
  /*************************************************************************/

   num_ele = (ex->smp_id == 2) ? 1 : *(ptr->N_SAMPLE);
   ret_val = ir_alloc_swp_data (num_ele, idf_data_ptr);
   if (ret_val != ALL_OKAY) { return (ret_val); }

  /**************************************************************************/
  /*  Allocate space to hold the number of elements in each cal. set, which */
  /*  is needed by convert_to_units since the values can change from header */
  /*  to header.                                                            */
  /**************************************************************************/

   if (ex->cal_sets != 0) {
      if (ExDa->base_cset == NO_MEMORY) {
         B = ex->cal_sets * sizeof (u_ByTe_2);
         if ((ExDa->base_cset = malloc (B)) == NO_MEMORY)
           return (RESET_CSET_MALLOC);
         ExDa->cset_num = (u_ByTe_2 *) ExDa->base_cset;
      }
   }

  return (ALL_OKAY);
}
