#include "util_str.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                      IR_ADVANCE_TIME_POINTERS SUBROUTINE                   *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine manipulates the two time pointers, time_row and time_col,  *
 *  so that the NEXT time sample is pointed to.  The values are dependent on  *
 *  the SEN_MODE value for the experiment / virtual / extension combination   *
 *  being processed.                                                          *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    char full_swp              flag that indicates if 1 value is being      *
 *                               requested or all values for the record (for  *
 *                               a scalar parameter only)                     *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_advance_time_pointers (full_swp)                                     *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    reg struct experiment_info a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    reg struct ptr_rec *ptr    a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_advance_time_pointers (ByTe_1  full_swp)
{
  extern struct general_info ginfo;

  register struct experiment_info *ex;
  register struct ptr_rec *ptr;
 
  ByTe_4 NCols;

  /***************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and */
  /*  data information for the experiment currently being processed.         */
  /***************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;

  /*************************************************************************/
  /* get number of columns of data in a sensor set.  Note that with matrix */
  /* sensors - all sensors in a sensor set have same start time.  A fwd    */
  /* then must force a change in sensor set.  For matrix sensors the       */
  /* sen_mode is applicable to the matrix elements and not to the sensors. */
  /*************************************************************************/

  NCols = (ex->smp_id != 3) ? *ptr->N_SEN : 1;

  /*************************************************************************/
  /*  The time increases down the column for this mode.                    */
  /*************************************************************************/

  if (ex->sen_mode == 0)
   {
     ptr->time_row = (ex->smp_id == 2 && !full_swp) ? ptr->time_row + 1 : 
                     ptr->time_row + *ptr->N_SAMPLE;

     /*********************************************************************/
     /*  If all rows have been processed, go to the next column of data.  */
     /*********************************************************************/

     if (ptr->time_row >= (int) (*ptr->N_SAMPLE))
      {
        ptr->time_row = 0;
        ++ptr->time_col;

        /*********************************************************************/
        /*  If all columns have been processed, go to the next sensor set.   */
        /*********************************************************************/

        if (ptr->time_col >= NCols)
         {
           ptr->time_col = 0;
           ++ptr->cur_sen_set;
           ptr->chg_sen_set = 1;
         }
      }
   }

  else if (ex->sen_mode == 1 || ex->sen_mode == 5)
   {
     ++ptr->time_col;

     /*********************************************************************/
     /*  If all columns have been processed, go to the next sensor set.   */
     /*********************************************************************/

     if (ptr->time_col >= NCols)
      {
        ptr->time_col = 0;
        ++ptr->cur_sen_set;
        ptr->chg_sen_set = 1;
      }
   }

  /**************************************************************************/
  /*  CODE MOD - CAG 01/1999 - problem with scalar/parallel corrected       */
  /*                                                                        */
  /*  For sen_mode equal to 2 and 6, the data for all sensors are taken at  */
  /*  the same time, and since all rows are returned at once. For sen_mode  */
  /*  equal to 3 and 7, all data for all sensors for all samples are taken  */
  /*  at the same time. In all cases, for non-scalar data advance to the    */
  /*  next sensor set.  For scalar data get to the next row of data         */
  /**************************************************************************/

  else if (ex->sen_mode == 2 || ex->sen_mode == 6 || ex->sen_mode == 3 || 
           ex->sen_mode == 7)
   {
      if (ex->smp_id != 2)
      {
          ++ptr->cur_sen_set;
          ptr->chg_sen_set = 1;
      }
      else
      {
         /********************************************************************/
         /*  If a single scalar value is being returned, go to the next row; */
         /*  otherwise, the whole row has been returned and go to the next   */
         /*  sensor set.                                                     */
         /********************************************************************/

          ptr->time_row = (!full_swp) ? ptr->time_row + 1 : 
                                        ptr->time_row + *ptr->N_SAMPLE;

         /*******************************************************************/
         /* If all the rows have been processed, go to the next sensor set. */
         /*******************************************************************/

         if (ptr->time_row >= (int) (*ptr->N_SAMPLE))
         {
            ptr->time_row = 0;
            ++ptr->cur_sen_set;
             ptr->chg_sen_set = 1;
         }
      }
   }

  /*************************************************************************/
  /*  The time increases across the row for this mode.  Time sampling is   */
  /*  different for scalar and sweeping instrument (data is laid down      */
  /*  differently.                                                         */
  /*************************************************************************/

  else if (ex->sen_mode == 4)
   {
     ++ptr->time_col;

     /******************************************************************/
     /*  If all the columns have been processed, go to the next row of */
     /*  data.                                                         */
     /******************************************************************/

     if (ptr->time_col >= NCols)
      {
        ptr->time_col = 0;
        ptr->time_row = (ex->smp_id == 2 && !full_swp) ? ptr->time_row + 1 : 
                        ptr->time_row + *ptr->N_SAMPLE;

        /*********************************************************************/
        /*  If all rows have been processed, go to the next sensor set.      */
        /*********************************************************************/

        if (ptr->time_row >= (int) (*ptr->N_SAMPLE))
         {
           ptr->time_row = 0;
           ++ptr->cur_sen_set;
           ptr->chg_sen_set = 1;
         }
      }
   }
}
