#include <math.h>
#include "util_str.h"
#include "user_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                             PAFORMAT1                                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine computes the pitch angle for each time period associated   *
 *  with each element of the sweep.  The dot product of the unit normal to    *
 *  the detector aperture with the local magnetic field is computed.  The     *
 *  value computed is in  radians and is is converted to degrees before it    *
 *  is returned.                                                              *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    struct pitch_info *pA      a pointer to the structure that holds pitch  *
 *                               angle information                            *
 *                               are returned                                 *
 *    void *UDF                  pointer to the data structure                *
 *    short SeN                  the sensor for which data is requested       *
 *    unsigned short nElem       the number of elements in the sweep          *
 *                                                                            *
 *  USAGE                                                                     *
 *    PaFormat0 (pA, pData, SeN, nElem)                                       *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sqrt ()                     returns the square root of the argument     *
 *    acos ()                     returns arc cosine of input                 *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    reg unsigned short Step     the sweep step being processed              *
 *    register double *nVec       ptr to the unit normal vector               *
 *    register double *mVec       ptr to the magnetic field vector            *
 *    register double *mEnd       loop termination variable                   *
 *    register ReaL_4 *pD         ptr to pitch angle data                     *
 *    register ReaL_4 *pDEnd      ptr to end of pitch angle data              *
 *    double Dot                  result of calculation for the numerator     *
 *    double Mag                  summation of the magnetic field elements    *
 *    double DotProd              computed dot product value                  *
 *    double rData                computed pitch angle value                  *
 *    int offset                  offset into allocated memory                *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void PaFormat1 (struct pitch_info *pA, void *UDF, ByTe_2 SeN, 
                u_ByTe_2 nElem)
{
   struct idf_data *ExDa;
   register ReaL_8 *nVec, *mVec, *mEnd, *MVec;
   register ReaL_4 *pD, *pDEnd, *aZb, *aZe;
   ReaL_8 Dot, Mag, DotProd, rData, dA;
   ReaL_8 AvgAz;
   ReaL_8 cN[3];
   ReaL_8 cM[3];

   /*********************************************************************/
   /*  Process each element of the sweep.                               */
   /*********************************************************************/

   ExDa = (struct idf_data *) UDF;
   aZb = (ReaL_4 *)ExDa->start_az;
   aZe = (ReaL_4 *)ExDa->stop_az;
   pD = ExDa->pitch_angles;
   pDEnd = pD + nElem;
   MVec = pA->data_val;
   for ( ; pD < pDEnd; ++aZb, ++aZe) {

      dA = *aZe - *aZb;
      if (ExDa->spin_rate > 0)
         AvgAz = (dA > 0.0) ? *aZb + dA/2.0 : *aZb + dA/2.0 + 180.0;
      else
         AvgAz = (dA < 0.0) ? *aZb + dA/2.0 : *aZb + dA/2.0 + 180.0;

      AvgAz = fmod(AvgAz, 360.0) * TORAD;

      nVec = pA->unit_normals + (SeN * 3);
      cN[0] = nVec[0] * cos (AvgAz) - nVec[1] * sin (AvgAz); 
      cN[1] = nVec[0] * sin (AvgAz) + nVec[1] * cos (AvgAz); 
      cN[2] = nVec[2]; 
      cM[0] = *MVec++;
      cM[1] = *MVec++;
      cM[2] = *MVec++;

      /*********************************************************************/
      /*  Do a summation from 1 to N for the numerator and denominator.    */
      /*  Since the unit normal are constants, the magnitude was computed  */
      /*  when the three components were retrieved.                        */
      /*********************************************************************/
      
      nVec = cN;
      mVec = cM;
      mEnd = cM + 3;
      Mag = Dot = 0.0;
      for ( ; mVec < mEnd; ++mVec ) {
         Dot += *nVec++ * *mVec;
         Mag += *mVec * *mVec;
      }
      
      /*********************************************************************/
      /*  Solve for the denominator and compute the dot product.           */
      /*********************************************************************/
      
      Mag = sqrt (Mag);
      DotProd = (-1.0 * Dot) / Mag / pA->mag_normal[SeN];
      rData = acos (DotProd);
      *pD++ = (ReaL_4) (rData / TORAD);
   }
}
