#include "util_str.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                          CorForConing                                    *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called in order to fill azimuthal start and stop      *
 *  angles for each data value returned for a given sensor.  If the spin    *
 *  rate indicates a parked or otherwise non-rotating status the SUN_SEN    *
 *  field holds the experiment angle multiplied by 100.  If the experiment  *
 *  is spinning, the calulation for the start and stop angles is performed  *
 *  and the values are returned to the user in allocated memory.  The       *
 *  starting angle values are always returned as values between 0.0 and     *
 *  359.  However, the ending angle can be negative if the deg_accum value  *
 *  is negative (negative direction) or could be greater than 350           *
 *  after adding a positive deg_accum value.                                *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    char full_swp           flag that indicates if 1 value is being       *
 *                            requested or all values for the record (for   *
 *                            a scalar parameter only)                      *
 *    void *UDF               ptr to the memory location for the structure  *
 *                            that holds returned data values (read_drec)   *
 *                                                                          *
 *  USAGE                                                                   *
 *    ir_fill_angle_array (full_swp, sensor, UDF)                           *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    ir_sample_time ()       returns the time associated with a single     *
 *                            data sample (row, column)                     *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info     structure that holds information concerning   *
 *        ginfo               the experiment that is being processed        *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data         structure that holds all of the currently     *
 *       *ExDa                returned data values to be processed          *
 *    struct experiment_info  a pointer to the structure that holds         *
 *          *ex               specific experiment information               *
 *    struct ptr_rec *ptr     a pointer to the structure which holds all    *
 *                            pointers to the header and data for the       *
 *                            experiment of interest                        *
 *    register float *s_ptr   ptr to the starting azimuthal angle array     *
 *    register float *e_ptr   ptr to the ending azimuthal angle array       *
 *    register float *f_end   loop termination variable                     *
 *    register float *fxang   fixed angle - non-rotating instrument         *
 *    register float *s1      fast short pointer looper                     *
 *    float deg_per_msec      number of degrees per millisecond             *
 *    float deg_accum         number of degrees covered by the acquisition  *
 *                            of a single measurement                       *
 *    float time_accum        acquisition time of a single measurement      *
 *    float C1                time correction constant                      *
 *    long btime_ms           start time of the measurement in milliseconds *
 *    long etime_ms           end time of the measurement in milliseconds   *
 *    long dummy_ns           dummy placeholder for nanoseconds element     *
 *    long base_time          time correction constant                      *
 *    unsigned short max_ele  the number of angle values to be returned     *
 *    unsigned short time_row the matrix row being processed                *
 *    short base_swp_off      sweep step offset for first element in column *
 *    char decrement_bday     flag indicating if start day needs to be      *
 *                            modified (if btime ends up negative)          *
 *    char decrement_eday     flag indicating if end day needs to be        *
 *                            modified (if etime ends up negative)          *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

void CorForConing (ByTe_1 full_swp, void *UDF)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct ptr_rec *ptr;
  register ReaL_4 *s_ptr, *e_ptr, *f_end, fxang;
  register ByTe_2 *s1;
  ReaL_4 deg_per_msec, deg_accum, time_accum, C1;
  ReaL_4 bTime;
  ByTe_4 btime_ms, etime_ms, dummy_ns, base_time;
  u_ByTe_2 max_ele, time_row;
  ByTe_2 base_swp_off;
  ByTe_1 decrement_bday, decrement_eday;

  ReaL_4 cP, sP, cT, sT, cW, sW;

  /***********************************************************************/
  /* Set a pointer to the structure which holds all pointers for header  */
  /* and data information for thxperiment currently being processed.     */
  /***********************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  ExDa = (struct idf_data *) UDF;

  /************************************************************************/
  /* For a full swp, max_ele is not just set to N_SAMPLE because the user */
  /* could have asked for one at a time and then changed to a full swp to */
  /* get the remainder of the elements.  We use time_row since we need to */
  /* pick up the rest of the elements in that column being processed (for */
  /* both SEN_MODE = 0 and 4) and time_row tells us how many have already */
  /* been processed for that sensor.                                      */
  /************************************************************************/

  max_ele = (ex->smp_id == 2 && !full_swp) ? 
                                       	1 : *ptr->N_SAMPLE - ptr->time_row;

  /************************************************************************/
  /*  Set up sines and cosines                                            */
  /************************************************************************/

  /************************************************************************/
  /*  The instrument is not spinning.                                     */
  /************************************************************************/

  if (*ptr->SPIN == 0)
   {
     s_ptr = ExDa->start_az;
     e_ptr = ExDa->stop_az;
     f_end = ExDa->start_az + max_ele;
     fxang = *(ptr->SUN_SEN) / 100.0;
     for (; s_ptr < f_end;)
      {
        *s_ptr += fxang;
        *e_ptr++ = *s_ptr++;
      }
   }
  else
   {
     time_row = (u_ByTe_2) ptr->time_row;

     /*****************************************************************/
     /*  When computing the angles, take into account fractions of    */
     /*  milliseconds (nanoseconds components).                       */
     /*****************************************************************/

     time_accum = ex->accum_ms + ex->accum_ns * 1.0e-6 + ex->lat_ms + ex->lat_ns*1.0e-6;
     deg_per_msec = 360.0/ *ptr->SPIN;
     deg_accum = time_accum * deg_per_msec;

     /*******************************************************************/
     /*  Determine the starting azimuthal angle for each data sample.   */
     /*                                                                 */
     /* If time advances down the column and the column timing is       */
     /* sequential, use DA_METHOD to determine start time of sample.    */
     /* In all cases, only the millisecond time component is of used.   */
     /*                                                                 */
     /* The formula for the angle is (Tsam - Tsunsen) * 1/Spin * 360    */
     /* which is added to the angle offset for the sensor.  This angle  */
     /* offset has already been preloaded into the start angle array    */ 
     /*******************************************************************/

     s_ptr = ExDa->start_az;
     f_end = ExDa->start_az + max_ele;

     base_time = ex->btime_ms % 86400000;
     C1 = base_time - *ptr->SUN_SEN;
     if (ex->sen_mode == 0 || ex->sen_mode == 2)
     {
        switch (ex->da_method)
        {
           case 0:
              bTime = time_row * time_accum;
              for ( ; s_ptr < f_end; )
              {
                 *s_ptr++ += (bTime + C1) * deg_per_msec;
                 bTime += time_accum;
              }
              break;
           case 1:
              s1 = ptr->SAMP_INDEX + time_row;
              for ( ; s_ptr < f_end; ++s1)
              {
                 bTime = *s1 * time_accum;
                 *s_ptr++ += (bTime + C1) * deg_per_msec;
              }
              break;
           case 2:
           case 3:
              s1 = ptr->SAMP_INDEX + time_row;
              base_swp_off = *ptr->SAMP_INDEX;
              if (*ptr->SAMP_INDEX > *(ptr->SAMP_INDEX + 1))
              {
                 for ( ; s_ptr < f_end; ++s1)
                 {
                    bTime = (base_swp_off - *s1) * time_accum;
                    *s_ptr++ += (bTime + C1) * deg_per_msec;
                 }
              }
              else
              {
                 for ( ; s_ptr < f_end; ++s1)
                 {
                    bTime = (*s1 - base_swp_off) * time_accum;
                    *s_ptr++ += (bTime + C1) * deg_per_msec;
                 }
              }
              break;
        }  
     }
     else
     {
         /****************************************************************/
         /*  If time advances down the column and the column timing is   */
         /*  parallel, use start time of first element being returned    */
         /****************************************************************/

         if (ex->sen_mode == 1 || ex->sen_mode == 3)
         {
            bTime = base_time;
            for ( ; s_ptr < f_end; )
                *s_ptr++ += C1 * deg_per_msec;
         }
         else
         {

         /****************************************************************/
         /*  Determine the time for the sample being processed.  Day     */
         /*  value is not used so no need to check decrement_day flag.   */
         /****************************************************************/

             for ( ; s_ptr < f_end; )
             {
                 ir_sample_time (max_ele, UDF, time_row, ptr->time_col, 
                                 &btime_ms, &dummy_ns, &etime_ms, &dummy_ns,
                                 &decrement_bday, &decrement_eday);
                 *s_ptr++ += (btime_ms - *ptr->SUN_SEN) * deg_per_msec;
                 ++time_row;
             }
          }
        }

      /****************************************************************/
      /*  Ensure that angle goes from 0.0 to 359.999...  For negative */
      /*  angles, return the compliment angle (-90 = 270).            */
      /****************************************************************/

        s_ptr = ExDa->start_az;
        f_end = ExDa->start_az + max_ele;
        for ( ; s_ptr < f_end; ++s_ptr)
        {
            if (*s_ptr < 0.0)
            {
               while (*s_ptr < 0.0)
                  *s_ptr += 360.0;
            }
            else
            {
               while (*s_ptr >= 360.0)
                  *s_ptr -= 360.0;
            }
        } 

     /*******************************************************************/
     /*  For this case of DA_METHOD, the stopping angle value is equal  */
     /*  to the starting angle of the NEXT sample.  For the last sample */
     /*  simply add the degrees covered by the accumulation time of the */
     /*  sample to the starting azimuthal angle for the last sample.    */
     /*******************************************************************/

        if ((ex->sen_mode == 0 || ex->sen_mode == 2) && ex->da_method == 3)
         {
           s_ptr = ExDa->start_az + 1;
           e_ptr = ExDa->stop_az;
           f_end = ExDa->stop_az + max_ele - 1;
           for (; e_ptr < f_end;)
             *e_ptr++ = *s_ptr++;
           *e_ptr = *(--s_ptr) + deg_accum;
         }
   
     /*******************************************************************/
     /*  Simply add the degrees covered by the accumulation time of the */
     /*  sample to each starting azimuthal angle value.                 */
     /*******************************************************************/

       else
        {
          s_ptr = ExDa->start_az;
          e_ptr = ExDa->stop_az;
          f_end = ExDa->stop_az + max_ele;
          for (; e_ptr < f_end;)
            *e_ptr++ = *s_ptr++ + deg_accum;
        }
    }

    ExDa->num_angle = max_ele;
}
