#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <memory.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                            IR_SENSOR_COMBO SUBROUTINE                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine determines the number of different combinations possible   *
 *  for the tables used by the sensors that are being utilized.  The TBL_OFF  *
 *  values for all tables and the CRIT_STATUS values are used and compared    *
 *  to see how many unique combinations of these tables are necessary instead *
 *  of allocating space for every table for the sensors that are being        *
 *  utilized.  Once the number of combinations have been determined, space is *
 *  allocated for the structure(s) that hold the information for the combos.  *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_sensor_combo ()                                                  *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                the size of the specified object in bytes      *
 *    malloc()                 allocates memory                               *
 *    free()                   frees allocated memory                         *
 *    memset ()                memory initialization routine                  *
 *    read_idf()               reads information from the IDF file            *
 *    ir_count_combo()         determines number of different combinations    *
 *    ir_get_sensor_tables()   reads table offset values from the IDF file    *
 *    ir_init_sensor_ptr()     initializes the sensor_ptr structures which    *
 *                             hold table information for the combinations    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    unsigned int B           the number of bytes to allocate and offsets    *
 *    long *tOffs              an array holding the table offsets for each    *
 *                             table                                          *
 *    char *crB                an array holding the critical action bytes     *
 *                             for table                                      *
 *    long rV                  holds value returned by called module          *
 *    long crit_act_sz         the critical action table size for each table  *
 *    short i                  looping variable                               *
 *    short num_combo          the number of unique table combinations needed *
 *                             for all sensors for the virtual instrument     *
 *                             being processed                                *
 *    char *oInfo              an array of 1 flag per table which indicates   *
 *                             if the table has different offset values       *
 *    char *cInfo              an array of 1 flag per table which indicates   *
 *                             if there are critical action values present    *
 *    char FirsT               flag indicating the first time this module is  *
 *                             called for the combination being processed     *
 *    void *tMem               temporary memory holding table information     *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_sensor_combo (void)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   size_t B;

   register ByTe_1 *c1, *c2;
   register ByTe_2 *s1;

   ByTe_4 rV, crit_act_sz, *tOffs;
   ByTe_4 sLp, sCp, sC, sL, nSen, nTbl;
   ByTe_2 i, num_combo;
   ByTe_1 *oInfo, *cInfo, *crB;
   ByTe_1 FirsT;
   void *tMem;

  /************************************************************************/
  /* Get base storages sizes up front.                                    */
  /************************************************************************/

   sLp = sizeof(ByTe_4 *);
   sCp = sizeof(ByTe_1 *);
   sL  = sizeof(ByTe_4);
   sC  = sizeof(ByTe_1);

  /************************************************************************/
  /*  Set pointer to exeriment information and get number of sensors and  */
  /*  number of tables defined in the VIDF.                               */
  /************************************************************************/

   ex = ginfo.expt;
   nSen = ex->num_sensor;
   nTbl = ex->num_tbls;

  /************************************************************************/
  /*  Malloc space for the sensor index array.  This holds the table      */
  /*  combination number that each sensor utilizes.                       */
  /************************************************************************/

   if (ex->bmem.base_tbl_index != NO_MEMORY) {
      free (ex->bmem.base_tbl_index);
      ex->bmem.base_tbl_index = NO_MEMORY;
   }

   B = nSen * sizeof(ByTe_2);
   if (B <= 0) { return (SCOM_INDEX_MALLOC); }

   if ((ex->bmem.base_tbl_index = malloc (B)) == NO_MEMORY)
      return (SCOM_INDEX_MALLOC);
   ex->index_sen_tbl = (ByTe_2 *)ex->bmem.base_tbl_index;

  /************************************************************************/
  /*  Initialize sensors combination to 0 unless the sensor is not being  */
  /*  used in which case set it to -1.                                    */
  /************************************************************************/

   s1 = ex->index_sen_tbl;
   if (ex->sensors_needed != NO_MEMORY) {
      c1 = ex->sensors_needed;
      for (i = 0; i < nSen; ++i, ++s1, ++c1) {
	 *s1 = (*c1 != 0) ? 0 : -1; 
      }
   } else { for (i = 0; i < nSen; ++i, ++s1) { *s1 = 0; } }

  /************************************************************************/
  /* Not much to do if there are no tables.  Only one combo is possible   */
  /************************************************************************/

   if (nTbl == 0) {
      ex->num_combo = 0;
      return (ALL_OKAY);
   }

  /************************************************************************/
  /*  Allocate temporary memory to hold:                                  */
  /*    1. The table offsets for each table                               */
  /*    2. The critical status bytes for each table (if exists)           */
  /*    3. Flag per table indicating if it has associated critical action */
  /*       information and different critical status bytes                */
  /*    4. Flag per table indicating if table offsets differ              */
  /************************************************************************/

   B = nTbl * nSen * (sL + sC) + 2 * nTbl * sC;
   if ((tMem = malloc (B)) == NO_MEMORY) { return (SCOM_TBL_MALLOC); }
   memset (tMem, '0', B);

  /**************************************************************************/
  /*  Set up the various pointers.                                          */
  /**************************************************************************/

   tOffs = (ByTe_4 *)tMem;
   B = nTbl * nSen * sL;
   crB = (ByTe_1 *)((ByTe_1 *)tMem + B);
   B += nTbl * nSen * sC;
   oInfo = (ByTe_1 *) ((ByTe_1 *)tMem + B);
   B += nTbl * sC;
   cInfo = (ByTe_1 *) ((ByTe_1 *)tMem + B);

  /**************************************************************************/
  /*  Find out how many tables associated with this VIDF have a critical    */
  /*  action associated with them and at the same time flag those tables    */
  /*  which do.                                                             */
  /**************************************************************************/

   ex->crit_action = 0;
   c1 = cInfo;
   c2 = oInfo;
   for (i = 0; i < nTbl; ++i, ++c1, ++c2) {
      *c2 = 1;
      rV = read_idf (ex->data_key, ex->exten, ex->version,
                        (ByTe_1 *) &crit_act_sz, _CritActSZ, i, 0L, 1);
      if (rV < 0) {
         if (tMem != NO_MEMORY) { free (tMem); }
         switch (rV) {
            case IDF_NOT_FOUND:  return (SCOM_IDF_NOT_FOUND);
            case IDF_MANY_BYTES: return (SCOM_IDF_MANY_BYTES);
            case IDF_TBL_NUM:    return (SCOM_IDF_TBL_NUM);
            case IDF_CON_NUM:    return (SCOM_IDF_CON_NUM);
            case IDF_NO_ENTRY:   return (SCOM_IDF_NO_ENTRY);
            default:             return ((ByTe_2) rV);
         }
      }
      *c1 = (crit_act_sz == 0) ? 0 : 1;
      ex->crit_action += *c1;
   }

  /**************************************************************************/
  /*  Retrieve table information from the IDF file.                         */
  /**************************************************************************/

   rV = ir_get_sensor_tables (oInfo, cInfo, tOffs, crB);
   if (rV != ALL_OKAY) { return ((ByTe_2)rV); }

  /*************************************************************************/
  /*  Determine the number of unique combinations necessary to process all */
  /*  requested sensors for the virtual instrument being processed.        */
  /*************************************************************************/

   num_combo = ir_count_combo (oInfo, cInfo, tOffs, crB);
   if (num_combo < 0) { return (num_combo); }

  /*************************************************************************/
  /*  Malloc the space to hold all the combinations requested.  Initialize */
  /*  the structure(s) which hold the unique combination of offsets.       */
  /*************************************************************************/

   if (ex->bmem.base_sen_ptr != NO_MEMORY) {
      free (ex->bmem.base_sen_ptr);
      ex->bmem.base_sen_ptr = NO_MEMORY;
      FirsT = 0;
   } else { FirsT = 1; }

   B = sizeof (struct sensor_tables) * num_combo;
   if (B <= 0) { return (SCOM_SEN_PTR_MALLOC); }

   if ((ex->bmem.base_sen_ptr = malloc (B)) == NO_MEMORY)
      return (SCOM_SEN_PTR_MALLOC);
   ex->sen_tbl_ptr = (struct sensor_tables *) ex->bmem.base_sen_ptr;
   ex->num_combo = num_combo;

  /*********************************************************************/
  /*  Reset chk_crit values since ir_get_sensor_tables() may modify    */
  /*  the values in preparation for the call to ir_count_combo ().     */
  /*********************************************************************/

   c1 = cInfo;
   for (i = 0; i < nTbl; ++i, ++c1) {
      rV = read_idf (ex->data_key, ex->exten, ex->version,
                         (char *) &crit_act_sz, _CritActSZ, i, 0L, 1);
      if (rV < 0) {
        if (tMem != NO_MEMORY) { free (tMem); }
         switch (rV) {
            case IDF_NOT_FOUND:  return (SCOM_IDF_NOT_FOUND);
            case IDF_MANY_BYTES: return (SCOM_IDF_MANY_BYTES);
            case IDF_TBL_NUM:    return (SCOM_IDF_TBL_NUM);
            case IDF_CON_NUM:    return (SCOM_IDF_CON_NUM);
            case IDF_NO_ENTRY:   return (SCOM_IDF_NO_ENTRY);
            default:             return ((ByTe_2) rV);
         }
      }

      *c1 = (crit_act_sz == 0) ? 0 : 1;
   }

   rV = ir_init_sensor_ptr (num_combo, crB, FirsT, cInfo);
   if (rV != ALL_OKAY) { return ((ByTe_2) rV); }

 /***************************************************************************/
 /*  Free memory that does not need to be saved (used for holding purposes).*/
 /***************************************************************************/

   if (tMem != NO_MEMORY) { free (tMem); }

   return (ALL_OKAY);
}
