#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           SELECT_SENSOR SUBROUTINE                         *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to reset the flags that indicate which of the    *
 *  sensors for each virtual instrument are going to be processed.  This      *
 *  routine was developed to help conserve allocated memory associated with   *
 *  sensor information. The routine FILE_OPEN sets the flags to indicate that *
 *  all sensors for a given virtual instrument are to be processed.  However, *
 *  this may not be true, and thus unnecessary sensor information structures  *
 *  are allocated. By calling this routine once for each sensor that is to be *
 *  processed, we can conserve memory.                                        *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long data_key       key which uniquely identifies the data set *
 *                                 being processed                            *
 *    char *exten                  filename extension for the data to be used *
 *    unsigned short vnum          version number to be associated with this  *
 *                                 combination (allows for multiple opens)    *
 *    short sensor                 the sensor needed for processing the data  *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = select_sensor (data_key, exten, vnum, sensor)                       *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_locate_ex()               determines if requested combination has    *
 *                                 already been processed and points to the   *
 *                                 correct structure allocated for the combo  *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    register char *cptr          pointer to the sensor flags                *
 *    register char *stop_cptr     loop termination variable                  *
 *    short rV                     the value returned by the called routine   *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 select_sensor (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                      ByTe_2 sensor)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;

   register ByTe_1 *c1, *stop_c1;
   ByTe_2 rV;
   ByTe_4 B;

   void *tP;

  /***********************************************************************/
  /*  Set a pointer to the structure which holds all information that is */
  /*  pertinent to the experiment currently being processed.  Since      */
  /*  ir_locate_ex() is called by more than one routine, return an error */
  /*  code that indicates which calling routine resulted in the error.   */
  /*  Since we pass a 0 for the last parameter, the only possible error  */
  /*  is that the requested combo was not found among processed combos.  */
  /***********************************************************************/
  
   if (!ginfo.called_locate) {
      rV = ir_locate_ex (data_key, exten, vnum, 0);
      if (rV != ALL_OKAY) { return (SEL_SEN_NOT_FOUND); }
   }

   ex = ginfo.expt;

  /***********************************************************************/
  /*  If the sensor_needed array has not been created then malloc memory */
  /*  for it and all sensor flags to zero.                               */
  /***********************************************************************/

    if (ex->sensors_needed == NO_MEMORY) {
       B = ex->num_sensor * sizeof (ByTe_1);
       if (B <= 0) { return (ALL_FLAG_MALLOC); }
       if ((tP = malloc (B)) == NO_MEMORY) { return (ALL_FLAG_MALLOC); }
       ex->sensors_needed = (ByTe_1 *) tP;

       c1 = ex->sensors_needed;
       stop_c1 = ex->sensors_needed + ex->num_sensor;
      for ( ; c1 < stop_c1;) { *c1++ = 0; }
    }

  /**************************************************************************/
  /*  Set the flag to indicate that the sensor passed in as a parameter     */
  /*  is needed in the processing of the data.                              */
  /**************************************************************************/

   *(ex->sensors_needed + sensor) = 1;

   return (ALL_OKAY);
}
