#include <memory.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include "OpSySInD.h"
#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"
#include "Server.h"
#include "libCfg.h"
#define NTRIES 100

/*****************************************************************************
 *                                                                           *
 *                        IR_REALTIME_DATA_FILES SUBROUTINE                  *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This module is called to determine if the realtime data files for the  *
 *  requested data set are available for usage.  All realtime data files are *
 *  kept in a specific directory.  In order to integrate with the listener   *
 *  cleanly, more than one attempt is made to open the data and/or header    *
 *  file in case processing is a little ahead of the opening of the new file *
 *  by the listener.  In addition, a check is made to ensure that the same   *
 *  file is not utilized more than one (listener has not moved the old file  *
 *  yet).                                                                    * 
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    short btime_yr            the start time requested (year component)    *
 *    short btime_day           the start time requested (day component)     *
 *    long btime_sec            the start time requested (seconds component) *
 *    long btime_nsec           the start time requested (nanoseconds)       *
 *    short etime_yr            the stop time requested (year component)     *
 *    short etime_day           the stop time requested (day component)      *
 *    long etime_sec            the stop time requested (seconds component)  *
 *    long etime_nsec           the stop time requested (nanoseconds)        *
 *    unsigned long data_key    key which uniquely identifies the data set   *
 *                              being processed                              *
 *    char *exten               filename extension for the data to be used   *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_realtime_data_files (btime_yr, btime_day, btime_sec, btime_nsec,*
 *                             etime_yr, etime_day, etime_sec, etime_nsec,   *
 *                             data_key, exten)                              *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                  the size of the specified object in bytes   *
 *    strcpy()                   copies a string to another string variable  *
 *    open()                     opens the file requested                    *
 *    fstat()                    gets directory information for a file       *
 *    close()                    closes the file associated with the file    *
 *                               descriptor specified                        *
 *    setitimer()                sets the alarm timer                        *
 *    select()                   used to provide a timeout for retry         *
 *    pbr()                      the playback data request routine           *
 *    sigvec ()                  software signal facilities                  *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info        structure that holds information concerning *
 *         ginfo                 the experiment that is being processed      *
 *    char plot_package          flag which indicates if the plotting or menu*
 *                               program made the call (will not be changed  *
 *                               from the default of plotting for the generic*
 *                               user)                                       *
 *    LinkList Projects          pointer to current configuration info       *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct experiment_info *ex a pointer to the structure that holds       *
 *                               specific experiment information             *
 *    struct stat file_info      directory information for a file            *
 *    struct itimerval dtold     alarm timer control structure               *
 *    struct itimerval dtnew     alarm timer control structure               *
 *    struct sigvec sigstate     software signal structure                   *
 *    struct timeval t           time value structure                        *
 *    struct timeval tset        time value structure                        *
 *    int rval1, rval2, rval3    holds value returned by the routine PBR()   *
 *    int masked                 flag checking for the presence of a timer   *
 *    short ret_code             return code                                 *
 *    char hname[]               the full path name of the header file       *
 *    char dname[]               the full path name of the data file         *
 *    char vname[]               the full path name of the VIDF file         *
 *    char try                   number of tries on a file open              *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 ****************************************************************************/

ByTe_2 ir_realtime_data_files (ByTe_2 btime_yr, ByTe_2 btime_day, 
                               ByTe_4 btime_sec, ByTe_4 btime_nsec, 
                               ByTe_2 etime_yr, ByTe_2 etime_day, 
                               ByTe_4 etime_sec, ByTe_4 etime_nsec, 
                               u_ByTe_4 data_key, ByTe_1 *exten)
{
   extern struct general_info ginfo;
   extern LinkList Projects;
   extern ByTe_1 plot_package;

   StrHier DSrc;
   Time_t btm, etm;
   struct experiment_info *ex;
   struct stat file_info;
   struct itimerval dtold, dtnew;
   struct sigaction sigstate;
   struct timeval t, tset;
   int rval1, rval2, rval3, masked;
   ByTe_4 dTm[3], vTm[3];
   ByTe_2 ret_code, params[5];
   ByTe_1 hname[NAME_SIZE], dname[NAME_SIZE], vname[NAME_SIZE], try;

   /*************************************************************************/
   /*  If  data files have already been opened, no need to open them again. */
   /*************************************************************************/

   ex = ginfo.expt;
   if (ex->fdh != -1)
     return (ALL_OKAY);

   key_to_fields (data_key, params);
   DSrc = SourceByNum (Projects, (ByTe_1 *) params[0], (ByTe_1 *) params[1],
                       (ByTe_1 *) params[2], (ByTe_1 *) params[3], 
                       (ByTe_1 *) params[4], (ByTe_1 *) 0);
   if (DSrc == NULL)
     return (RTIME_BAD_SRC);

   btm.yr = btime_yr;
   btm.day = btime_day;
   btm.msec = (btime_sec * 1000) + (btime_nsec / 1000000);

   etm.yr = etime_yr;
   etm.day = etime_day;
   etm.msec = (etime_sec * 1000) + (etime_nsec / 1000000);

   /*************************************************************************/
   /*  When plotting the data, all three files must be online and ready to  */
   /*  go.  For now, the database only works in terms of seconds, so the    */
   /*  nanoseconds parameter is ignored.  This may change later.            */
   /*************************************************************************/
    
   if (plot_package)
    {
      rval1 = pbr (btm, etm, DSrc, _HEADER, exten, hname, dTm);
      rval2 = pbr (btm, etm, DSrc, _DATA, exten, dname, dTm);
      rval3 = pbr (btm, etm, DSrc, _VIDF, exten, vname, vTm);

      if (rval1 < 0)
        return ((ByTe_2) rval1);
      else if  (rval2 < 0)
        return ((ByTe_2) rval2);
      else if  (rval3 < 0)
        return ((ByTe_2) rval3);

      ret_code = ALL_OKAY;
    }

   /*************************************************************************/
   /*  For the menus, only the VIDF file needs to be online.                */
   /*************************************************************************/
    
   else
    { 
      rval1 = pbr (btm, etm, DSrc, _VIDF, exten, vname, vTm);
      if (rval1 != ALL_OKAY)
        return ((ByTe_2) rval1);
      ret_code = ALL_OKAY;
    }

   /*************************************************************************/
   /*  If there is a signal handler present, disable the alarm timer. Save  */
   /*  any old timer setting in dtold.                                      */
   /*  Set dtold.it_value to dtold.it_interval to make sure the value       */
   /*  field is non-zero.                                                   */
   /*  Set tset to .5 seconds.                                              */
   /*************************************************************************/

   masked = 0;
   sigaction(SIGALRM, NULL, &sigstate);
   if (sigstate.sa_handler != SIG_DFL)
    {
      masked = 1;

      memset((ByTe_1 *)&dtnew, 0, (size_t) (sizeof(dtnew)));
      memset((ByTe_1 *)&dtold, 0, (size_t) (sizeof(dtold)));

      setitimer(ITIMER_REAL, &dtnew, &dtold);
      dtold.it_value = dtold.it_interval;
    }

   tset.tv_sec  = 0;
   tset.tv_usec = 500000;

  /**************************************************************************/
  /*  Check to make sure that all applicable files can be opened.  Save the */
  /*  names of the files opened. Only the plotting package needs the header */
  /*  and data files to be on-line.  The menus only needs the IDF file.     */
  /**************************************************************************/
 
   if (plot_package)
    {
      strcpy (ex->header_name, hname);

     /***********************************************************************/
     /*  The program will try to open the named file many times if either   */
     /*  the file was not there or if the same file has been opened (timing */
     /*  between this program and listener).                                */
     /***********************************************************************/

      for (try = NTRIES ; try ; try--) {
         if ((ex->fdh = open (hname, O_RDONLY)) >= 0) {

#ifdef _CyGnUsC
            setmode (ex->fdh, O_BINARY);
#endif
#ifdef WIN32
      _setmode (ex->fdh, O_BINARY);
#endif

            fstat (ex->fdh, &file_info);

            /*****************************************************************/
            /*  If the same file has been opened, try again.                 */
            /*****************************************************************/

            if (file_info.st_ino == ex->inh) {
               close (ex->fdh);
               ex->fdh = -1;
            } else {
               ex->inh = file_info.st_ino;
               break;
            }
         }

         /*****************************************************************/
         /*  Call select to wait .5 seconds.                              */
         /*  Wait for the timer to expire before trying again.            */
         /*****************************************************************/

         t = tset;
         select(0, (fd_set *) 0x0, (fd_set *) 0x0, (fd_set *) 0x0, &t);
       }

      /*****************************************************************/
      /*  Error opening the designated header file.                    */
      /*****************************************************************/

      if (!try)
      {
        if (masked) setitimer(ITIMER_REAL, &dtold, (struct itimerval *) 0x0);
        return (RTIME_NO_HEADER);
      }

      strcpy (ex->data_name, dname);

      /************************************************************************/
      /*  The program will try to open the named file many times if either    */
      /*  the file was not there or if the same file has been opened (timing  */
      /*  between this program and listener).                                 */
      /************************************************************************/

      for (try = NTRIES ; try ; try--)
       {
         if ((ex->fdd = open (dname, O_RDONLY)) >= 0)
          {

#ifdef _CyGnUsC
            setmode (ex->fdd, O_BINARY);
#endif
#ifdef WIN32
      _setmode (ex->fdd, O_BINARY);
#endif

            fstat (ex->fdd, &file_info);

            /*****************************************************************/
            /*  If the same file has been opened, try again.                 */
            /*****************************************************************/

            if (file_info.st_ino == ex->ind) {
               close (ex->fdd);
               ex->fdd = -1;
            } else {
               ex->ind = file_info.st_ino;
               break;
            }
          }

         /*****************************************************************/
         /*  Call select to wait .5 seconds.                              */
         /*  Wait for the timer to expire before trying again.            */
         /*****************************************************************/

         t = tset;
         select(0, (fd_set *) 0x0, (fd_set *) 0x0, (fd_set *) 0x0, &t);
       }

      /*****************************************************************/
      /*  Error opening the designated data file.                      */
      /*****************************************************************/

      if (!try)
      {
        if (masked) setitimer(ITIMER_REAL, &dtold, (struct itimerval *) 0x0);
        return (RTIME_NO_DATA);
      }
    }

   /*****************************************************************/
   /*  No need to retry on the VIDF file since the listener is not  */
   /*  manipulating these files (only the header and data files).   */
   /*****************************************************************/

   strcpy (ex->idf_name, vname);
   if ((ex->fdi = open (vname, O_RDONLY)) < 0)
   {
      if (masked) setitimer(ITIMER_REAL, &dtold, (struct itimerval *) 0x0);
      return (RTIME_NO_VIDF);
   }

#ifdef _CyGnUsC
   setmode (ex->fdi, O_BINARY);
#endif
#ifdef WIN32
      _setmode (ex->fdi, O_BINARY);
#endif


   if (masked) setitimer(ITIMER_REAL, &dtold, (struct itimerval *) 0x0);

   return (ret_code);
}
