#include "ret_codes.h"
#include "util_str.h"
#include "libbase_udf.h"
 
/*******************************************************************************
 *                                                                             *
 *                       IR_PITCH_LOS_NEXT_FILE SUBROUTINE                     *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called whenever the LOS_STATUS or NEXT_FILE_STATUS code  *
 *  is returned by read_drec() for pitch angle component sensors.  The         *
 *  existing data set is closed and the next set of IDFS files are retrieved   *
 *  so that the processing of the algorithm can continue.  If the data files   *
 *  are not online, the code does not attempt to promote the data; an error    *
 *  code will be returned if the attempt to open the next IDFS data set cannot *
 *  be accomplished.  This code works on the assumption of playback mode.  It  *
 *  does not work for real-time.                                               *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           *
 *    short which_comp             index specifying which of the 3 components  *
 *                                 is being processed                          *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_pitch_los_next_file (pa_ptr, which_comp)                          *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    ir_locate_ex()               determines if requested combination has     *
 *                                 already been pand points to the             *
 *                                 correct structure allocated for the combo   *
 *    file_pos()                   positions the data/header files for the     *
 *                                 data source at the desired start time       *
 *    reset_experiment_info ()     closes the opened IDFS data set and gets    *
 *                                 the next set of IDFS data files             *
 *    adjust_time ()               corrects time elements if too many seconds  *
 *                                 (day crossing) or too many days (year       *
 *                                 crossing)                                   *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info          structure holding information concerning    *
 *        ginfo                    the experiment that is being processed      *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info       a pointer to the structure that holds       *
 *          *ex                    specific experiment information             *
 *    struct idf_data *PA_DATA     structure holding the data for the pitch    *
 *                                 angle component being processed             *
 *    long new_start_sec,          the time of day being requested for the     *
 *       new_start_nsec            next set of IDFS data files                 *
 *    long time_nsec               the nanoseconds component for the time of   *
 *                                 day, which is modified by 1 millisecond     *
 *    long time_diff               time value for calculating new nanosecond   *
 *    short ret_val                holds value returned by called module       *
 *    short new_year, new_day      year and day time values for next data file *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_pitch_los_next_file (struct pitch_info *pa_ptr, ByTe_2 which_comp)
{
   extern struct general_info ginfo;
 
   struct experiment_info *ex;
   struct idf_data *PA_DATA;
   ByTe_4 new_start_sec, new_start_nsec, time_nsec, time_diff;
   ByTe_2 ret_val, new_year, new_day;

   /*********************************************************************/
   /*  Update the start time to reflect when processing stopped for the */
   /*  current data file.  An attempt is made to get the next data file */
   /*  (if one exists) to continue processing.  Since the database      */
   /*  resolution is in terms of milliseconds, add 1 millisecond to     */
   /*  ensure that the next data file, not the current data file, will  */
   /*  be retrieved from the database.                                  */
   /*********************************************************************/

   PA_DATA = (struct idf_data *) pa_ptr->idf_data_ptr[which_comp];
   new_year = PA_DATA->eyear;
   new_day = PA_DATA->eday;
   time_nsec = PA_DATA->enano + 1000000;
   new_start_sec = (PA_DATA->emilli + (time_nsec / 1000000)) / 1000;
 
   time_diff = PA_DATA->emilli - (new_start_sec * 1000);
   time_nsec += time_diff * 1000000;
   new_start_nsec = time_nsec % 1000000000;
   adjust_time (&new_year, &new_day, &new_start_sec, 1);

   pa_ptr->file_status = reset_experiment_info (pa_ptr->data_key, pa_ptr->exten,
                                    pa_ptr->version, new_year,
                                    new_day, new_start_sec, new_start_nsec,
                                    pa_ptr->stop_year, pa_ptr->stop_day,
                                    pa_ptr->stop_milli, pa_ptr->stop_nano);

   /*************************************************************************/
   /*  Reset this element of the structure even if an error was encountered */
   /*************************************************************************/

   ret_val = ir_locate_ex (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version, 0);
   ex = ginfo.expt;

   if (pa_ptr->file_status != ALL_OKAY)
      return (pa_ptr->file_status);

   /*********************************************************************/
   /*  Position the data file at the requested start time.  Must be     */
   /*  called again if NEXT_FILE_STATUS or LOS_STATUS is encountered.   */
   /*********************************************************************/

   ret_val = file_pos (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version, 
                       pa_ptr->idf_data_ptr[which_comp], new_year, new_day,
                       new_start_sec, new_start_nsec, pa_ptr->stop_year,
                       pa_ptr->stop_day, pa_ptr->stop_milli, pa_ptr->stop_nano);
   if (ret_val != ALL_OKAY)
     return (ret_val);

   return (ALL_OKAY);
}
