#include <stdlib.h>
#include "util_str.h"
#include "gen_defs.h"
#include "libbase_udf.h"
#include "ret_codes.h"
#include "idf_defs.h"

/*******************************************************************************
 *                                                                             *
 *                           IR_MODE_TBL_SIZE  SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called in order to determine the total number of bytes   *
 *  that are needed to hold all integer table values for all mode tables for   *
 *  the mode in question.  No bytes are included for ascii tables since these  *
 *  tables are kept in a different structure from the integer table values.    *
 *  In the sizeof function, use float instead of long since, in the calling    *
 *  module, the malloced space is set to a floating point pointer.             *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    short mode                   index into the mode_tables structure        *
 *    unsigned int *bytes          the number of bytes to allocate             *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_mode_tbl_size (mode, &bytes)                                      *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    sizeof ()                    the size of the specified object in bytes   *
 *    abs()                        returns the absolute value of a number      *
 *    read_idf()                   reads information from the IDF file         *
 *    ir_mode_tbl_size_rval ()     returns an error code unique to this module *
 *                                 for each possible READ_IDF() error code     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    struct mode_tables *mptr     a pointer to the structure which holds the  *
 *                                 mode-dependent table information            *
 *    register short i             looping variable                            *
 *    long rval                    the value returned by the called routine    *
 *    short num_states             the number of states defined for the status *
 *                                 byte of interest                            *
 *    char which_tbl               the table number being processed            *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_mode_tbl_size (ByTe_2 mode, size_t *bytes)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct mode_tables *mptr;
   register ByTe_2 i;
   ByTe_4 rval;
   ByTe_2 num_states;
   ByTe_1 which_tbl;

   /*************************************************************************/
   /*  Point to the correct sensor_tables structure.                        */
   /*************************************************************************/

   ex = ginfo.expt;
   mptr = ex->mode_tbl_ptr + mode;

   /**********************************************************************/
   /*  Loop over all defined mode tables in order to determine how much  */
   /*  space to allocate for INTEGER table values.  Ascii strings are    */
   /*  saved in a different structure.                                   */
   /**********************************************************************/

   *bytes = 0;
   for (i = 0; i < mptr->num_tbls; ++i)
    {
      which_tbl = *(mptr->tbl_num + i);

      /*********************************************************************/
      /*  The table is an expanded look up table (no coefficients).        */
      /*********************************************************************/

      if (*(mptr->tbl_fmt + i) == 0)
       {
         /*****************************************************************/
         /*  This table is a function of raw status (mode) data.          */
         /*****************************************************************/
  
         rval = read_idf (ex->data_key, ex->exten, ex->version, 
                          (ByTe_1 *) &num_states, _StateS, 0, mode, 1);
         if (rval < 0)
           return (ir_mode_tbl_size_rval (rval));

         *(mptr->tbl_size + i) = num_states;
         *bytes += sizeof (ReaL_4) * num_states;
       }

      /********************************************************************/
      /*  The table holds coefficients (table is not in expanded form).   */
      /********************************************************************/

      else if (*(mptr->tbl_fmt + i) > 0)
       {
         /*****************************************************************/
         /*  This table is to be expanded.                                */
         /*****************************************************************/
   
         if (*(mptr->tbl_expand + i) == 1)
          {
            rval = read_idf (ex->data_key, ex->exten, ex->version, 
                             (ByTe_1 *) &num_states, _StateS, 0, mode, 1);
            if (rval < 0)
              return (ir_mode_tbl_size_rval (rval));

            *(mptr->tbl_size + i) = num_states;
            *bytes += sizeof (ReaL_4) * num_states;
          }

         /*****************************************************************/
         /*  The table is not to be expanded; therefore, the number of    */
         /*  bytes needed is based upon the number of coefficients.   The */
         /*  TBL_EXPAND flag is hard-coded to 0 for tbl_type 2 tables.    */
         /*****************************************************************/

         else if (*(mptr->tbl_expand + i) == 0)
          {
            if (*(ex->tbl_type + which_tbl) == 0) 
             {
               *(mptr->tbl_size + i) = *(mptr->tbl_fmt + i);
               *bytes += sizeof (ReaL_4) * *(mptr->tbl_fmt + i);
             }
            else
             {
               *(mptr->tbl_size + i) = *(mptr->tbl_fmt + i) * ex->swp_len;
               *bytes += sizeof (ReaL_4) * *(mptr->tbl_fmt + i) * ex->swp_len;
             }
          }
       }

      /****************************************************************/
      /*  There is no table defined.  Tbl_size will be zero.          */
      /****************************************************************/

      else if (*(mptr->tbl_fmt + i) < 0)
       {
         *(mptr->tbl_size + i) = 0;
         *(mptr->tbl_ptr + i) = NO_MEMORY;
         *bytes += 0;
       }
    }

   return (ALL_OKAY);
}

/*******************************************************************************
 *                                                                             *
 *                           IR_MODE_TBL_SIZE_RVAL SUBROUTINE                  *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine returns an error code unique to the MODE_TBL_SIZE module    *
 *  based upon the error code returned by the call to READ_IDF().              *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    long ret_val                the error code returned by READ_IDF()        *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_mode_tbl_size_rval (ret_val)                                      *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_mode_tbl_size_rval (ByTe_4 ret_val)
{
  switch (ret_val)
   {
      case IDF_NOT_FOUND:
        return (MODE_TBL_SZ_IDF_NOT_FOUND);
      case IDF_MANY_BYTES:
        return (MODE_TBL_SZ_IDF_MANY_BYTES);
      case IDF_TBL_NUM:
        return (MODE_TBL_SZ_IDF_TBL_NUM);
      case IDF_CON_NUM:
        return (MODE_TBL_SZ_IDF_CON_NUM);
      case IDF_NO_ENTRY:
        return (MODE_TBL_SZ_IDF_NO_ENTRY);
      default:
        return ((ByTe_2) ret_val);
   }
}
