#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_FIX_POINTERS SUBROUTINE                       *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to assign the pointers for those information     *
 *  elements whose size may vary from header to header.  Everytime a new      *
 *  header record is read, this routine is called to assign the pointers to   *
 *  the correct memory locations dynamically allocated for the header         *
 *  information.                                                              *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *UDF                    memory location for the structure that     *
 *                                 holds returned data values (read_drec)     *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_fix_pointers (UDF)                                               *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    ir_number_of_cal_elements () determines the no. of calibration elements *
 *                                 associated with the sample sequence values *
 *                                 and the sensor data                        *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure that holds information concern-  *
 *                                 ing the experiment that is being processed *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    struct ptr_rec *ptr          a pointer to the structure which holds     *
 *                                 all pointers to the header and data for    *
 *                                 the experiment of interest                 *
 *    long header_offset           index into the memory allocated for the    *
 *                                 header information                         *
 *    long PrefOff                 index into the memory allocated for the    *
 *                                 data information                           *
 *    long tnb                     the number of bytes to bypass in order to  *
 *                                 get past the sensor data                   *
 *    long bit_off                 the total number of bits that need to be   *
 *                                 bypassed in order to get to the cal. data  *
 *    short rV                     holds the value returned by called module  *
 *    short num_sample             the number of samples returned             *
 *    short n_sen                  the number of sensors defined              *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_fix_pointers (void *UDF)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct ptr_rec *ptr; 
   ByTe_4 header_offset, PrefOff, tnb, BitOff;
   ByTe_4 Cs, Ss, Ls, uSs, SBits, TotCol, SBytes;
   ByTe_2 rV, num_sample, n_sen;

  /*************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header   */
  /*  and data information for the experiment currently being processed.   */
  /*************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;
   Cs = sizeof (ByTe_1);
   Ss = sizeof (ByTe_2);
   uSs = sizeof (u_ByTe_2);
   Ls = sizeof (ByTe_4);

  /*************************************************************************/
  /*  Determine offset value up to SAMP_INDEX element in header record.    */
  /*  The header offset value must bypass the SAMP_INDEX array to get to   */
  /*  the next header element.                                             */
  /*************************************************************************/

   num_sample = *(ptr->N_SAMPLE);

   header_offset = 4 * Ss + Cs + sizeof (u_ByTe_1) + 4 * Ls + uSs;

   if (ex->smp_id == 3) { header_offset += 3 * uSs; }

   header_offset += (ex->smp_id == 2) ? Ss : Ss * num_sample; 
   ptr->SENSOR_INDEX = (ByTe_2 *)(ex->HEADER_MEM + header_offset);

  /*************************************************************************/
  /*  The SENSOR_INDEX element is an array of dimension N_SEN.             */
  /*************************************************************************/

   n_sen = *(ptr->N_SEN);
   header_offset += n_sen * Ss;
   ptr->DQUAL = (u_ByTe_1 *) (ex->HEADER_MEM + header_offset);

  /*************************************************************************/
  /*  The D_QUAL element is an array of dimension N_SEN.                   */
  /*************************************************************************/

  header_offset += n_sen * sizeof (u_ByTe_1);
  ptr->MODE_INDEX = (u_ByTe_1 *)(ex->HEADER_MEM + header_offset);

  /*************************************************************************/
  /*  Determine pointer locations for the calibration values within the    */
  /*  data matrix.  The code is placed here since the number of sensors    */
  /*  can change from read to read.                                        */
  /*************************************************************************/

   PrefOff = Ls * (4 +  ex->max_nss);
   rV = ir_number_of_cal_elements (UDF);
   if (rV != ALL_OKAY) { return (rV); }

  /*************************************************************************/
  /*  Set all calibration offsets now as well as updating the DATA offset  */
  /*  should that be needed                                                */
  /*************************************************************************/

  /*************************************************************************/
  /*  Byte_size indicates how many bits are used per data element within   */
  /*  each byte.  For 5, 6 and 7 bit data, there are pad bits so that      */
  /*  there is only one sample per byte (takes 8 bits).  For 3 bit data,   */
  /*  use 4 since the data is laid down as a 4-bit value with the most     */
  /*  significant bit set to zero.  The rest of the cases divide evenly    */
  /*  into the byte.                                                       */
  /*************************************************************************/

   if (ex->tdw_len <= 8) {
      SBits = (ex->tdw_len >= 5) ? 8 :(ex->tdw_len == 3) ? 4 : ex->tdw_len;
      SBytes = 1;
   } else {
      SBits = (ex->tdw_len > 8 && ex->tdw_len <= 16) ? 16 : 32;
      SBytes = SBits / 4;
   }

  /*************************************************************************/
  /* Determine the total number of bytes that need to be bypassed in order */
  /* to get to the calibration data.  It is assumed that the calibration   */
  /* data will always start on a byte boundary (bit 0).                    */
  /*************************************************************************/

   TotCol = (ex->smp_id != 3) ? *(ptr->N_SEN) : *(ptr->N_SEN) * *(ptr->N_COLS);
   BitOff = num_sample * TotCol * SBits;
   tnb = (BitOff / 8) + ex->AncPBytes * ptr->NumPreAnc + PrefOff ;
   if (BitOff % 8 != 0) { ++tnb; }

  /*************************************************************************/
  /*  If the data pointer has not been updated for preface cal data do it  */
  /*  at this time.  Data always begins on a full byte boundary            */
  /*************************************************************************/

   if (ptr->NumGenAnc == 0) {
      ptr->CAL_GENERAL = NO_MEMORY;
   } else { ptr->CAL_GENERAL = (u_ByTe_1 *) (ex->DATA_MEM + tnb); }

   tnb +=  ex->AncGBytes * ptr->NumGenAnc;

  /*************************************************************************/
  /*  If there are no cal. sets for the sample sequence values, the cal.   */
  /*  values for the data immediately follows the data for the sensors if  */
  /*  it exists; else, the cal. values for the sample sequence values      */
  /*  immediately follow the data for the sensors, which  in turn is       */
  /*  followed by the cal.  values for the sensor data, if it exists.      */
  /*************************************************************************/

   if (ptr->NumSenAnc == 0) {
      ptr->CAL_SENSOR = NO_MEMORY;
   } else { ptr->CAL_SENSOR = (u_ByTe_1 *)(ex->DATA_MEM + tnb); }
  
   return (ALL_OKAY);
}
