#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "user_defs.h"
#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                        IR_CONVERT_MODE_DATA SUBROUTINE                   *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called to convert the mode data for the requested     *
 *  mode into units by applying the specified tables and table operations   *
 *  in the specified order to achieve the desired output.  The order is     *
 *  implied by the table specification.                                     *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    short which_mode             the mode being processed                 *
 *    char num_tbls                number of tables to apply                *
 *    char *tbls_to_apply          array of tables to be applied            *
 *    short *tbl_oper              array of table operations                *
 *    float *DataOut               array in which data is returned in the   *
 *                                 converted units                          *
 *    void *UDF                    memory location ptr for structure that   *
 *                                 holds returned data values (read_drec)   *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = ir_convert_mode_data (which_mode, num_tbls, tbls_to_apply,        *
 *                           tbl_oper, &DataOut, UDF)                       *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    ir_transfer_data ()          converts raw integer telemetry to the    *
 *                                 raw units (integer or float)             *
 *    ir_idf_ex_func ()            performs higher order table operations   *
 *    IDF_OPER[]._math_drv ()      performs arithmetic operation defined    *
 *                                 for the table being processed            *
 *    CombineOp()                  performs arithmetic operation defined    *
 *                                 to combine the permanent and temporary   *
 *                                 accumulators                             * 
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info ginfo    structure holding information concerning *
 *                                 the experiment that is being processed   *
 *    struct def_oper IDF_OPER[]   array of pointers to the routines that   *
 *                                 perform the arithmetic operation defined *
 *    void *Tmp_Space              scratch space for various uses           *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa        structure holding all of the currently   *
 *                                 returned data values to be processed     *
 *    struct experiment_info *ex   a pointer to the structure that holds    *
 *                                 specific experiment information          *
 *    struct mode_tables *mptr     a pointer to the structure holding the   *
 *                                 mode-dependent table information         *
 *    register char J              looping variables                        *
 *    long data_start              offset into the data at which to start   *
 *                                 picking up data values                   *
 *    long num_data                the number of data values being returned *
 *    long hold_val                number of successive data values within  *
 *                                 a data set that one anc value is to be   *
 *                                 applied to                               *
 *    long num_ele_in_tbl          no. of elements in table being applied   *
 *    ByTe_2 BasOp                 basic table operation                    *
 *    ByTe_2 ExOp                  extended (after table) operation         *
 *    ByTe_2 BufOp                 buffer operation modifier                * 
 *    ByTe_2 PosOp                 positive operator                        * 
 *    char *mode                   ptr to the mode data from READ_DREC()    *
 *    char dtype                   format of the data - uns int, int, float *
 *    char fn_of_raw               flag indicating if table is a function   *
 *                                 of raw data or processed data            *
 *    char *DFrom                  ptr to the data values which the table   *
 *                                 values are to be applied to              *
 *    char tdw_len                 word-length of data item                 *
 *    char tbl_ind                 index to get to the table in question    *
 *    float *DataFrom              this is the input data array             *
 *    float *DataTo                this is the output data array            *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

ByTe_2 ir_convert_mode_data (ByTe_1 which_mode, ByTe_1 num_tbls, 
                             ByTe_1 *tbls_to_apply, ByTe_2 *tbl_oper, 
                             ReaL_4 *DataOut, ByTe_1 IFmt, void *UDF)
{
  extern struct general_info ginfo;
  extern struct def_oper IDF_OPER[];
  extern void   *Tmp_Space;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct mode_tables *mptr;
  register ReaL_8 *d1;
  register ByTe_1 J;
  register ByTe_2 *OP;
  register ByTe_1 *TbL, *Tend;
  ReaL_8 RetD[1], *RetB;
  ReaL_8 *DataFrom, *DataTo;
  ReaL_4 *f1;
  ByTe_4 data_start, num_data, hold_val, num_ele_in_tbl;
  ByTe_4 T, F;
  ByTe_2 BasOp, ExOp, BufOp, PosOp;
  ByTe_2 A, C, R;
  ByTe_1 *mode, dtype, fn_of_raw, *DFrom, tdw_len, tbl_ind;

  /***********************************************************************/
  /*  Bad mode value?                                                    */
  /***********************************************************************/

  ExDa = (struct idf_data *) UDF;
  if (which_mode > (ByTe_2) ExDa->mode_len)
    return (CONV_MODE_BAD_MODE);

  /***********************************************************************/
  /*  Transfer the integer raw value into the data array to be returned  */
  /*  to the user.  There is no D_TYPE for mode data.  It is always an   */
  /*  unsigned long.  Return just the mode in question, so one value.    */
  /***********************************************************************/

  ex = ginfo.expt;
  mode = (ByTe_1 *) ExDa->mode;
  data_start = which_mode;
  num_data = 1;
  ir_transfer_data (RetD, mode, num_data, 0, data_start);
  RetB = Tmp_Space;
  DataTo = RetD;

  /***********************************************************************/
  /*  Since tdw_len is used in the IDF_OPER routines only when dtype = 1,*/
  /*  set this value to zero. (dtype = 0 or 2 for mode values)           */
  /***********************************************************************/

  mptr = ex->mode_tbl_ptr + which_mode;
  tdw_len = 0;

  /*********************************************************************/
  /*  If tables are to be applied, convert to the data accordingly.    */
  /*********************************************************************/

  OP = tbl_oper;
  TbL = tbls_to_apply;
  Tend = TbL + num_tbls;
  for (  ; TbL < Tend; ++OP, ++TbL)
   {
     /*******************************************************************/
     /*  Check for combination of accumulators since operations are NOT */
     /*  table specific.                                                */
     /*******************************************************************/

     PosOp = (*OP < 0 ) ? -(*OP): *OP;              /* positive operator     */
     BufOp = PosOp / 1000;                          /* buffer specific op    */
     ExOp = (PosOp % 1000) / 10;                    /* extended operation    */
     BasOp = (*OP % 10);                            /* basic operation       */
     if (BasOp < 0)                                 /* negative base oper?   */
        BasOp = 9 - BasOp;                          /* make it positive      */
     
     if (BufOp > 1 && BufOp < 20)                   /* buf-buf operation     */
     {                                              /* BEG BUFFER OPERATION  */
        R = BufOp - 2;                              /* reduced buffer OP     */
        A = R % 4;                                  /* seq ops 0-3,0-3,0-3   */
        C = R / 4;                                  /* oper block            */
        if (C < 3)                                  /* 2-way buf operations  */
        {                                           /* BEGIN 2-WAY OPS       */
           T = A;                                   /* finding TO buffer     */
           F = (A == 0) ? C + 1 : (A == 3) ? C :    /* finding FROM buffer   */
                        (C < 2) ? R/2 : 3*(A % 2);  /* offset                */
        }                                           /* END 2-WAY OPS         */
        else                                        /* 1-way buf operations  */
        {                                           /* BEGIN 1-WAY OPS       */
           T = (R % 3) + 4;                         /* finding TO buffer     */
           F = (R > 16) ? (R % 16) : 0;             /* finding FROM buffer   */
        }                                           /* END 1-WAY OPS         */

        F = (F - 1) * ex->swp_len;                  /* From byte offset      */
        T = (T - 1) * ex->swp_len;                  /* To byte offset        */

        DataTo = (T < 0) ? RetD : RetB + T;         /* set the DataTo buffer */
        DataFrom = (F < 0) ? RetD : RetB + F;       /* set DataFrom buffer   */

        CombineOp (DataTo, DataFrom, num_data, BasOp); /* do basic operation */

        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/

        if (ExOp > 0)
          ir_idf_ex_func (ExOp, DataTo, num_data, UDF);
        continue;
      }

     /********************************************************************/
     /*  Find the requested table in the array of tables held for modes. */
     /********************************************************************/

     tbl_ind = -1;
     for (J = 0; J < mptr->num_tbls; ++J)
      {
        if (*(mptr->tbl_num + J) == *TbL)
         {
           tbl_ind = J;
           break;
         }
      }
     if (tbl_ind == -1)
       return (CONV_MODE_BAD_TBL_NUM);

     /********************************************************************/
     /*  A check is made upfront for ASCII tables.                       */
     /********************************************************************/

     if (*(ex->tbl_type + *TbL) == 1)
       continue;
     if (*(ex->tbl_var + *TbL) != 4 && *(ex->tbl_var + *TbL) != 5)
       return (CONV_MODE_MISMATCH);

     /********************************************************************/
     /*  Is this table to be applied for the conversion of the data?     */
     /*  A size of 0 indicates that no table is defined (tbl_fmt < 0);   */
     /*  therefore, this table is bypassed for data conversion.          */
     /********************************************************************/

     if (*(mptr->tbl_size + tbl_ind) != 0)
      {
        fn_of_raw = (*(mptr->tbl_var + tbl_ind) == 4) ? 1 : 0;
        T = ((BufOp % 18) - 1) * ex->swp_len;
        DataTo = (T < 0) ? RetD : RetB + T;
        hold_val = 1;
   
        /************************************************************/
        /*  The table is a function of raw mode data.               */
        /************************************************************/

        if (*(mptr->tbl_var + tbl_ind) == 4) 
         {
           DFrom = mode;
           dtype = 0;
           data_start = which_mode;
         }

        /***************************************************************/
        /* The table is a function of a processed mode data, which is  */
        /* always floating point data (since processed).               */
        /***************************************************************/

        else
         {
           DFrom = (ByTe_1 *)DataTo;
           dtype = 2;
           data_start = 0;
         }

        /****************************************************************/
        /*  Perform the operation defined for the table.                */
        /****************************************************************/
   
        if (*(ex->tbl_type + *TbL) == 2 || *(ex->tbl_type + *TbL) == 4)
          num_ele_in_tbl = *(mptr->tbl_size + tbl_ind) / ex->swp_len;
        else
          num_ele_in_tbl = *(mptr->tbl_size + tbl_ind);

        IDF_OPER[BasOp]._math_drv (DataTo, DFrom, data_start, 
                    *(mptr->tbl_ptr + tbl_ind), num_ele_in_tbl,
                    num_data, *(mptr->tbl_expand + tbl_ind), 
                    dtype, hold_val, fn_of_raw, 
                    *(ex->tbl_type + *TbL), UDF, ex->swp_len);

        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/
   
        if (ExOp > 0)
          ir_idf_ex_func (ExOp, DataTo, num_data, UDF);
      }
   }
  /************************************************************************/
  /* Transfer to output array                                             */
  /************************************************************************/

   RetB = DataTo;
   if (IFmt == 1)
   {
      d1 = (ReaL_8 *)DataOut;
      *d1 = *RetB;
   }
   else
   {
      f1 = DataOut;
      *f1 = *RetB;
   }

  return (ALL_OKAY);
}
