#include <string.h>
#include "ret_codes.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                          IR_CHECK_IDF_DATA_MEMORY SUBROUTINE               *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to make sure that the size of the various        *
 *  arrays in the idf_data structure are of sufficient size since the size of *
 *  the arrays is dependent upon information that is dynamic (found in the    *
 *  header record).                                                           *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    unsigned long data_key   key which uniquely identifies the data set     *
 *                             being processed                                *
 *    char *exten              the filename extension for the data to be used *
 *    unsigned short vnum      version number to be associated with this      *
 *                             combination (allows for multiple opens)        *
 *    void *idf_data_ptr       ptr to the memory location for the structure   *
 *                             that holds returned data values (read_drec)    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_check_idf_data_memory (data_key, exten, vnum, idf_data_ptr)      *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_locate_ex()           determines if the requested combination has    *
 *                             already been processed and points to the       *
 *                             correct structure allocated for the combo      *
 *    ir_check_dependent_on_header() makes sure that memory dependent upon    *
 *                             header information is of sufficient size       *
 *    ir_check_cal_size ()     makes sure enough space is allocated in        *
 *                             idf_data structure for cal. data               *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    short rV                 holds the value returned by the called routine *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

ByTe_2 ir_check_idf_data_memory (u_ByTe_4 data_key, ByTe_1 *exten, 
                                 u_ByTe_2 vnum, void *idf_data_ptr)
{
   ByTe_2 rV; 

  /**************************************************************************/
  /*  Check to see if the combination being processed has been processed    */
  /*  before.  If not, an error condition - probably didn't call FILE_OPEN. */
  /*  Since ir_locate_ex() is called by more than one routine, return an    */
  /*  error code that indicates which calling routine resulted in the error.*/
  /*  Since a 0 is passed for the last parameter, the only possible error is*/
  /*  that the requested combination was not found among processed combos.  */
  /**************************************************************************/
  
   rV = ir_locate_ex (data_key, exten, vnum, 0);
   if (rV != ALL_OKAY) { return (CHK_DATA_NOT_FOUND); }
 
  /**************************************************************************/
  /*  Set a pointer to structure which holds all pointers for header and    */
  /*  data information for the experiment currently being processed.        */
  /**************************************************************************/

   rV = ir_check_dependent_on_header (idf_data_ptr);
   if (rV != ALL_OKAY) { return (rV); }

   rV = ir_check_cal_size (idf_data_ptr);
   if (rV != ALL_OKAY) { return (rV); }

   return (ALL_OKAY);
}
