#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                         ADJUST_TIME SUBROUTINE                             *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This module adjusts the time components when a day boundary has been    *
 *  crossed.  It adjusts if either previous day has been crossed or next day  *
 *  has been crossed.  Year boundaries and leap years are taken into account. *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    short *year              Year component of time to adjust               *
 *    short *day               Julian day (DOY) component of time to adjust   *
 *    long *time               Time of day                                    *
 *    char time_unit           flag indicating unit for time of day -         *
 *                              1 = seconds                                   *
 *                              2 = milliseconds                              *
 *                                                                            *
 *  OUTPUT VARIABLES                                                          *
 *    short *year              Adjusted year component of time                *
 *    short *day               Adjusted Julian day (DOY) component of time    *
 *    long *time               Adjusted time of day component of time         *
 *                                                                            *
 *  USAGE                                                                     *
 *    adjust_time (&year, &day, &time, time_unit);                            *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    short days_in_year       Used for determining number of day in the      *
 *                             year -- takes leap year into account           *
 *    long comp_value          maximum value for time unit                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 ******************************************************************************/

void adjust_time (ByTe_2 *year, ByTe_2 *day, ByTe_4 *time, ByTe_1 time_unit)
{
   ByTe_2 days_in_yr;
   ByTe_4 comp_value;

  /*******************************************************/
  /*  Adjust time components if previous day is crossed  */
  /*******************************************************/

   comp_value = (time_unit == 1) ? 86400 : 86400000;
   while (*time < 0)
   {
      *time += comp_value;
      *day -= 1;

     /***************************************/
     /*  Check if previous year is crossed  */
     /***************************************/

      if (*day < 1)
      {
         *year -= 1;

        /*********************************/
        /*  Take leap year into account  */
        /*********************************/

         if ((*year % 4 == 0 && *year % 100 != 0) || *year % 400 == 0)
            days_in_yr = 366;
         else
            days_in_yr = 365;

         *day = days_in_yr;
      }
   }

  /***************************************************/
  /*  Adjust time components if next day is crossed  */
  /***************************************************/

   while (*time >= comp_value)
   {
      *time -= comp_value;
      *day += 1;

     /*********************************/
     /*  Take leap year into account  */
     /*********************************/

      if ((*year % 4 == 0 && *year % 100 != 0) || *year % 400 == 0)
         days_in_yr = 366;
      else
         days_in_yr = 365;

     /***********************************/
     /*  Check if next year is crossed  */
     /***********************************/

      if (*day > days_in_yr)
      {
         *day = 1;
         *year += 1;
      }
   }
}
