#include "libbase_udf.h"
#include "util_str.h"

#define LeaP(Y) (((Y) % 4 == 0 && (Y) % 100 != 0) || (Y) % 400 == 0) 

/******************************************************************************
 *                                                                            *
 *                              TIMETBLVAL SUBROUTINE                         *
 *  AUTHOR                                                                    *
 *    Chris Gurgiolo                                                          *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine computes value to be used in a time based table.  The      *
 *  value is the difference in the sensor time from the base time given in    *
 *  table in the units requested by the table.                                *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    float *Tbl               pointer to the beginning of the table elements *
 *                             being used                                     *
 *    void *idf_data_ptr       ptr to memory location for structure that      *
 *                             holds returned data values (read_drec)         *
 *  USAGE                                                                     *
 *    TimeTblVal (&Tbl, idf_data_ptr)                                         *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct idf_data *UDF     structure holding all of the currently         *
 *                             returned data values to be processed           *
 *    register float *f1       pointer to table elements                      *
 *    float V                  return value                                   *
 *    float dD                 day difference between sample and base times   *
 *    float dMs                msec difference between sample and base times  *
 *    float dNs                nsec difference between sample and base times  *
 *    long Yr                  base year                                      *
 *    long Dy                  base day                                       *
 *    long Ms                  base milliseconds                              *
 *    long Ns                  base nanoseconds                               *
 *    long I                   looping variable                               *
 *    long Id                  time base of return variable                   *
 *    long  yR                 sample year                                    *
 *    long  dY                 sample day                                     *
 *    long  mS                 sample millisecond                             *
 *    long  nS                 sample nanosecond                              *
 *                                                                            *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ReaL_4 TimeTblVal (ReaL_4 *Tbl, void *idf_data_ptr)
{
   struct idf_data *UDF;
   register ReaL_4 *f1;

   ReaL_4 V;
   ReaL_4 dD, dMs, dNs;
   ByTe_4 Yr, Dy, Ms, Ns;
   ByTe_4 yR, dY, mS, nS;

   ByTe_4 Id, I;

   UDF = (struct idf_data *)idf_data_ptr;           /* UDF read data         */

   f1 = Tbl;                                        /* table elements        */
   Yr = (ByTe_4)(*f1++ + .05);                      /* base year             */
   Dy = (ByTe_4)(*f1++ + .05);                      /* base day of year      */
   Ms = (ByTe_4)(*f1++ + .05);                      /* base msec of day      */
   Ns = (ByTe_4)(*f1++ + .05);                      /* base nsec of msec     */
   Id = (ByTe_4)(*f1 + .05);                        /* output format         */

   yR = UDF->byear;                                 /* sample beginning yr   */
   dY = UDF->bday;                                  /* sample beginning day  */
   mS = UDF->bmilli;                                /* sample beginning msec */
   nS = UDF->bnano;                                 /* sample beginning nsec */

   if (Yr != yR)                                    /* cross a year?         */
   {                                                /* BEGIN MULTI YEAR      */
      dD = LeaP(Yr) ? 366 - Dy : 365 - Dy;          /* days left in base yr  */

      for (I = Yr + 1; I < yR; ++I)                 /* loop over remain yrs  */
         dD += LeaP(I) ? 366 : 365;                 /* compute days;         */

      dD += dY;                                     /* add days in last yr   */
   }                                                /* END MULTI YEAR        */
   else                                             /* all in same year      */
      dD = dY - Dy;                                 /* just diff in days     */

   dMs = mS - Ms;                                   /* diff in milliseconds  */
   if (dMs < 0)                                     /* need correction?      */
   {                                                /* BEGIN MSEC CORRECTION */
      dMs += 86400000;                              /* day wrap correction   */
      --dD;                                         /* one less day          */ 
   }                                                /* END MSEC CORRECTION   */

   dNs = nS - Ns;                                   /* diff in nanoseconds   */
   if (dNs < 0)                                     /* need correction?      */
   {                                                /* BEGIN NSEC CORRECTION */
      dNs += 1000000000;                            /* day wrap correction   */
      --dNs;                                        /* one less millisecond  */ 
   }                                                /* END NSEC CORRECTION   */

   switch (Id)                                      /* switch on output fmt  */
   {                                                /* BEGIN FMT SWITCH      */
      case 0:                                       /* START BASE IS YEAR    */
        V = dD / 365.0 + dMs / 3.1536e10;           /* convert to years      */
      break;                                        /* STOP BASE IS YEAR     */
      case 1:                                       /* START BASE IS DAY     */
        V = dD + dMs / 8.64e7;                      /* convert to days       */
      break;                                        /* STOP BASE IS DAY      */
      case 2:                                       /* START BASE IS HOUR    */
        V = dD * 24 + dMs / 3.6e6;                  /* convert to hours      */
      break;                                        /* STOP BASE IS HOUR     */
      case 3:                                       /* START BASE IS MIN     */
        V = dD * 1440 + dMs / 6.0e4;                /* convert to minutes    */
      break;                                        /* STOP BASE IS MIN      */
      case 4:                                       /* START BASE IS SEC     */
        V = dD * 8.64e4 + dMs / 1000.0  + dNs / 1.0e9;  /* convert to sec   */
      break;                                        /* STOP BASE IS SEC      */
      case 5:                                       /* START BASE IS MSEC    */
        V = dD * 8.64e7 + dMs + dNs / 1.0e6;        /* convert to msec       */
      break;                                        /* STOP BASE IS MSEC     */
      case 6:                                       /* START BASE IS USEC    */
        V = dD * 8.64e10 + dMs * 1000 + dNs / 1.0e3;  /* convert to usec     */
      break;                                        /* STOP BASE IS USEC     */
      case 7:                                       /* START BASE IS NSEC    */
        V = dD * 8.64e13 + dMs * 1.0e6 + dNs;       /* convert to nsec     */
      break;                                        /* STOP BASE IS NSEC     */
   }                                                /* END FMT SWITCH        */

   return (V);
}
