
;-------------------------------------------------------------------------------
;
;  Unit Name : l_ZC2Rtns.pro
;
;  Purpose   : This file contains routines to read, write, and plot the
;              data for set 2.  Set 2 includes the start/stop singles data
;              and the binned data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               06/00          v1.0            Original Implementation
;   ELH               01/02          v1.0.8          Modified to handle either
;                                                    binned or event data
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure : l_ZC2DumpData
;
;  Description: Writes data to a text file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_pltpos              struct        I        generic plot parameters
;  l_sngls               struct*       I        singles data parameters
;  l_binned              struct*       I        binned data parameters
;  l_sngls_setup         struct        I        singles plot parameters
;  l_binned_setup        struct        I        binned/coincidences plot parameters
;  l_event_setup         struct        I        event processing & plotting parameters
;  l_event               struct*       I        pointers to the event data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_ZC2DumpData, l_setup, l_pltpos, l_sngls, l_binned, l_sngls_setup, $
l_binned_setup, l_event_setup, l_event
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   ;-------------------------------
   ; create text dump filename
   ;-------------------------------
   l_setup.l_filedesc = 'sum'
   l_GenFileName, l_setup, '0'
   filename = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'

   ;-------------------------------
   ; open text dump file
   ;-------------------------------
   openw, ofd, filename, /get_lun

   ;-------------------------------
   ; write standard header info
   ; to text file
   ;-------------------------------
   l_pltpos.dmp_hdr_title = 'Singles & Coincidences Text Dump'
   l_DumpHdrInfo, ofd, l_setup, l_pltpos, filename

   printf, ofd, '--------------- Singles and Coincidences Text Dump ---------------------'
   printf, ofd, ' '
   printf, ofd, '   Column Definitions:  '
   printf, ofd, ' '
   printf, ofd, '      UTC        : Universal Time Coordinate (corrected)'
   printf, ofd, '      Azi        : Azimuthal spin sector (0-44)'
   printf, ofd, ' '
   printf, ofd, '-------------------------------------------------------------------------'
   printf, ofd, ' '
   printf, ofd, ' '
   printf, ofd, ' '


   if (l_sngls_setup.start eq 'Y') then begin
     nitems = n_elements ((*l_sngls.filledStartUdfTimePtr))
     l_DumpData, ofd, 'Start Singles', (*l_sngls.startDmpPtr), $
                 (*l_sngls.filledStartUdfTimePtr), nitems, $
                 l_setup.l_num_spinsec, l_setup
   endif

   if (l_sngls_setup.stop eq 'Y') then begin
     nitems = n_elements ((*l_sngls.filledStartUdfTimePtr))
     l_DumpData, ofd, 'Stops Singles', (*l_sngls.stopDmpPtr), $
                 (*l_sngls.filledStartUdfTimePtr), nitems, $
                 l_setup.l_num_spinsec, l_setup
   endif

   if (l_binned_setup.use_binned eq 'Y') then begin
      l_BinnedDmpSpectData, ofd, l_binned, l_binned_setup, l_roi, l_setup
   endif else begin
     if (l_event_setup.use_event eq 'Y') then begin
        l_EventDmpSpectData, ofd, l_event, l_event_setup, l_setup
     endif
   endelse

   close, ofd
   free_lun, ofd


RETURN
END


;-------------------------------------------------------------------------------
;  FUNCTION : l_ZC2Proc
;
;  Description:   Combination data set #2
;      stack plot of the singles start, stop, and coincidences
;      (from the binned data)
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table parameters 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_ZC2Proc, l_setup, l_setup_file, l_pltpos, l_rgb_table 
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   year      = strmid(systime(0), 20, 23)
   l_pltpos.plot_hdr_title = '!5Singles & Coincidences'

   ;-------------------------------------------------
   ;  Don't need to call handle missing data
   ;  only need 1 value from the housekeeping for 
   ;  instrument state values at the bottom of the plot
   ;-------------------------------------------------

   status = l_HskpGetData (l_setup, l_setup_file, l_pltpos, $
                           l_hskp_setup, l_hskp)
   if (status eq FAILURE) then return, FAILURE


   ;-------------------------------------------------
   ; orbit and attitude data 
   ;-------------------------------------------------
   status = l_OARead (l_oa, $
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE


   ;-------------------------------------------------
   ; read the singles data setup file & generic setup
   ; file
   ;-------------------------------------------------

   l_SnglsReadSetup, l_sngls_setup, l_setup_file

   if ((l_sngls_setup.start eq 'Y') or $
       (l_sngls_setup.stop eq 'Y')) then begin
       status = l_SnglsGetData (l_setup, $
                                l_setup_file, $
                                l_pltpos, $
                                l_sngls_setup, $
                                l_sngls)
       if (status eq FAILURE) then return, FAILURE

       l_SnglsAdjExData, l_sngls_setup, l_setup.l_num_spinsec, l_sngls

       l_SnglsHndlMissData, l_sngls_setup, l_sngls, $
                            l_setup, l_oa
   endif



   status = l_BinnedEventSelect (l_setup, l_setup_file, l_pltpos, $
                                 l_roi0, l_binned, l_binned_setup, $
                                 l_event, l_event_setup, l_memd, l_memd_setup, $
                                 l_evsp, l_evsp_setup, l_hskp, l_oa, davail)
   if (status eq FAILURE) then return, FAILURE


   ;----------------------------------------
   ; set up the z buffer to write images to
   ;----------------------------------------
   nloops = 1
   if (l_setup.l_postscript eq 'Y') then nloops = 2

   line_plots = 'N'
   l_pltpos.linePlots = 'N'
   if (l_setup.l_line_plots eq 'Y') then line_plots = 'Y' 
   pwr_bars = 'N'
   if (l_setup.l_pwrsup_bars eq 'Y') then pwr_bars = 'Y'

   ;----------------------------------------
   ; generate a line plot
   ;----------------------------------------
   if (line_plots eq 'Y') then begin 

      l_setup.l_pwrsup_bars = 'N'
      l_pltpos.linePlots = l_setup.l_line_plots

      for i = 0, nloops-1 do begin
        l_pltpos.ps = 'N'
        nplots = 0L

        plotted = 0L
        npages   = 0L
        pages = 0L
        web     = l_setup.l_via_web

        nrows = l_sngls_setup.nplots

        if (l_binned_setup.use_binned eq 'Y') then begin
            nrows = nrows + l_binned_setup.nplots
        endif

        if (l_event_setup.use_event eq 'Y') then begin
            nrows = nrows + l_event_setup.nplots
        endif


        npages = CEIL(nrows / MAX_PLOTS)

        if (nrows - MAX_PLOTS gt 0) then npages = npages + 1
        if (npages eq 0) then npages = 1
        if (npages gt 0) then windx = intarr (npages)

        nplots = nrows
   
        ;----------------------------------------
        ; based on the number of plots, adjust
        ; the size of each plot to fit on a
        ; page
        ;----------------------------------------
        l_AdjPlotSize, l_pltpos, l_setup, nplots

        l_setup.l_filedesc = 'sum'
        l_setup.l_filedesc = l_setup.l_filedesc + 'ln' 

        if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
           l_pltpos.ps = 'Y'
           erase, 0 
           l_GenFileName, l_setup, '0'
           l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                               l_setup.l_filename + '.ps'
        endif

        l_SetUpZBuffer, l_pltpos, $
                        l_setup, $
                        l_rgb_table, $
                        colorbar, $
                        nplots


        if ((l_sngls_setup.start eq 'Y') or $
            (l_sngls_setup.stop eq 'Y')) then begin
             l_SnglsPlot, l_sngls_setup, $
                          l_sngls, $
                          l_pltpos, $
                          nplots, $
                          windx, $
                          pages, $
                          plotted,$ 
                          l_setup, $
                          l_oa, $
                          l_rgb_table, $
                          colorbar
        endif


       if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin

           l_BinnedEventPlot, l_setup, $
                              l_binned_setup, $
                              l_event_setup, $
                              l_binned, $
                              l_event, $
                              l_pltpos, $
                              l_roi0, $
                              l_oa, $
                              l_hskp, $
                              l_rgb_table, $
                              nplots, windx, plotted, $ 
                              pages, colorbar
       endif

        if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
           device, /close_file
        endif
     endfor
   endif    ; -------- end line plot --------


   if (!D.NAME eq 'Z') then begin
     if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
        device, /close
     endif
   endif


   ;----------------------------------------
   ; reinitialize plotting coordinates for
   ; spectrograms
   ;----------------------------------------
   l_pltpos.ypltmx = l_pltpos.top_ypltmx
   l_pltpos.ypltmn = l_pltpos.top_ypltmn

   line_plots = 'N'
   l_pltpos.linePlots = 'N' 
   l_setup.l_line_plots = 'N' 
   l_setup.l_pwrsup_bars = pwr_bars
   
   ;----------------------------------------
   ; generate a spectrogram plot
   ;----------------------------------------
   if (line_plots eq 'N') then begin

      for i = 0, nloops-1 do begin
          l_pltpos.ps = 'N'

          nplots = 0L
          plotted = 0L
          npages   = 0L
          pages   = 0L
          web     = l_setup.l_via_web

          nrows = l_sngls_setup.nplots

          if (l_binned_setup.use_binned eq 'Y') then begin
            nrows = nrows + l_binned_setup.nplots
          endif

          if (l_event_setup.use_event eq 'Y') then begin
            nrows = nrows + l_event_setup.nplots
          endif

          npages = CEIL(nrows / MAX_PLOTS)

          if (nrows - MAX_PLOTS gt 0) then npages = npages + 1
          if (npages eq 0) then npages = 1
          if (npages gt 0) then windx = intarr (npages)

          nplots = nrows

          ;----------------------------------------
          ; based on the number of plots, adjust 
          ; the size of each plot to fit on a 
          ; page
          ;----------------------------------------
          l_AdjPlotSize, l_pltpos, l_setup, nplots


          l_setup.l_filedesc   = 'sum'
          l_setup.l_filedesc   = l_setup.l_filedesc + 'sp' 

          ;----------------------------------------
          ; setup for postscript file
          ;----------------------------------------
          if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
             l_pltpos.ps = 'Y'
             l_GenFileName, l_setup, '0'
             l_pltpos.filename = l_setup.l_prod_dst + '/' + $
                                 l_setup.l_filename + '.ps'
          endif

          ;----------------------------------------
          ; create Z buffer 
          ;----------------------------------------
          l_SetUpZBuffer, l_pltpos, $
                          l_setup, $
                          l_rgb_table, $
                          colorbar, $
                          nplots

          if (l_setup.l_pwrsup_bars eq 'Y') then begin
            l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
          endif


          if ((l_sngls_setup.start eq 'Y') or $
              (l_sngls_setup.stop eq 'Y')) then begin

              l_SnglsPlot, l_sngls_setup, $
                           l_sngls, $
                           l_pltpos, $
                           nplots, $
                           windx, $
                           pages, $
                           plotted,$ 
                           l_setup, $
                           l_oa, $
                           l_rgb_table, $
                           colorbar, $
                           l_hskp
          endif

          if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin

              l_BinnedEventPlot, l_setup, $
                                 l_binned_setup, $
                                 l_event_setup, $
                                 l_binned, $
                                 l_event, $
                                 l_pltpos, $
                                 l_roi0, $
                                 l_oa, $
                                 l_hskp, $
                                 l_rgb_table, $
                                 nplots, windx, plotted, $
                                 pages, colorbar

          endif

          if (i eq 1) and (l_setup.l_postscript eq 'Y') then begin
             device, /close_file
          endif
     endfor

   endif    ;-------- end spectrogram plots --------

   if (!D.NAME eq 'Z') then begin
      if (l_setup.l_via_web eq 'Y') or (l_setup.l_gen_sum eq 'Y') then begin
         device, /close
      endif
   endif


  if (l_setup.l_dumps eq 'Y') then begin
      l_ZC2DumpData, l_setup, l_pltpos, l_sngls, l_binned, $
                     l_sngls_setup, l_binned_setup, l_event_setup, l_event
  endif

  if (l_setup.l_dumps_bkdata eq 'Y') then begin
     ;-------------------------------
     ; create text dump filename
     ;-------------------------------
     if (l_binned_setup.use_binned eq 'Y') AND (l_setup.l_flux eq 'Y') AND $
        (l_event_setup.use_event eq 'N') then begin
        l_setup.l_filedesc = 'bin'
        l_GenFileName, l_setup, '0'
        fname = l_setup.l_prod_dst + '/' + l_setup.l_filename + 'bk' + '.dmp'
        openw, ofd1, fname, /get_lun
        l_BinnedDumpBKData, ofd1, l_binned, l_binned_setup, l_setup, l_event
        close, ofd1
        free_lun, ofd1

        ;-------------------------
        ; VERIFICATION STUFF
        ;-------------------------
        ;BinDump, l_binned

     endif 

     if (l_event_setup.use_event eq 'Y') then begin
        l_setup.l_filedesc = 'ev'
        l_GenFileName, l_setup, '0'
        fname = l_setup.l_prod_dst + '/' + l_setup.l_filename + 'bk' + '.dmp'
        openw, ofd1, fname, /get_lun
        l_EventDumpBKData, ofd1, l_event, l_event_setup, l_setup
        close, ofd1
        free_lun, ofd1

        ;----------------------------------------------------------
        ; VERIFICATION STUFF
        ; used to compare the binned and event background data
        ; commented out 03/05/2003 ELH, if this particular dump is needed
        ; uncomment, recompile, and select the appropriate binned AND
        ; event options in the setup files in order to generate this
        ; ASCII text dump
        ; IF this needs to be turned on, the l_EventDumpRtns.pro 
        ; needs to be compiled 
        ;----------------------------------------------------------
         ;if (l_binned_setup.use_binned eq 'Y') then begin
        ;   l_setup.l_filedesc = 'be'
        ;   l_GenFileName, l_setup, '0'
        ;   fname = l_setup.l_prod_dst + '/' + l_setup.l_filename + '.dmp'
        ;   openw, ofd1, fname, /get_lun
        ;   BECmpDmp, ofd1, l_binned, l_event, l_event_setup, l_setup
        ;   close, ofd1
        ;   free_lun, ofd1
        ;endif
        ;----------------------------------------------------------

     endif


   endif




   ;-----------------------------------------
   ; dump the power supply and steering 
   ; controller data to a text file
   ;-----------------------------------------
   if (l_setup.l_pwrsup_bars eq 'Y') then begin
      l_PwSupCmdDump, l_hskp, l_setup, l_pltpos
   endif

   if ((l_sngls_setup.start eq 'Y') or $
       (l_sngls_setup.stop eq 'Y')) then begin
       l_SnglsTerm, l_sngls
   endif


   if (l_binned_setup.use_binned eq 'Y') OR (l_event_setup.use_event eq 'Y') then begin
      l_BinnedEventTerm, l_roi0, l_binned, l_binned_setup, $
                         l_event, l_event_setup, l_memd, $
                          l_evsp, l_hskp, l_oa, l_setup, davail
   endif

   l_OATerm, l_oa
   l_HskpTerm, l_hskp

   RETURN, SUCCESS

END
