
;-------------------------------------------------------------------------------
;
;  Unit Name : l_UdfRtns.pro
;
;  Purpose   : This file contains routines which interface with the COTS,
;              Universal Data Format (UDF) application.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               06/00            v1.0            Original Implementation
;  ELH               05/01            v1.0            Modified to interface with
;                                                     ChrisG's myIDL
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function :  l_UdfGetDataHandle
;  Description:
;       Determine if definitive or quicklook data is available.  Open the file.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  >0                            file handle 
;  -1                            failure
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name found
;                                               in $UDF_DATA in the experiment
;                                               directory under IMAGE1 or IMAGE1Q
;                                               data)
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  data_source           string        O        indicates if QL or definitive data
;                                               read
;  RAW                   IDL keyword   I        use UDF raw format
;  GROUP                 IDL keyword   I        use UDf group format
;  COLLAPSE              IDL keyword   I        use UDf collapse format
;  
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_UdfGetDataHandle, inst, expr, vinst, btime, etime, $
  		 data_source, RAW=raw, GROUP=group, COLLAPSE=collapse
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   On_Error, 2 

   keyD = udf_key(['IMAGE', 'IMAGE1', inst, expr, vinst])
   keyQ = udf_key(['IMAGE', 'IMAGE1Q', inst, expr, vinst])


   case 1 of
       Keyword_Set(group): begin
           Catch, udf_open_err
           if udf_open_err EQ 0 then begin
              ;print, 'UDF OPENG Definitive : ', btime
              uh  = udf_open(keyD, btime, etime, BASEUNIT=0, SFORMAT=2)
              data_source = 'D'
           endif else begin
              ;print, 'UDF CLOSEG Definitive : ', btime
              Catch, /cancel 
              Catch, udf_open_err
              if udf_open_err EQ 0 then begin 
                 ;print, 'UDF OPENG Quicklook: ', btime
                 uh  = udf_open(keyQ, btime, etime, BASEUNIT=0, SFORMAT=2)
                 data_source = 'Q'
              endif else begin
                 ;print, 'UDF CLOSEG ERROR: ', btime
                 print, !Error_State.Msg
                 RETURN, FAILURE 
                 ;exit, status=udf_open_err
              endelse
           endelse
          ;Catch, /cancel 
         end
       Keyword_Set(collapse): begin
           Catch, udf_open_err
           if udf_open_err EQ 0 then begin
              ;print, 'UDF OPENC Definitive : ', btime
              uh  = udf_open(keyD, btime, etime, BASEUNIT=0, SFORMAT=1)
              data_source = 'D'
           endif else begin
              ;print, 'UDF CLOSEC Definitive : ', btime
              Catch, /cancel
              Catch, udf_open_err
              if udf_open_err EQ 0 then begin
                 ;print, 'UDF OPENC Quicklook : ', btime
                 uh  = udf_open(keyQ, btime, etime, BASEUNIT=0, SFORMAT=1)
                 data_source = 'Q'
              endif else begin
                 ;print, 'UDF CLOSEC ERROR: ', btime
                 ;udf_close, uh
                 print, !Error_State.Msg
                 RETURN, FAILURE
                 ;exit, status=udf_open_err
              endelse
           endelse
           ;Catch, /cancel
         end
       else: begin
           Catch, udf_open_err
           if udf_open_err EQ 0 then begin
              ;print, 'UDF OPENR Definitive : ', btime, keyD
              uh  = udf_open(keyD, btime, etime, BASEUNIT=0, SFORMAT=0)
              ;print, 'UDF OPENR Def uh = ', uh
              data_source = 'D'
           endif else begin
              Catch, /cancel
              Catch, udf_open_err
              if udf_open_err EQ 0 then begin
                 ;print, 'UDF OPENR Quicklook : ', btime, keyQ
                 uh  = udf_open(keyQ, btime, etime, BASEUNIT=0, SFORMAT=0)
                 data_source = 'Q'
              endif else begin
                 print, !Error_State.Msg
                 RETURN, FAILURE 
                 ;exit, status=udf_open_err
              endelse
           endelse
           ;Catch, /cancel
         end
   endcase


   RETURN, uh

END

;-------------------------------------------------------------------------------
;  Function :  l_UdfGetNumRec
;  Description:
;       Determine the number of records given time 1 to time 2.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  >0                            the number of records in this pass
;  -1                            failure 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name found
;                                               in $UDF_DATA in the experiment
;                                               directory under IMAGE1 or IMAGE1Q
;                                               data)
;  data_source           string        O        indicates if QL or definitive data
;                                               read
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_UdfGetNumRec, btime, etime, inst, expr, vinst, data_source
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   nrec = 0L

   fh = l_UdfGetDataHandle(inst, expr, vinst, btime, etime, $
                           data_source, /GROUP)

;print, 'open fh = ', fh

   if (fh eq -1) then begin
      udf_close, fh
      RETURN, FAILURE 
   endif

   while not udf_eof(fh) do begin
       d = udf_read(fh)
       nrec = nrec+1L
   endwhile

   ;print, tag_names (d)

   udf_close, fh
;print, 'udf_close fh = ', fh

   RETURN, nrec

END
