
;-------------------------------------------------------------------------------
;
;  Unit Name : l_TimeRtns.pro
;
;  Purpose   : This file contains routines related to time manipulations.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               06/00          v1.0            Original Implementation
;   ELH               11/21/01       v1.0.7          Added msec field in ConvSec2Arr
;                                                    procedure
;   ELH               01/07/02       v1.0.7          Fixed condition in l_FindDOYSinceLaunch
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function : Convert
;  Description:
;      Convert the time into seconds.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     UDF time array [yr,doy,hh,mm,ss]
;                                converted to seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  time_array            array         I        UDF time array [yr,doy,hh,mm,ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


FUNCTION Convert, time_array

    timeNsecs = 0.0D

    doy  = double(time_array.doy)  * (86400.)
    hr   = double(time_array.hour) * (3600.)
    min  = double(time_array.min)  * (60.)
    sec  = double(time_array.sec)
    msec = double(time_array.msec)/1000.

    timeNsecs = doy + hr + min + sec + msec

    RETURN, timeNsecs

END


;-------------------------------------------------------------------------------
;  Function: l_LeapYr
;  Description:
;      Determine if the year is a leap year. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  0                             NOT a leap year
;  1                             leap year 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  yr                    long          I        current year
;  year_in_seconds       double        I        current year converted to seconds
;  n_days                long          O        # of days in this year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_LeapYear, yr, year_in_seconds, n_days

    if ( (((yr mod 4) eq 0) and $
          ((yr mod 100) ne 0)) or $
          ((yr mod 400) eq 0) ) then begin
             ;a leap year (60sec * 60min * 24hrs * 366 days)
             n_days = 366
             year_in_seconds = 31622400  
             flag = 1 
    endif else begin
             ;not a leap year (60sec * 60min * 24hrs * 365 days)
             year_in_seconds = 31536000  
             n_days = 365
             flag = 0
    endelse

    RETURN, flag 

END


;-------------------------------------------------------------------------------
;  Function: FindDOYSinceLaunch
;  Description:
;      Find the correct day of year based on the epoch year of day 85 year 2000.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  new_doy                       current day of year
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  year_of_file          long          I        date of file
;  EPOCH_YR              long          I        launch year (2000)
;  day_of_year           long          I        day of year for current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              01/07/02         v1.0.7           Changed day_of_year lt
;                                                     total_ndays to le
;-------------------------------------------------------------------------------

FUNCTION FindDOYSinceLaunch, year_of_file, EPOCH_YR, day_of_year

    nyrs_passed = year_of_file - EPOCH_YR + 1L

    nDays_Array = intarr (nyrs_passed)

    ;-------------------------------------------------------------
    ; launch year was a leap year
    ;-------------------------------------------------------------
    nDays_Array[0] = 366L

    total_ndays = 366L
    for ii = 1L, nyrs_passed - 1L do begin
        flag = l_LeapYear(EPOCH_YR+ii, yr_secs, days)
        nDays_Array[ii] = days
        total_ndays = total_ndays + days
    endfor

    new_doy = day_of_year 


    nitems = nyrs_passed 

    if (day_of_year le total_ndays) then begin
        nitems =  nyrs_passed - 1L 
    endif

    for ii = 0L, nitems - 1L do begin
        new_doy = new_doy - nDays_Array[ii]
    endfor


    return, new_doy

end


;-------------------------------------------------------------------------------
;  Function : MET2UTCSTR
;  Description:
;      Convert MET to UTC into string format.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  string                        return string of doy, hh,mm,ss 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  value                 double        I        MET in seconds
;  year_of_file          long          I        current year of data file
;  drift                 float         I        amount the s/c clock has 
;                                               drifted 
;  
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH                10/19/01       v1.0.7           removed value conversion
;                                                     to long 
;-------------------------------------------------------------------------------


FUNCTION MET2UTCSTR, value, year_of_file, drift

   temp = value

   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34

 
   ; convert to seconds
   ;epoch_yr_sec   = l_LeapYear (epoch_yr)
   epoch_doy_sec  = epoch_doy * 86400
   epoch_hr_sec   = epoch_hr  * 3600
   epoch_min_sec  = epoch_min * 60
   epoch_sec_sec  = epoch_sec

   timeNsecs = epoch_doy_sec + epoch_hr_sec + $
               epoch_min_sec + epoch_sec_sec

   ; met is in 100 millisecs, therefore, divide by 10 to get seconds
   temp = temp/10.

   ; add the met epoch delta which is doy 0 - 85, then add met from epoch met
   ; to get the current met
   temp = temp + timeNsecs

   temp = temp - drift

   ;yr_sec = l_LeapYear (year)
   ;yr     = long(temp)/yr_sec


   doy    = long(temp/86400)
   hour   = long((temp - (86400 * doy))/3600)
   minute = long((temp - (86400 * doy) - (3600 * hour))/60)
   sec    = long(temp mod 60)

   new_doy = FindDOYSinceLaunch (year_of_file, epoch_yr, doy) 


   RETURN, STRING (new_doy, hour, minute, sec, $
                   FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

END


;-------------------------------------------------------------------------------
;  Function : ConvSec2Arr
;  Description:
;      Given time in seconds convert into array 
;      format [year, doy, hour, minute, seconds]
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  time_array                    [yr, doy, hh, mm, ss]
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  temp                  double        I        time in seconds
;  year                  long          I        current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              11/21/01         v1.0.7           Added msec 
;-------------------------------------------------------------------------------

FUNCTION ConvSec2Arr, temp, year

   doy    = long(temp)  /86400
   hour   = (long(temp) - (86400 * doy))/3600
   minute = (long(temp) - (86400 * doy) - (3600 * hour))/ 60
   sec    = long(temp) mod 60
   msec   = fix(sec / 1000)
   ;if (doy gt 366) then doy = 1

   time_array = intarr(6)
   time_array = [year, doy, hour, minute, sec, msec]

   RETURN, time_array

END

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------
FUNCTION ConvSec2Struct, tme, year

   tmp = replicate ({UDF_TIMES_STRUCT}, 1) 
   time_array = ConvSec2Arr (tme, year)

   tmp.year = time_array[0]
   tmp.doy  = time_array[1]
   tmp.hour = time_array[2]
   tmp.min  = time_array[3]
   tmp.sec  = time_array[4]
   tmp.msec = time_array[5]

   RETURN, tmp

END

;-------------------------------------------------------------------------------
;  Function : ConvToUtc
;  Description:
;      Convert time in seconds into UTC.
;
;  Return Value:  type = string
;  Value                         Description
;  -------------------------     ------------------------------
;  time string                   formatted time string
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  value                 double        I        time in seconds
;  print_form            long          I        indicates print format of time
;                                               1 = just doy 
;                                               2 = doy, hh, mm, ss 
;                                               all else = hh, mm 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvToUtc, value, print_form 

   doy    =  long(value)/86400
   hour   =  (long(value) - (86400 * doy))/3600
   minute =  (long(value) - (86400 * doy) - (3600 * hour))/ 60
   sec    =  long(value) mod 60
   if (doy gt 366) then begin
      day = doy - 366
      doy = day 
   endif

   if (print_form eq 1) then begin  ; = 1 for monthly, just days
      RETURN, STRING (doy, FORMAT="(i3)")
   endif else begin
      if (print_form eq 2) then begin  ; =2 for dump file
         RETURN, STRING (doy, hour, minute, sec, $
                 FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")
      endif else begin  ; print_form = 0 for 1 day 
         RETURN, STRING (hour, minute, $
                        FORMAT="(i2.2,':',i2.2)")
      endelse
   endelse

END


;-------------------------------------------------------------------------------
;  Function : ConvArrTimeToSecs
;  Description:
;      Given the time in an array format [year, doy, hour, minute, seconds] into
;      seconds.
;
;  Return Value:  type = double
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     converted time in array format [yr, doy,hh,mm,ss]
;                                into seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  time_array            array         I        [yr,doy,hh,mm,ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvArrTimeToSecs, time_array 


    timeNsecs = 0.0D

    doy  = double(time_array[1]) * 86400
    hr   = double(time_array[2]) * 3600
    min  = double(time_array[3]) * 60
    sec  = double(time_array[4])
    msec  = double(time_array[4])/1000

    timeNsecs = doy + hr + min + sec + msec

    RETURN, timeNsecs

END


;-------------------------------------------------------------------------------
;  Procedure :  l_ConvSec2DOYTime
;  Description:
;    Determine the day of year based on the time in seconds given.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  tsec                  double        I        time in seconds
;  doy                   long          I        day of year
;  hr                    long          I        hour
;  min                   long          I        minute
;  sec                   long          I        seconds
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_ConvSec2DOYTime, tsec, doy, hr, min, sec
    rr = tsec
    doy = rr / 86400
    rr = rr mod 86400
    hr = rr / 3600
    rr = rr mod 3600
    min = rr / 60
    rr = rr mod 60
    sec = rr
    RETURN
END


;-------------------------------------------------------------------------------
;  Procedure : l_ConvDOY2YMD
;  Description:
;      Convert the day of year into year, month, day.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  doy                   long          I        day of year
;  yr                    long          O        year
;  mo                    long          O        month
;  dy                    long          O        day
;  
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_ConvDOY2YMD, doy, yr, mo, dy

    months = indgen(12) + 1
    numdays = intarr(13);
    numdays[0] = 0
    numdays[1] = 31
    numdays[2] = 28
    numdays[3] = 31
    numdays[4] = 30
    numdays[5] = 31
    numdays[6] = 30
    numdays[7] = 31
    numdays[8] = 31
    numdays[9] = 30
    numdays[10] = 31
    numdays[11] = 30
    numdays[12] = 31

    if ( (((yr mod 4) eq 0) and ((yr mod 100) ne 0)) or ((yr mod 400) eq 0) ) then begin
	numdays[2] = 29;
    endif

    md1=0
    md2=0

    for ii=0,11 do begin
      md1 = md1 + numdays[ii]
      md2 = md1 + numdays[ii+1]

      if ((doy gt md1) and (doy le md2)) then begin
	     dy = doy - md1
	     mo = months[ii]
      endif
    endfor

END

;-------------------------------------------------------------------------------
;  Function :  l_ConvYMD2DOY
;  Description:
;      Convert year,month, day into day of year. 
;
;  Return Value:  type = long
;  Value                         Description
;  -------------------------     ------------------------------
;  doy                            day of year
;
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  yr                    long          I        year
;  mo                    long          I        month
;  dy                    long          I        day
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_ConvYMD2DOY, yr, mo, dy

    EPOCH_YR = 2000

    nyrs_passed = yr - EPOCH_YR + 1L
    nDays_Array = intarr (nyrs_passed)

    ;-------------------------------------------------------------
    ; launch year was a leap year
    ;-------------------------------------------------------------
    nDays_Array[0] = 366L

    total_ndays = 366L
    for ii = 1L, nyrs_passed - 1L do begin
        flag = l_LeapYear(EPOCH_YR+ii, yr_secs, days)
        nDays_Array[ii] = days
    endfor

    nyrs_passed = yr - EPOCH_YR

    doy = nDays_Array[0]
    for ii = 1, nyrs_passed - 1L do begin
       doy = doy + nDays_Array[ii]
    endfor

    months = indgen(12) + 1
    numdays = intarr(13);
    numdays[0] = 0
    numdays[1] = 31
    numdays[2] = 28
    numdays[3] = 31
    numdays[4] = 30
    numdays[5] = 31
    numdays[6] = 30
    numdays[7] = 31
    numdays[8] = 31
    numdays[9] = 30
    numdays[10] = 31
    numdays[11] = 30
    numdays[12] = 31

    if ( (((yr mod 4) eq 0) and ((yr mod 100) ne 0)) or ((yr mod 400) eq 0) ) then begin
	numdays[2] = 29;
    endif

    for ii=0,mo-1 do begin
	   doy = doy + numdays[ii];
    endfor

    doy = doy + dy

    RETURN, doy
END

;-------------------------------------------------------------------------------
;  Function: l_GetTime
;  Description:
;      Take UDF time format and convert into seconds.
;
;  Return Value:  type = double
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     UDF time converted into seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  dataTime             UDF struct     I        UDF data time, s/c corrected
;  doy                  long           O        day of year
;  hr                   long           O        hour
;  min                  long           O        minute
;  sec                  long           O        seconds
; 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_GetTime, dataTime, doy, hr, min, sec

   timeNsecs = 0.0D

   doy  = double(dataTime.doy)  * 86400.
   hr   = double(dataTime.hour) * 3600.
   min  = double(dataTime.min)  * 60.
   sec  = double(dataTime.sec)
   msec = double(dataTime.msec)/1000.

   timeNsecs = doy + hr + min + sec + msec

   RETURN, timeNsecs

END


;-------------------------------------------------------------------------------
;  Function : l_InputTimeConv
;  Description: 
;       Determines if the day of year or month/day was
;       entered for the time.  If month/day entered, it must
;       be converted to day of year for the UDF
;
;  Return Value:  type = long array
;  Value                         Description
;  -------------------------     ------------------------------
;  dtime                         [yr, doy, hh, mm, ss]
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  the_date              long[]        I        user input date
;  the_time              long[]        I        user input time
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_InputTimeConv, the_date, the_time


   dtime = intarr(5)
   EPOCH_YR = 2000

   if (the_date[2] gt 0) then begin

       ;---------------------------------------------
       ; year/month/day used, need to convert to doy
       ;---------------------------------------------
       doy = l_ConvYMD2DOY (the_date[0], the_date[1], $
                            the_date[2])
       new_doy = FindDOYSinceLaunch (the_date[0], epoch_yr, doy) 

       dtime[0] = the_date[0]
       dtime[1] = new_doy
       dtime[2] = the_time[0]
       dtime[3] = the_time[1]
       dtime[4] = the_time[2]

   endif else begin

       ;---------------------------------------------
       ;  doy was used in the input date/time
       ;---------------------------------------------

       dtime[0] = the_date[0]
       dtime[1] = the_date[1]
       dtime[2] = the_time[0]
       dtime[3] = the_time[1]
       dtime[4] = the_time[2]

   endelse

   return, dtime

END

;-------------------------------------------------------------------------------
;  Procedure:  AddExtraTime
;  Description:
;       Adds additional minutes to the given time.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_tme             long[]       I/O       user specified start time
;                                               [yr, doy, hh, mm, ss]
;  stop_tme              long[]       I/O       user specified stop time
;                                               [yr, doy, hh, mm, ss]
;  extra                 long          I        add extra minutes to 
;                                               user start/stop times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO AddExtraTime2Start, start_tme, extra

  if (start_tme[3] eq 0) AND $
     (start_tme[2] eq 0) then begin
     if (start_tme[1] eq 1L) then begin
       leap_yr = 365L
       leap_yr_flag = l_LeapYear (start_tme[0] - 1L, secs, n_days) 
       if (leap_yr_flag) then leap_yr = 366L
     endif else begin
       start_tme[1] = start_tme[1] - 1L
     endelse
     start_tme[2] = 23L
     start_tme[3] = 60L - extra
  endif else begin
     if (start_tme[3] eq 0) AND $
        (start_tme[2] ne 0) then begin
        if (start_tme[3] eq 0) then begin
           start_tme[3] = 60L - extra
           start_tme[2] = start_tme[2] - 1
        endif else begin
           if (start_tme[3] gt extra) then begin
              start_tme[3] = start_tme[3] - extra 
           endif else begin
              start_tme[3] = start_tme[3]
           endelse
        endelse
     endif else begin
        start_tme[3] = start_tme[3] - extra
     endelse
  endelse

RETURN
END


PRO AddExtraTime2Stop, stop_tme, extra


  if (stop_tme[3] eq 59L) AND $
     (stop_tme[2] eq 23L) then begin
     ;stop_tme[1] = stop_tme[1] + 1
     ;leap_yr = 365L
     ;leap_yr_flag = l_LeapYear (stop_tme[0], secs, n_days)
     ;if (leap_yr_flag) then leap_yr = 366L
     ;if (stop_tme[1] gt leap_yr) then begin
     ;   stop_tme[1] = 1L 
     ;endif
     ;stop_tme[2] = 0L
     ;stop_tme[3] = extra 
  endif else begin
     if (stop_tme[3] eq 59L) AND $
        (stop_tme[2] ne 23L) then begin
        if (stop_tme[3] eq 59L) then begin
           stop_tme[3] = extra 
           stop_tme[2] = stop_tme[2] + 1
           stop_tme[4] = 0L
           if (stop_tme[2] gt 23L) then begin
              stop_tme[2] = 0
              stop_tme[1] = stop_tme[1] + 1
           endif
        endif else begin
           stop_tme[3] = stop_tme[3] + extra 
        endelse
     endif else begin
           stop_tme[3] = stop_tme[3] + extra
     endelse
  endelse



RETURN
END

;-------------------------------------------------------------------------------
;  Procedure:  SubExtraTime
;  Description:
;       Subtract additional minutes to the given time.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_tme             long[]       I/O       user specified start time
;                                               [yr, doy, hh, mm, ss]
;  stop_tme              long[]       I/O       user specified stop time
;                                               [yr, doy, hh, mm, ss]
;  extra                 long          I        add extra minutes to 
;                                               user start/stop times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO SubExtraTime2Stop, stop_tme, extra

  if (stop_tme[3] eq 0) AND $
     (stop_tme[2] eq 0) then begin
     if (stop_tme[1] eq 1L) then begin
       leap_yr = 365L
       leap_yr_flag = l_LeapYear (stop_tme[0] - 1L, secs, n_days) 
       if (leap_yr_flag) then leap_yr = 366L
     endif else begin
       stop_tme[1] = stop_tme[1] - 1L
     endelse
     stop_tme[2] = 23L
     stop_tme[3] = 60L - extra
  endif else begin
     if (stop_tme[3] eq 0) AND $
        (stop_tme[2] ne 0) then begin
        if (stop_tme[3] eq 0) then begin
           stop_tme[3] = 60L - extra
           stop_tme[2] = stop_tme[2] - 1
        endif else begin
           if (stop_tme[3] gt extra) then begin
              stop_tme[3] = stop_tme[3] - extra 
           endif else begin
              stop_tme[3] = stop_tme[3]
           endelse
        endelse
     endif else begin
        stop_tme[3] = stop_tme[3] - extra
     endelse
  endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Function: ChckStartTime
;  Description: Check to see if the start time is at the beginning of the day.
;     HH:MM:SS = 00:00:00
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0.7          Original implementation
;-------------------------------------------------------------------------------

FUNCTION ChckStartTime, tme
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   status = FAILURE

   if (tme[2] eq 0) then begin 
      if (tme[3] eq 0) then begin 
         if (tme[4] eq 0) then begin 
           if (tme[5] eq 0) then begin 
              status = SUCCESS 
           endif
         endif
      endif
   endif

   RETURN, status

END


;-------------------------------------------------------------------------------
;  Function: ChckStopTime
;  Description: Check to see if the stop time is at the end of the day.
;     HH:MM:SS = 23:59:59
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v0.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ChckStopTime, tme
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   status = FAILURE 

   if (tme[2] eq 23) then begin 
      if (tme[3] eq 59) then begin 
         if (tme[4] eq 59) then begin 
            if (tme[5] eq 59) then begin 
               status = SUCCESS 
            endif
         endif
      endif
   endif

   RETURN, status

END
