;-------------------------------------------------------------------------------
;
;  Unit Name : l_TimeRtns.pro
;
;  Purpose   : This file contains routines related to time manipulations.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               06/00          v1.0            Original Implementation
;   ELH               11/21/01       v1.0.7          Added msec field in ConvSec2Arr
;                                                    procedure
;   ELH               01/07/02       v1.0.7          Fixed condition in FindDOYSinceLaunch
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------


;-------------------------------------------------------------------------------
;  Function: l_LeapYr
;
;  Description: Determine if the year is a leap year. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  0                             NOT a leap year
;  1                             leap year 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  yr                    long          I        current year
;  year_in_seconds       double        I        current year converted to seconds
;  n_days                long          O        # of days in this year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_LeapYear, yr, year_in_seconds, n_days

    if ( (((yr mod 4) eq 0) and $
          ((yr mod 100) ne 0)) or $
          ((yr mod 400) eq 0) ) then begin
             ;a leap year (60sec * 60min * 24hrs * 366 days)
             n_days = 366
             year_in_seconds = 31622400  
             flag = 1 
    endif else begin
             ;not a leap year (60sec * 60min * 24hrs * 365 days)
             year_in_seconds = 31536000  
             n_days = 365
             flag = 0
    endelse

    RETURN, flag 

END


;-------------------------------------------------------------------------------
;  Function: FindDOYSinceLaunch
;
;  Description:
;      Find the calendar day of year given the year and the day of year since
;  LENA launch.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  new_doy                       current day of year
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  year_of_file          long          I        date of file
;  EPOCH_YR              long          I        launch year (2000)
;  day_of_year           long          I        day of year for current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              01/07/02         v1.0.7           Changed day_of_year lt
;                                                     total_ndays to le
;-------------------------------------------------------------------------------

FUNCTION FindDOYSinceLaunch, year_of_file, day_of_year
COMMON EPOCH_DATA, EPOCH_YR


    nyrs_passed = year_of_file - EPOCH_YR + 1L

    nDays_Array = intarr (nyrs_passed)

    ;-------------------------------------------------------------
    ; launch year was a leap year
    ;-------------------------------------------------------------
    nDays_Array[0] = 366L

    total_ndays = 366L
    for ii = 1L, nyrs_passed - 1L do begin
        flag = l_LeapYear(EPOCH_YR+ii, yr_secs, days)
        nDays_Array[ii] = days
        total_ndays = total_ndays + days
    endfor

    new_doy = day_of_year 

    nitems = nyrs_passed 

    if (day_of_year le total_ndays) then begin
        nitems =  nyrs_passed - 1L 
    endif

    for ii = 0L, nitems - 1L do begin
        new_doy = new_doy - nDays_Array[ii]
    endfor

    return, new_doy

end


;-------------------------------------------------------------------------------
;  Function : MET2UTCSTR
;
;  Description: Convert MET to UTC into string format.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  string                        return string of doy, hh,mm,ss 
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  value                 double        I        MET in seconds
;  year_of_file          long          I        current year of data file
;  drift                 float         I        amount the s/c clock has 
;                                               drifted 
;  
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH                10/19/01       v1.0.7           removed value conversion
;                                                     to long 
;-------------------------------------------------------------------------------


FUNCTION MET2UTCSTR, value, year_of_file, drift

   temp = value

   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34

 
   ; convert to seconds
   ;epoch_yr_sec   = l_LeapYear (epoch_yr)
   epoch_doy_sec  = epoch_doy * 86400
   epoch_hr_sec   = epoch_hr  * 3600
   epoch_min_sec  = epoch_min * 60
   epoch_sec_sec  = epoch_sec

   timeNsecs = epoch_doy_sec + epoch_hr_sec + $
               epoch_min_sec + epoch_sec_sec

   ; met is in 100 millisecs, therefore, divide by 10 to get seconds
   temp = temp/10.

   ; add the met epoch delta which is doy 0 - 85, then add met from epoch met
   ; to get the current met
   temp = temp + timeNsecs

   temp = temp - drift

   ;yr_sec = l_LeapYear (year)
   ;yr     = long(temp)/yr_sec


   ;year   = long(temp/86400.*366.)
   ;doy    = long((temp - (86400. * 366. * year))/86400.)
   ;hour   = long((temp - (86400. * 366. * year) - (86400. * doy))/3600.)
   ;minute = long((temp - (86400. * 366. * year) - (86400. * doy) - (3600. * hour))/60.) 

   doy    = long(temp - 86400.)
   hour   = long((temp - (86400. * doy))/3600.)
   minute = long((temp - (86400. * doy) - (3600. * hour))/60.) 
   sec    = long(temp mod 60.)

   new_doy = FindDOYSinceLaunch (year_of_file, doy) 


   RETURN, STRING (new_doy, hour, minute, sec, $
                   FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")

END


;-------------------------------------------------------------------------------
;  Procedure: DaysSinceLaunch
;
;  Description:  Find the day of year with day one being the launch day.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  time_array            long[]        I        current date/time
;  new_doy               long          O        day of year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DaysSinceLaunch, time_array, new_doy

   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34

   new_doy = 0L
   if (time_array[0] eq epoch_yr) then begin
      new_doy = time_array[1] - epoch_doy
   endif else begin
      nyrs = time_array[0] - epoch_yr + 1L
      for i = 0L, nyrs-1L do begin
         flag = l_LeapYear(epoch_yr+i, yr_secs, days)
         new_doy = new_doy + days
      endfor
      new_doy = new_doy - epoch_doy + time_array[1]
   endelse

RETURN
END


;-------------------------------------------------------------------------------
;  Function : Convert
;
;  Description: Convert the time into seconds.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     UDF time array [yr,doy,hh,mm,ss]
;                                converted to seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  time_array            array         I        UDF time array [yr,doy,hh,mm,ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


FUNCTION Convert, time_array

   timeNsecs = 0.0D

   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34

   new_doy = 0L
   if (time_array.year eq epoch_yr) then begin
      new_doy = time_array.doy - epoch_doy
   endif else begin
      nyrs = time_array.year - epoch_yr
      for i = 0L, nyrs-1L do begin
         flag = l_LeapYear(epoch_yr+i, yr_secs, days)
         new_doy = new_doy + days
      endfor
      new_doy = new_doy - epoch_doy + time_array.doy
      ;new_doy = new_doy + time_array.doy
   endelse

   doy  = double(new_doy) * 86400.


   ;year = double(time_array.year) * (86400. * 366.)
   ;doy  = double(time_array.doy) * 86400.

   hr   = double(time_array.hour) * (3600.)
   min  = double(time_array.min)  * (60.)
   sec  = double(time_array.sec mod 60.)
   msec = double(time_array.msec)/1000.

   ;timeNsecs = year + doy + hr + min + sec + msec
   timeNsecs = doy + hr + min + sec + msec


   RETURN, timeNsecs

END

;-------------------------------------------------------------------------------
;  Procedure: FindCalDOY
;
;  Description:  Find the calendar day of year given the launch day of
;      year.  Where, January 1st is day of year one.
;
;  Return Value:  type = long
;  Value                         Description
;  -------------------------     ------------------------------
;  new_doy                       calendar day of year
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  year                  long          I        current year
;  doy                   long          I        current day of year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION FindCalDOY, year, doy
COMMON EPOCH_DATA, EPOCH_YR


   new_doy = 0L
   nyrs = year - EPOCH_YR

   if (year eq EPOCH_YR) then begin
      new_doy = doy + 85
   endif else begin
      new_doy = doy - 281L
      for i = 1L, nyrs - 1L do begin
          flag = l_LeapYear(EPOCH_YR+i, yr_secs, days)
          new_doy = new_doy - days 
      endfor 
   endelse

   RETURN, new_doy
END

;-------------------------------------------------------------------------------
;  Function: ConvSec2Arr_CalDOY
;
;  Description: Convert the time in seconds to an array where the day of year
;       is the calendar day of year.
;
;  Return Value:  type = long[]
;  Value                         Description
;  -------------------------     ------------------------------
;  time_array                    [year, day of year, hour, min, sec]
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  temp                  double        I        current time in seconds
;  year                  long          I        current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvSec2Arr_CalDOY, temp, year


   doy    = fix(temp/86400.)
   new_doy = FindCalDOY (year, doy)
   hour   = fix((temp - (86400. * doy))/3600.)
   minute = fix((temp - (86400. * doy) - (3600. * hour))/ 60.)
   sec    = fix(temp mod 60.)
   msec   = fix(sec/1000.)

   time_array = intarr(6)
   time_array = [year, new_doy, hour, minute, sec, msec]

   RETURN, time_array
END

;-------------------------------------------------------------------------------
;  Function : ConvSec2Arr
;
;  Description:
;      Given time in seconds convert into array 
;      format [year, doy, hour, minute, seconds]
;
;  Return Value:  type = long[]
;  Value                         Description
;  -------------------------     ------------------------------
;  time_array                    [yr, doy, hh, mm, ss]
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  temp                  double        I        time in seconds
;  year                  long          I        current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH              11/21/01         v1.0.7           Added msec 
;-------------------------------------------------------------------------------

FUNCTION ConvSec2Arr, temp, year
COMMON EPOCH_DATA, EPOCH_YR


   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34


   doy    = fix(temp/86400.)
   new_doy = FindCalDOY (year, doy)

   hour   = fix((temp - (86400. * doy))/3600.)
   minute = fix((temp - (86400. * doy) - (3600. * hour))/ 60.)
   sec    = fix(temp mod 60.)
   msec   = fix(sec/1000.)

   time_array = intarr(6)
   time_array = [year, new_doy, hour, minute, sec, msec]


   RETURN, time_array

END

;-------------------------------------------------------------------------------
;  Function: ConvSec2Struct
;
;  Description:
;
;  Return Value:  type = struct
;  Value                         Description
;  -------------------------     ------------------------------
;  tme                           time in struture format
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  tme                   double        I        time in seconds 
;  year                  long          I        current year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvSec2Struct, tme, year

   tmp = replicate ({UDF_TIMES_STRUCT}, 1) 
   time_array = ConvSec2Arr (tme, year)

   tmp.year = time_array[0]
   tmp.doy  = time_array[1]
   tmp.hour = time_array[2]
   tmp.min  = time_array[3]
   tmp.sec  = time_array[4]
   tmp.msec = time_array[5]

   RETURN, tmp

END

;-------------------------------------------------------------------------------
;  Function : ConvToUtc
;
;  Description: Convert time in seconds into UTC.
;
;  Return Value:  type = string
;  Value                         Description
;  -------------------------     ------------------------------
;  time string                   formatted time string
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  value                 double        I        time in seconds
;  print_form            long          I        indicates print format of time
;                                               1 = just doy 
;                                               2 = doy, hh, mm, ss 
;                                               all else = hh, mm 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvToUtc, value, print_form

   doy    = fix(value/86400.)
   hour   = fix((value - (86400. * doy))/3600.)
   minute = fix((value - (86400. * doy) - (3600. * hour))/60.)
   sec    = fix(value mod 60.)
   if (doy gt 366) then begin
      day = doy - 366
      doy = day 
   endif

   if (print_form eq 1) then begin  ; = 1 for monthly, just days
      RETURN, STRING (doy, FORMAT="(i3)")
   endif else begin
      if (print_form eq 2) then begin  ; =2 for dump file
         RETURN, STRING (doy, hour, minute, sec, $
                 FORMAT="(i3,' ',i2.2,':',i2.2,':',i2.2)")
      endif else begin  ; print_form = 0 for 1 day 
         RETURN, STRING (hour, minute, $
                        FORMAT="(i2.2,':',i2.2)")
      endelse
   endelse

END


;-------------------------------------------------------------------------------
;  Function : ConvArrTimeToSecs
;
;  Description:
;      Given the time in an array format [year, doy, hour, minute, seconds] into
;      seconds.
;
;  Return Value:  type = double
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     converted time in array format [yr, doy,hh,mm,ss]
;                                into seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  time_array            array         I        [yr,doy,hh,mm,ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ConvArrTimeToSecs, time_array 


    timeNsecs = 0.0D

    ; mission start
    epoch_yr  = 2000
    epoch_doy = 85
    epoch_hr  = 21
    epoch_min = 30
    epoch_sec = 34

    new_doy = 0L
    if (time_array[0] eq epoch_yr) then begin
       new_doy = time_array[1] - epoch_doy
    endif else begin
       nyrs = time_array[0] - epoch_yr
       for i = 0L, nyrs-1L do begin
          flag = l_LeapYear(epoch_yr+i, yr_secs, days)
          new_doy = new_doy + days
       endfor
       new_doy = new_doy - epoch_doy + time_array[1]
       ;new_doy = new_doy + time_array[1]
    endelse
 
    doy  = double(new_doy * 86400.)
    hr   = double(time_array[2] * 3600.)
    min  = double(time_array[3] * 60.)
    sec  = double(time_array[4])
    msec = double(time_array[4]/1000.)


    timeNsecs = doy + hr + min + sec + msec


    RETURN, timeNsecs

END


;-------------------------------------------------------------------------------
;  Procedure : l_ConvDOY2YMD
;
;  Description: Convert the day of year into year, month, day.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  doy                   double        I        day of year
;  yr                    double        I        year
;  mo                    double        O        month
;  dy                    double        O        day
;  
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_ConvDOY2YMD, doy, yr, mo, dy

    months = indgen(12) + 1
    numdays = intarr(13)
    numdays[0] = 0
    numdays[1] = 31
    numdays[2] = 28
    numdays[3] = 31
    numdays[4] = 30
    numdays[5] = 31
    numdays[6] = 30
    numdays[7] = 31
    numdays[8] = 31
    numdays[9] = 30
    numdays[10] = 31
    numdays[11] = 30
    numdays[12] = 31

    if ( (((yr mod 4) eq 0) and ((yr mod 100) ne 0)) or ((yr mod 400) eq 0) ) then begin
	numdays[2] = 29;
    endif

    md1=0
    md2=0

    for ii=0,11 do begin
      md1 = md1 + numdays[ii]
      md2 = md1 + numdays[ii+1]

      if ((doy gt md1) and (doy le md2)) then begin
	     dy = doy - md1
	     mo = months[ii]
      endif
    endfor

END

;-------------------------------------------------------------------------------
;  Function :  l_ConvYMD2DOY
;
;  Description:
;      Convert year,month, day into day of year since LENA launch. Beginning with
;  3/25/2000 (day of year is 85) as day 1. This is specific to the LENA mission!
;
;  Return Value:  type = double 
;  Value                         Description
;  -------------------------     ------------------------------
;  doy                            day of year
;
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  yr                    double        I        year
;  mo                    double        I        month
;  dy                    double        I        day
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_ConvYMD2DOY, yr, mo, dy
COMMON EPOCH_DATA, EPOCH_YR


    doy = 0L

    nyrs_passed = yr - EPOCH_YR + 1L
    nDays_Array = intarr (nyrs_passed)

    ;-------------------------------------------------------------
    ; launch year was a leap year
    ;-------------------------------------------------------------
    nDays_Array[0] = 366L

    total_ndays = 366L
    for ii = 1L, nyrs_passed - 1L do begin
        flag = l_LeapYear(EPOCH_YR+ii, yr_secs, days)
        nDays_Array[ii] = days
    endfor

    nyrs_passed = yr - EPOCH_YR
    if (nyrs_passed gt 0) then begin
       doy = nDays_Array[0]
       for ii = 1, nyrs_passed - 1L do begin
          doy = doy + nDays_Array[ii]
       endfor
    endif


    months = indgen(12) + 1
    numdays = intarr(13);
    numdays[0] = 0
    numdays[1] = 31
    numdays[2] = 28
    numdays[3] = 31
    numdays[4] = 30
    numdays[5] = 31
    numdays[6] = 30
    numdays[7] = 31
    numdays[8] = 31
    numdays[9] = 30
    numdays[10] = 31
    numdays[11] = 30
    numdays[12] = 31

    if ( (((yr mod 4) eq 0) and ((yr mod 100) ne 0)) or ((yr mod 400) eq 0) ) then begin
	numdays[2] = 29;
    endif


    for ii=0,mo-1 do begin
	   doy = doy + numdays[ii];
    endfor

    doy = doy + dy


    RETURN, doy
END

;-------------------------------------------------------------------------------
;  Function: l_GetTime
;
;  Description: Take UDF time format and convert into seconds.
;
;  Return Value:  type = double
;  Value                         Description
;  -------------------------     ------------------------------
;  timeNsecs                     UDF time converted into seconds
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  dataTime             UDF struct     I        UDF data time, s/c corrected
;  doy                  double         O        day of year
;  hr                   double         O        hour
;  min                  double         O        minute
;  sec                  double         O        seconds
; 
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_GetTime, dataTime, doy, hr, min, sec

   timeNsecs = 0.0D

   ;year = double(dataTime.year  * (86400. * 366.))

   ; mission start
   epoch_yr  = 2000
   epoch_doy = 85
   epoch_hr  = 21
   epoch_min = 30
   epoch_sec = 34

   new_doy = 0L
   if (dataTime.year eq epoch_yr) then begin
      new_doy = dataTime.doy - epoch_doy
   endif else begin
      nyrs = dataTime.year - epoch_yr
      for i = 0L, nyrs-1L do begin
         flag = l_LeapYear(epoch_yr+i, yr_secs, days)
         new_doy = new_doy + days
      endfor
      new_doy = new_doy - epoch_doy + dataTime.doy
   endelse



   doy  = double(new_doy * 86400.)
   hr   = double(dataTime.hour * 3600.)
   min  = double(dataTime.min  * 60.)
   sec  = double(dataTime.sec)
   msec = double(dataTime.msec/1000.)


   timeNsecs = doy + hr + min + sec + msec


   RETURN, timeNsecs

END


;-------------------------------------------------------------------------------
;  Function : l_InputTimeConv
;
;  Description: 
;       Determines if the day of year or month/day was
;       entered for the time.  If month/day entered, it must
;       be converted to day of year for the UDF
;
;  Return Value:  type = long array
;  Value                         Description
;  -------------------------     ------------------------------
;  dtime                         [yr, doy, hh, mm, ss]
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  the_date              long[]        I        user input date
;  the_time              long[]        I        user input time
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


FUNCTION l_InputTimeConv, the_date, the_time

   dtime = intarr(5)

   if (the_date[2] gt 0) then begin

       ;---------------------------------------------
       ; year/month/day used, need to convert to doy
       ;---------------------------------------------
       doy = l_ConvYMD2DOY (the_date[0], the_date[1], $
                            the_date[2])
       new_doy = FindDOYSinceLaunch (the_date[0], doy) 

       dtime[0] = the_date[0]
       dtime[1] = new_doy
       dtime[2] = the_time[0]
       dtime[3] = the_time[1]
       dtime[4] = the_time[2]

   endif else begin

       ;---------------------------------------------
       ;  doy was used in the input date/time
       ;---------------------------------------------

       dtime[0] = the_date[0]
       dtime[1] = the_date[1]
       dtime[2] = the_time[0]
       dtime[3] = the_time[1]
       dtime[4] = the_time[2]

   endelse

   return, dtime

END

;-------------------------------------------------------------------------------
;  Procedure:  AddExtraTime
;
;  Description: Adds additional minutes to the given time.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_tme             long[]       I/O       user specified start time
;                                               [yr, doy, hh, mm, ss]
;  stop_tme              long[]       I/O       user specified stop time
;                                               [yr, doy, hh, mm, ss]
;  extra                 long          I        add extra minutes to 
;                                               user start/stop times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO AddExtraTime2Start, start_tme, extra

  if (start_tme[3] eq 0) AND $
     (start_tme[2] eq 0) then begin
     if (start_tme[1] eq 1L) then begin
       leap_yr = 365L
       leap_yr_flag = l_LeapYear (start_tme[0] - 1L, secs, n_days) 
       if (leap_yr_flag) then leap_yr = 366L
     endif else begin
       start_tme[1] = start_tme[1] - 1L
     endelse
     start_tme[2] = 23L
     start_tme[3] = 60L - extra
  endif else begin
     if (start_tme[3] eq 0) AND $
        (start_tme[2] ne 0) then begin
        if (start_tme[3] eq 0) then begin
           start_tme[3] = 60L - extra
           start_tme[2] = start_tme[2] - 1
        endif else begin
           if (start_tme[3] gt extra) then begin
              start_tme[3] = start_tme[3] - extra 
           endif else begin
              start_tme[3] = start_tme[3]
           endelse
        endelse
     endif else begin
        if (start_tme[3] lt extra) then begin
           orig = start_tme[3]
           start_tme[3] = 60L - extra + orig 
        endif else begin
           start_tme[3] = start_tme[3] - extra
        endelse
     endelse
  endelse

RETURN
END


PRO AddExtraTime2Stop, stop_tme, extra


  if (stop_tme[3] eq 59L) AND $
     (stop_tme[2] eq 23L) then begin
     ;stop_tme[1] = stop_tme[1] + 1
     ;leap_yr = 365L
     ;leap_yr_flag = l_LeapYear (stop_tme[0], secs, n_days)
     ;if (leap_yr_flag) then leap_yr = 366L
     ;if (stop_tme[1] gt leap_yr) then begin
     ;   stop_tme[1] = 1L 
     ;endif
     ;stop_tme[2] = 0L
     ;stop_tme[3] = extra 
  endif else begin
     if (stop_tme[3] eq 59L) AND $
        (stop_tme[2] ne 23L) then begin
        if (stop_tme[3] eq 59L) then begin
           stop_tme[3] = extra 
           stop_tme[2] = stop_tme[2] + 1
           stop_tme[4] = 0L
           if (stop_tme[2] gt 23L) then begin
              stop_tme[2] = 0
              stop_tme[1] = stop_tme[1] + 1
           endif
        endif else begin
           stop_tme[3] = stop_tme[3] + extra 
        endelse
     endif else begin
           stop_tme[3] = stop_tme[3] + extra
     endelse
  endelse



RETURN
END

;-------------------------------------------------------------------------------
;  Procedure:  SubExtraTime
;
;  Description: Subtract additional minutes to the given time.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  start_tme             long[]       I/O       user specified start time
;                                               [yr, doy, hh, mm, ss]
;  stop_tme              long[]       I/O       user specified stop time
;                                               [yr, doy, hh, mm, ss]
;  extra                 long          I        add extra minutes to 
;                                               user start/stop times
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO SubExtraTime2Stop, stop_tme, extra

  if (stop_tme[3] eq 0) AND $
     (stop_tme[2] eq 0) then begin
     if (stop_tme[1] eq 1L) then begin
       leap_yr = 365L
       leap_yr_flag = l_LeapYear (stop_tme[0] - 1L, secs, n_days) 
       if (leap_yr_flag) then leap_yr = 366L
     endif else begin
       stop_tme[1] = stop_tme[1] - 1L
     endelse
     stop_tme[2] = 23L
     stop_tme[3] = 60L - extra
  endif else begin
     if (stop_tme[3] eq 0) AND $
        (stop_tme[2] ne 0) then begin
        if (stop_tme[3] eq 0) then begin
           stop_tme[3] = 60L - extra
           stop_tme[2] = stop_tme[2] - 1
        endif else begin
           if (stop_tme[3] gt extra) then begin
              stop_tme[3] = stop_tme[3] - extra 
           endif else begin
              stop_tme[3] = stop_tme[3]
           endelse
        endelse
     endif else begin
        stop_tme[3] = stop_tme[3] - extra
     endelse
  endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Function: ChckStartTime
;
;  Description: Check to see if the start time is at the beginning of the day.
;     HH:MM:SS = 00:00:00
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0.7          Original implementation
;-------------------------------------------------------------------------------

FUNCTION ChckStartTime, tme
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   status = FAILURE

   if (tme[2] eq 0) then begin 
      if (tme[3] eq 0) then begin 
         if (tme[4] eq 0) then begin 
           if (tme[5] eq 0) then begin 
              status = SUCCESS 
           endif
         endif
      endif
   endif

   RETURN, status

END


;-------------------------------------------------------------------------------
;  Function: ChckStopTime
;
;  Description: Check to see if the stop time is at the end of the day.
;     HH:MM:SS = 23:59:59
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v0.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION ChckStopTime, tme
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

   status = FAILURE 

   if (tme[2] eq 23) then begin 
      if (tme[3] eq 59) then begin 
         if (tme[4] eq 59) then begin 
            ;if (tme[5] eq 59) then begin 
               status = SUCCESS 
            ;endif
         endif
      endif
   endif

   RETURN, status

END


;-------------------------------------------------------------------------------
;  Procedure: SyncData2Time
;
;  Description: Synchronize the data to time.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  theData               float[]       I        the data
;  theTime               double[]      I        the times
;  theDataUdfTime        struct[]      I        UDF times
;  l_setup               struct        I        general setup parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;
;  NOTES:
;     works only for 2 dim arrays, i.e. singles, binned data, event
;     does not work for HKSP data
;-------------------------------------------------------------------------------

PRO SyncData2Time, theData, theTime, theDataUdfTime, l_setup
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA


   time_nitems = n_elements (theDataUdfTime)

   new_theDataUdfTime = replicate ({UDF_TIMES_STRUCT}, time_nitems)
   data_nitems = n_elements (theData) 


   col_items = data_nitems/time_nitems
   new_theData = fltarr (col_items, time_nitems)
   new_theData[*,*] = NO_DATA

   tme_nitems = n_elements (theTime)
   new_theTime = dblarr (tme_nitems)


   cnt = 0L
   tme_arr = dblarr(6)

   start_time = ConvArrTimeToSecs (l_setup.l_start_dtime)
   stop_time = ConvArrTimeToSecs (l_setup.l_stop_dtime)

   start_time = start_time - 60.

   start_idx = 0L
   if (tme_nitems ne time_nitems) then begin
      stop_idx = 44L
      end_idx = 45L
      increments = 45L
   endif else begin
      stop_idx = 0L 
      end_idx = 1L
      increments = 1L
   endelse


   for i = 0L, time_nitems - 1L do begin
      tme_arr[0] = theDataUdfTime[i].year
      tme_arr[1] = theDataUdfTime[i].doy
      tme_arr[2] = theDataUdfTime[i].hour
      tme_arr[3] = theDataUdfTime[i].min
      tme_arr[4] = theDataUdfTime[i].sec
      tme_arr[5] = theDataUdfTime[i].msec

      tme_arr_secs = ConvArrTimeToSecs (tme_arr)
      if (tme_arr_secs ge start_time) AND (tme_arr_secs le stop_time) then begin
         new_theData[*, cnt] = theData[*, i]
         new_theTime[start_idx:stop_idx] = theTime[i*end_idx]
         new_theDataUdfTime[cnt] = theDataUdfTime[i]
         start_idx = stop_idx + 1L
         stop_idx = stop_idx + increments 
         cnt = cnt + 1L
      endif
   endfor


   if (cnt gt 0) then begin
      theData = new_theData[*, 0:cnt-1]
      theDataUdfTime = new_theDataUdfTime[0:cnt-1]
      theTime = new_theTime[0:stop_idx-increments]
   endif else begin
      theData = new_theData
      theDataUdfTime = new_theDataUdfTime
      theTime = new_theTime
   endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: DetYrByDOY
;
;  Description:   Given the day of year, determine which year it is in.  This is LENA
;   specific, based on the LENA launch date.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  days_in_yr_struct     struct[]      I        number of days in a year
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      EPOCH_DATA      EPOCH_YR        int         G          launch year
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO DetYrByDOY, days_in_yr_struct
COMMON EPOCH_DATA, EPOCH_YR


    days_in_yr = { DAYS_IN_YR, $
        start_doy             : 0L, $
        stop_doy              : 0L, $
        year                  : 0L  $
    }

    days_in_yr_struct = replicate ({DAYS_IN_YR}, 100)

    beg_doy = 1L
    end_doy = 0L

    for yr = 0L, 99L do begin
        new_year = EPOCH_YR + yr
        flag = l_LeapYear (new_year, year_in_seconds, n_days)
        end_doy = end_doy + n_days
        days_in_yr_struct[yr].start_doy = beg_doy 
        days_in_yr_struct[yr].stop_doy =  end_doy
        days_in_yr_struct[yr].year = new_year
        beg_doy = end_doy + 1L
    endfor

RETURN
END
