
;-------------------------------------------------------------------------------
;
;  Unit Name : l_Roi0Rtns.pro
;
;  Purpose   : This file contains routines to read, write, and plot the ROI0 
;              data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH               04/01           v1.0            Original Implementation
;                                                        .background data
;   ELH              08/31/01       v1.0.7            Read extra data for adjusting
;                                                     data to sun, earth, moon
;   ELH               01/02          v1.0.8           Added parm in l_HndlMissData to
;                                                     handle rolling data based on
;                                                     spin angle input
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure :  l_RoiAdjExData
;
;  Description:
;      Extra spins are needed when doing sun, moon, earth adjustments.  Therefore,
;  if extra data are read at the beginning or end of the user specified time period,
;  this routine will distinguish the actual data and the extra spins read.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  addedStartflag        long          I        flag to indicate if extra spin
;                                               was read at the beginning of the
;                                               time period
;  addedStopflag         long          I        flag to indicate if extra spin
;                                               was read at the end of time the
;                                               period
;  SPIN_SECTORS          long          I        the number of ROI bins 
;  l_roi0                struct *      O        ROI0 data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               12/13/01        v1.0.8           Separated from GetData routine 
;-------------------------------------------------------------------------------


PRO l_RoiAdjExData, addedStartFlag, addedStopFlag, SPIN_SECTORS, l_roi0
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   ;----------------------------------------------------
   ; adjust the arrays if extra data was read
   ;----------------------------------------------------

   extra_data = make_array (SPIN_SECTORS, 2L, /FLOAT, Value = NO_DATA)
   
   numRecs = n_elements ((*l_roi0.timePtr))
   tme_nrec = round(numRecs/ SPIN_SECTORS)


   if (addedStartFlag eq 1L) AND (addedStopFlag eq 1L) then begin
       (*l_roi0.timePtr) = (*l_roi0.timePtr)[SPIN_SECTORS:numRecs]
       (*l_roi0.startUdfTimePtr)  =  (*l_roi0.startUdfTimePtr)[1:tme_nrec-2L]
       (*l_roi0.stopUdfTimePtr)  =  (*l_roi0.stopUdfTimePtr)[1:tme_nrec-2L]
       extra_data[*,0] = (*l_roi0.dataPtr)[0:SPIN_SECTORS-1L]
       extra_data[*,1] = (*l_roi0.dataPtr)[numRecs-SPIN_SECTORS:numRecs-1L]
       ptr = ptr_new (extra_data)
       l_roi0.dataExtraPtr = ptr 
       (*l_roi0.dataPtr) = (*l_roi0.dataPtr)[SPIN_SECTORS:numRecs-SPIN_SECTORS]
   endif else begin
     if (addedStartFlag eq 1L) AND (addedStopFlag eq 0L) then begin
       (*l_roi0.timePtr) = (*l_roi0.timePtr)[SPIN_SECTORS:numRecs-1L]
       (*l_roi0.startUdfTimePtr)  =  (*l_roi0.startUdfTimePtr)[1:tme_nrec-1L]
       (*l_roi0.stopUdfTimePtr)  =  (*l_roi0.stopUdfTimePtr)[1:tme_nrec-1L]
       extra_data[*,0] = (*l_roi0.dataPtr)[0:SPIN_SECTORS-1L]
       extra_data[*,1] = NO_DATA
       ptr = ptr_new (extra_data)
       l_roi0.dataExtraPtr = ptr 
       (*l_roi0.dataPtr) = (*l_roi0.dataPtr)[SPIN_SECTORS:numRecs-1L]
     endif else begin
       if (addedStartFlag eq 0L) AND (addedStopFlag eq 1L) then begin
         (*l_roi0.timePtr) = (*l_roi0.timePtr)[0:numRecs-SPIN_SECTORS]
         (*l_roi0.startUdfTimePtr)  =  (*l_roi0.startUdfTimePtr)[0:tme_nrec-2L]
         (*l_roi0.stopUdfTimePtr)  =  (*l_roi0.stopUdfTimePtr)[0:tme_nrec-2L]
         extra_data[*,0] = NO_DATA
         extra_data[*,1] = (*l_roi0.dataPtr)[numRecs-SPIN_SECTORS:numRecs-1L]
         ptr = ptr_new (extra_data)
         l_roi0.dataExtraPtr = ptr 
         (*l_roi0.dataPtr) = (*l_roi0.dataPtr)[0:numRecs-SPIN_SECTORS]
       endif else begin
         extra_data[*,*] = NO_DATA
         ptr = ptr_new (extra_data)
         l_roi0.dataExtraPtr = ptr 
       endelse
     endelse
  endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure :  l_RoiGetData
;
;  Description:
;      Determines the number of records and calls the routine to read the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameters
;  l_roi0                struct *      O        ROI0 data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;  ELH               08/31/01        v1.0.7           Code to read extra spins 
;-------------------------------------------------------------------------------

PRO l_RoiGetData, l_setup, l_roi0
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   inst    = 'LENA' 
   expr    = 'LENASCI'
   vinst   = 'IMLROI0' 


   numRecs = 0

   start_tme = l_setup.l_start_dtime
   stop_tme = l_setup.l_stop_dtime

   ;----------------------------------------------------
   ; if the start time is not 00:00:00, then add extra
   ; minutes to the begin time
   ;----------------------------------------------------
   addedStartFlag = 0L
   if (ChckStartTime(start_tme) eq FAILURE) then begin
      AddExtraTime2Start, start_tme, 2
      addedStartFlag = 1L
   endif

   ;----------------------------------------------------
   ; if the stop time is not 23:59:59, then add extra
   ; minutes to the end time
   ;----------------------------------------------------
   addedStopFlag = 0L
   if (ChckStopTime(stop_tme) eq FAILURE) then begin
      AddExtraTime2Stop, stop_tme, 2 
      addedStopFlag = 1L
   endif


   ;----------------------------------------------------
   ; estimate the number of spins based on user specified
   ; time range
   ;----------------------------------------------------
   est_spins = l_UdfGetNumRec (start_tme, $
                               stop_tme,  $
                               inst, expr, vinst, data_source )

   numRecs = l_setup.l_num_roibins * est_spins


   ptr= ptr_new()
   l_roi0 = {                       $
      timePtr                 :ptr, $
      filledTimePtr           :ptr, $
      startUdfTimePtr         :ptr, $
      filledStartUdfTimePtr   :ptr, $
      stopUdfTimePtr          :ptr, $
      dataPtr                 :ptr, $
      filledDataPtr           :ptr, $
      dataExtraPtr            :ptr, $
      dataDmpPtr              :ptr  $
   }

 
   ;----------------------------------------------------
   ; read the ROI data
   ;----------------------------------------------------
   l_RoiReadData, inst, expr, vinst,     $
                   l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime,  $
                   numRecs, l_roi0


   l_RoiAdjExData, addedStartFlag, addedStopFlag, l_setup.l_num_roibins, l_roi0


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure :  l_RoiReadData
;
;  Description: Read the ROI data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  inst                  string        I        instrument (LENA)
;  expr                  string        I        experiment (LENASCI or LENAHSKP)
;  vinst                 string        I        virtual instrument (name from
;                                               $UDF_DATA in the LENASCI or LENAHSKP
;                                               of the IMAGE1 or IMAGE1Q directory)
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  nele                  long          I        total # of records this pass
;  l_roi0                struct *      O        ROI0 data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_RoiReadData, inst, expr, vinst, btime, etime, nele , l_roi0



   data = fltarr (32L*nele) 
   sctime = dblarr (32L*nele)
   start_array = replicate ({UDF_TIMES_STRUCT}, nele)
   stop_array = replicate ({UDF_TIMES_STRUCT}, nele)
   data_source = ''


   fh = l_UdfGetDataHandle (inst, expr, vinst, btime, etime, data_source, /GROUP)

   indx = 0L
   idx  = 0L    

   while not udf_eof (fh) do begin

       d = udf_read (fh)

       start_array[idx].year = d.btime.year
       start_array[idx].doy  = d.btime.doy
       start_array[idx].hour = d.btime.hour
       start_array[idx].min  = d.btime.min
       start_array[idx].sec  = d.btime.sec
       start_array[idx].msec = d.btime.msec

       stop_array[idx].year = d.etime.year
       stop_array[idx].doy  = d.etime.doy
       stop_array[idx].hour = d.etime.hour
       stop_array[idx].min  = d.etime.min
       stop_array[idx].sec  = d.etime.sec
       stop_array[idx].msec = d.etime.msec

       idx = idx + 1L

       bt = l_GetTime (d.btime, doy, hr, min, sec)
       for i = 0, 31 do begin
         data [indx] = d.roi[i,0]
         sctime[indx]= bt
         indx = indx + 1L
       endfor

   endwhile 

   udf_close, fh

   ;----------------------------------------------------
   ; adjust array size to actual number of data elements
   ; read
   ;----------------------------------------------------
   ptr = ptr_new (data[0:indx-1])
   l_roi0.dataPtr = ptr

   ptr = ptr_new (sctime[0:indx-1])
   l_roi0.timePtr = ptr

   ptr = ptr_new (start_array[0:idx-1])
   l_roi0.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array[0:idx-1])
   l_roi0.stopUdfTimePtr = ptr


END


;-------------------------------------------------------------------------------
;  Procedure :  l_RoiReadSetup
;
;  Description: Read the options selected specific to ROI data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_plot_setup          struct       I/O       ROI0 plot specific parameters
;  l_setup_file          string        I        setup file name
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_RoiReadSetup, l_plot_setup, l_setup_file


   openr, inlun, /get_lun, l_setup_file

   l_plot_setup = {               $
       roi0      :  'N',  $
       color_min :  0.0D, $
       color_max :  0.0D, $
       nplots    :  0L    $
   }

   ;----------------------------------------
   ; read roi data setup parameters
   ;----------------------------------------

   l_plot_setup.nplots = 0L

   line = ' '
   while not EOF(inlun) do begin		;While there is text left, output it.
       readf, inlun, line
       IF (STRMID(line, 0, 1) ne ';') and (STRLEN(line) gt 0) then begin
           parts = STR_SEP(line, ' ')
           if (n_elements (parts) gt 2) then begin
           case parts[0] of
	       'l_roi0'    : begin
                            l_plot_setup.roi0=STRUPCASE(parts[2])
		                    if parts[2] eq 'Y' then begin
		                       l_plot_setup.nplots = l_plot_setup.nplots + 1L
                            endif
		                 end
           'l_roi0_color_min' : begin
                                   l_plot_setup.color_min = double(parts[2])
                                end 
           'l_roi0_color_max' : begin
                                   l_plot_setup.color_max = double(parts[2])
                                end 
           else: begin
	       end
           endcase				;End of CASE statement.
           endif
       endif
   endwhile

   close, inlun
   free_lun, inlun

RETURN
END


;-------------------------------------------------------------------------------
; Procedure :  l_RoiShowSetup
;   Used for debug purposes.
;-------------------------------------------------------------------------------

PRO l_RoiShowSetup, l_plot_setup

print, '____LENA Roi0 Data Setup Parameters'
print, 'L_Roi0  : ', l_plot_setup.roi0
print, 'L_NPLOTS: ', l_plot_setup.nplots

RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_RoiHndlMissData
;
;  Description: Fill any data gaps in the ROI data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_roi_setup           struct        I        ROI processing & plotting parameters
;  l_roi                 struct *      I        pointers to ROI data
;  l_setup               struct        I        general setup parameters
;  l_oa                  struct *      I        pointers to OA data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_RoiHndlMissData, l_roi_setup, l_roi, l_setup, l_oa


      data_format = l_setup.l_num_roibins
      adjFlag   = 'NO'
      center_sector = 0L

      nspins    = floor(n_elements((*l_roi.dataPtr))/l_setup.l_num_roibins)


      ;--------------------------------------------------
      ; handle data gaps
      ;--------------------------------------------------


      l_HndlMissData, (*l_roi.timePtr),      $
                      filled_time,           $     
                      nspins,                $
                      (*l_roi.dataPtr),      $
                      filled_data,           $
                      data_format ,          $
                      center_sector,         $
                      l_setup,               $
                      adjFlag,               $
                      l_oa,                  $
                      (*l_roi.startUdfTimePtr), $
                      filled_start_udf_time, $
                      (*l_roi.dataExtraPtr), $
                      l_setup.l_spin_ang_zero


     ptr = ptr_new (filled_time)
     l_roi.filledTimePtr = ptr

     ptr = ptr_new (filled_data)
     l_roi.filledDataPtr = ptr

     ptr = ptr_new (filled_start_udf_time)
     l_roi.filledStartUdfTimePtr = ptr


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure :  l_RoiPlot
;
;  Description: Plot the data.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_roi                 struct *      I        ROI0 data parameters
;  l_pltpos              struct        I        generic plot parameters
;  nplots                long          I        total # of plots to be plotted
;  windx                 long[]        I        window index
;  pages                 long          I        # pages needed for plots
;  plotted               long          I        # plots already plotted
;  l_setup               struct        I        generic setup parameters
;  l_oa                  struct *      I        OA data parameters
;  data_format           long          I        single scalar value or 45 elements
;  l_rgb_table           struct        I        generic color table values
;  colorbar              float[]       I        color bar values
;  l_hskp                struct *       I        HSKP data parameters
;  min_color             double        I        minimum color to scale to
;  max_color             double        I        maximum color to scale to
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_RoiPlot, l_roi, l_pltpos, $
nplots, windx, pages, plotted, l_setup , l_oa, $
data_format, l_rgb_table, colorbar, l_hskp, min_color, max_color
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNT, NO_DATA



   if (l_setup.l_log eq 'Y') then begin
      yaxis_title  = '!5log Cnt Rate [Hz]'
   endif else begin
      yaxis_title  = '!5linear Cnt'
   endelse

   ytitle_frame = '!5TOF Bin'
   ytitle = '!5ROI'

 
   l_setup.l_d_start_dtime[0] = (*l_roi.startUdfTimePtr)[0].year
   l_setup.l_d_start_dtime[1] = (*l_roi.startUdfTimePtr)[0].doy
   l_setup.l_d_start_dtime[2] = (*l_roi.startUdfTimePtr)[0].hour
   l_setup.l_d_start_dtime[3] = (*l_roi.startUdfTimePtr)[0].min
   l_setup.l_d_start_dtime[4] = (*l_roi.startUdfTimePtr)[0].sec
   l_setup.l_d_start_dtime[5] = (*l_roi.startUdfTimePtr)[0].msec

   max_ele = n_elements((*l_roi.stopUdfTimePtr)[*])-1L

   l_setup.l_d_stop_dtime[0] = (*l_roi.stopUdfTimePtr)[max_ele].year
   l_setup.l_d_stop_dtime[1] = (*l_roi.stopUdfTimePtr)[max_ele].doy
   l_setup.l_d_stop_dtime[2] = (*l_roi.stopUdfTimePtr)[max_ele].hour
   l_setup.l_d_stop_dtime[3] = (*l_roi.stopUdfTimePtr)[max_ele].min
   l_setup.l_d_stop_dtime[4] = (*l_roi.stopUdfTimePtr)[max_ele].sec
   l_setup.l_d_stop_dtime[5] = (*l_roi.stopUdfTimePtr)[max_ele].msec


   l_DetMjMnTicks, l_setup.l_start_dtime, $
                   l_setup.l_stop_dtime, $
                   l_setup.l_d_start_dtime, $
                   l_setup.l_d_stop_dtime, $
                   l_pltpos

   l_pltpos.ymn_tick = 0L


   roi_data = (*l_roi.filledDataPtr) 

   if (l_setup.l_dumps eq 'Y') then begin
      ptr = ptr_new (roi_data)
      l_roi.dataDmpPtr = ptr 
   endif

   ;--------------------------------------------------
   ; convert to log
   ;--------------------------------------------------
   if (l_setup.l_log eq 'Y') then begin
      l_Conv2Log, roi_data, log_data, min_value, max_value
      roi_data = log_data
   endif else begin
      NO_CNTS = 0.0
      non_zero = where (roi_data ne NO_CNTS, wcnt)
      if (wcnt gt 0) then begin
         min_value = min (roi_data[non_zero])
      endif else begin
         min_value = min (roi_data)
      endelse

      max_value = max (roi_data)
   endelse

   if (l_setup.l_auto_scale eq 'Y') then begin
      l_setup.l_color_min = min_value
      l_setup.l_color_max = max_value
   endif else begin
      if (l_setup.l_sep_plot_scale eq 'Y') then begin
         l_setup.l_color_min = min_color
         l_setup.l_color_max = max_color
      endif
   endelse



   if (l_setup.l_line_plots eq 'Y') then begin

      diff = max_value - min_value
      if (diff gt 1) then begin
         new_max = (max_value/4.) + max_value
         max_value = new_max
      endif else begin
         max_value = max_value + diff
      endelse

      l_PlotStuff, nplots,                   $
                   (*l_roi.filledTimePtr),   $
                   roi_data,                 $
                   min_value,                $
                   max_value,                $
                   yaxis_title,              $
                   l_pltpos,                 $
                   plotted,                  $
                   l_oa,                     $
                   l_setup.l_start_dtime


    endif else begin  ; a spectrogram plot

       l_ScaleColor, roi_data, scaled_data, $
                     l_setup.l_color_min, $
                     l_setup.l_color_max , $
                     l_setup.l_color_tbl



       ;------------------------------------------------------------
       ; correct the image positioning, rotate 240 degrees
       ;------------------------------------------------------------
       rot_data = rotate(scaled_data, 3)

       yrng=[1,l_setup.l_num_roibins]

       img_pos = [l_pltpos.xpltmn,l_pltpos.ypltmn,$
                  l_pltpos.xpltmx,l_pltpos.ypltmx]

       position = float(img_pos)

       xstart = position[0]
       ystart = position[1]
       ximage = l_pltpos.axis_size/ l_pltpos.xin
       yimage = l_pltpos.ypltmx - l_pltpos.ypltmn
       xsize  = round(l_pltpos.window_width * ximage)
       ysize  = round(l_pltpos.window_height * yimage)

       spect_data = congrid (rot_data, $
                             xsize, $
                             ysize)


       TV, spect_data, xstart, ystart, $
           XSIZE=ximage, YSIZE=yimage, /ORDER, $
           /NORMAL


       bar_pos = [l_pltpos.xpltmx+0.015, $
                  l_pltpos.ypltmn, $
                  l_pltpos.xpltmx+0.025, $
                  l_pltpos.ypltmx]

       bar_info = {AXIS_DATA, char_size: l_pltpos.plot_font, $
                   ytitle:yaxis_title, $
                   yformat:'(F6.1)', $
                   yrange:[l_setup.l_color_min, $
                   l_setup.l_color_max]}

       l_DrawColorbar, colorbar, l_setup, bar_pos, l_pltpos, $
                       RAXIS=bar_info

       l_PlotFrame, nplots, (*l_roi.filledTimePtr), yrng, img_pos, l_pltpos, $
                    plotted, l_oa, ytitle_frame, l_setup.l_start_dtime, $
                    l_hskp


   endelse

print, 'roiplot nplots = ', nplots

   yctr = (l_pltpos.ypltmn + l_pltpos.ypltmx)/2.0
   xyouts, 0.02, yctr, ytitle, size=l_pltpos.md_font, $
           /NORMAL, alignment = 0.2, orientation=90.0

   l_pltpos.ypltmx = l_pltpos.ypltmn - l_pltpos.yspace
   l_pltpos.ypltmn = l_pltpos.ypltmx - l_pltpos.ydel
   plotted = plotted + 1L
   l_ChckCompPg, plotted, nplots, pages, $
                 l_setup, windx, l_pltpos, l_rgb_table

   NO_CNTS       = -9999.


   RETURN 

END


;-------------------------------------------------------------------------------
;  Procedure :  l_RoiTerm
;
;  Description: Free memory allocated.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_roi0                struct *      I        ROI0 data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_RoiTerm, l_roi0


   if (ptr_valid (l_roi0.dataPtr)) then begin
       ptr_free, l_roi0.dataPtr
   endif

   if (ptr_valid (l_roi0.filledDataPtr)) then begin
       ptr_free, l_roi0.filledDataPtr
   endif

   if (ptr_valid (l_roi0.dataExtraPtr)) then begin
       ptr_free, l_roi0.dataExtraPtr
   endif

   if (ptr_valid (l_roi0.dataDmpPtr)) then begin
       ptr_free, l_roi0.dataDmpPtr
   endif

   if (ptr_valid (l_roi0.timePtr)) then begin
       ptr_free, l_roi0.timePtr
   endif

   if (ptr_valid (l_roi0.filledTimePtr)) then begin
       ptr_free, l_roi0.filledTimePtr
   endif

   if (ptr_valid (l_roi0.startUdfTimePtr)) then begin
       ptr_free, l_roi0.startUdfTimePtr
   endif

   if (ptr_valid (l_roi0.filledStartUdfTimePtr)) then begin
       ptr_free, l_roi0.filledStartUdfTimePtr
   endif

   if (ptr_valid (l_roi0.stopUdfTimePtr)) then begin
       ptr_free, l_roi0.stopUdfTimePtr
   endif


END

;-------------------------------------------------------------------------------
;  Function :  l_RoiProc
;
;  Description: Main.
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       unsuccessful completion
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_setup               struct        I        generic setup parameter
;  l_setup_file          string        I        setup file name
;  l_pltpos              struct        I        generic plot parameters
;  l_rgb_table           struct        I        generic color table parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

FUNCTION l_RoiProc, l_setup, l_setup_file, l_pltpos, l_rgb_table
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA 
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO


   plotted   = 0L
   pages     = 0L
   nplots    = 0L
   l_pltpos.plot_hdr_title = 'ROI'


   l_RoiReadSetup, l_roi0_setup, l_setup_file

   ;-------------------------------------------------
   ; read the corresponding OA data
   ;-------------------------------------------------
   status = l_OARead (l_oa, $
                      l_setup.l_start_dtime, $
                      l_setup.l_stop_dtime, $
                      l_pltpos)
   if (status eq FAILURE) then return, FAILURE

   status = l_HskpGetData (l_setup, $
                   l_setup_file, $
                   l_pltpos, $
                   l_hskp_setup, $
                   l_hskp)
   if (status eq FAILURE) then return, FAILURE

   l_RoiGetData , l_setup, l_roi0
   l_RoiHndlMissData, l_roi_setup, l_roi, l_setup, l_oa

   nplots    = nplots + l_roi0_setup.nplots
   l_SetUpZBuffer, l_pltpos, $
                   l_setup, $
                   l_rgb_table, $
                   colorbar, $
                   nplots

   npages = CEIL(nplots/MAX_PLOTS)
   if (nplots - MAX_PLOTS gt 0) then npages = npages + 1
   if (npages eq 0) then npages = 1
   if (npages gt 0) then windx = intarr (npages)

   ;-------------------------------------------------
   ; do a spectrogram plot
   ; determine the number of plots and calculate the
   ; size of each plot to fit the page
   ;-------------------------------------------------
   if (npages gt 1) then begin
      plots_per_page = 5
      if (l_setup.l_pwrsup_bars eq 'Y') then plots_per_page = 6
   endif else begin
      plots_per_page = nplots
   endelse

   l_AdjPlotSize, l_pltpos, l_setup, plots_per_page

   if (l_setup.l_pwrsup_bars eq 'Y') then begin
      l_DrawOpBars, l_hskp, l_setup, l_pltpos, l_rgb_table, l_oa
      nplots = nplots - 1L
      plotted   = 1L
   endif


    l_setup.l_filedesc = 'roi0'
    l_setup.l_filedesc = l_setup.l_filedesc + 'sp'

    l_RoiPlot, l_roi0, $
               l_pltpos, $
               nplots, $
               windx, $
               pages, $
               plotted, $
               l_setup, $
               l_oa , $
               l_setup.l_num_roibins, $
               l_rgb_table, $
               colorbar, $
               l_hskp, $
               l_roi0_setup.color_min, $
               l_roi0_setup.color_max


   l_RoiTerm, l_roi0
   l_HskpTerm, l_hskp
   l_OATerm, l_oa

   RETURN, SUCCESS

END
