
;-------------------------------------------------------------------------------
;
;  Unit Name : l_RamRtns.pro
;
;  Purpose   : This file contains all routines related to reading, processing, and
;              calculating the ram locations.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH             04/23/03        v1.0.11          Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Procedure: l_DrawRamDirection
;
;  Description:  Draw the ram direction traces on the plot.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;   X                    float[]       I        X-axis values, time
;   Y                    float[]       I        Y-axis values, ram data
;   img_pos              long[]        I        position of plot
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;     PLOTDEFS         MAX_PLOTS        int         G         max # plots per page
;                      MAX_WEB_SPINS    int         G         max # spins per plot
;                                                             when using web interface
;                      NO_CNTS          double      G         no counts indicator
;                      NO_DATA          double      G         no data indicator
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------

PRO l_DrawRamDirection, X, Y, img_pos
COMMON PLOTDEFS, MAX_PLOTS, MAX_WEB_SPINS, NO_CNTS, NO_DATA

   blankticks = strarr (30)
   blankticks(*) = ' '

   ntck = 15
   if (n_elements (X) ge 60) AND (n_elements (X) lt 240)  then begin
      ntck = 8
   endif else begin
      if (n_elements (X) lt 60) then begin
         ntck = 1 
      endif
   endelse

   nVsum = fix (n_elements(X)/ntck)

   ;-------------------------------------------
   ; reduce the number of points to plot
   ;-------------------------------------------
   x_tmp = X
   y_tmp = Y

   X = dblarr (n_elements(x_tmp))
   Y = fltarr (n_elements(y_tmp))

   for ii = 0L, nVsum - 1L do begin
      X[ii] = x_tmp[ii*ntck]
      Y[ii] = y_tmp[ii*ntck]
   endfor

  
   X = X[0:ii-1L]
   Y = Y[0:ii-1L]


   PLOT, X, Y, /normal, /noerase, $
         xstyle=1, $
         ystyle=1, $
         xrange = [min(X), max(X)], $ 
         yrange = [0, 44], $ 
         ticklen = -0.01, $
         xtickname = blankticks, $
         ytickname = blankticks, $
         position = img_pos, $
         PSYM = 7,  $             ; X
         symsize = 0.75, $
         color = 0                     ; white 


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_DetermineRamDir
;
;  Description: Determine which spin sector the ram data is located.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  ram_markers           float[]       I        filled for data gaps, ram sectors 
;  ram_dir               float[]       I        location of ram sectors after
;                                               data adjusted for various traces
;  l_setup               struct        I        general setup parameters
;  l_oa                  struct *      I        pointers to OA data
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_DetermineRamDir, ram_markers, ram_dir, l_setup, l_oa

      ram_dir = ram_markers
      sectors = l_setup.l_num_spinsec - 1L

      if (l_setup.l_angle_zero eq 'E') then begin
         ;---------------------------------------------------------
         ; find the distance the earth was moved to center position
         ;---------------------------------------------------------
         diff = (l_setup.l_earth_center - l_setup.l_spin_ang_zero)

         ;---------------------------------------------------------
         ; subtract from the sun position the amount earth moved
         ;---------------------------------------------------------
         gt_indx = where (ram_markers ge diff, gtcnt)

         if (gtcnt gt 0) then begin
            ram_dir[gt_indx] = ram_markers[gt_indx] - diff
         endif

         ;---------------------------------------------------------
         ; position of sun when readjusted, crosses over the
         ; 44, 0 boundary
         ;---------------------------------------------------------
         lt_indx = where (ram_markers lt diff, ltcnt)
         if (ltcnt gt 0) then begin
            ram_dir[lt_indx] = sectors - (diff - ram_markers[lt_indx] - 1)
         endif

         indx = where (ram_dir ge 45, cnt)
         if (cnt gt 0) then begin
           ram_dir[indx] = ram_dir[indx] - l_setup.l_num_spinsec
         endif

     endif else begin
          if (l_setup.l_angle_zero eq 'S') then begin

             sun = (*l_oa.lena_sun_sectPtr)
             sun_dist = fltarr (n_elements(sun))
             ;---------------------------------------------------------
             ; subtract from the ram position the amount sun moved
             ; to be centered at zero spin angle
             ;---------------------------------------------------------
             gt_indx = where (sun ge l_setup.l_spin_ang_zero, gtcnt)

             if (gtcnt gt 0) then begin
                sun_dist[gt_indx] = sun[gt_indx] - l_setup.l_spin_ang_zero
                ram_dir[gt_indx] = ram_markers[gt_indx] - sun_dist
                zero_indx = where (ram_dir lt 0, zcnt)
                if (zcnt gt 0) then begin
                    ram_dir[zero_indx] = l_setup.l_num_spinsec + ram_dir[zero_indx]
                endif

             endif

             ;---------------------------------------------------------
             ; position of s/c when readjusted, if s/c sector is at
             ; 33 will cross the 44, 0 boundary
             ;---------------------------------------------------------
             lt_indx = where (sun lt l_setup.l_spin_ang_zero, ltcnt)
             if (ltcnt gt 0) then begin
                sun_dist = l_setup.l_spin_ang_zero - sun[lt_indx]
                indx_gt33 = where (ram_dir[lt_indx] ge $
                          (sectors - l_setup.l_spin_ang_zero), gt33cnt)
                ram_dir[lt_indx] = ram_markers[lt_indx] + sun_dist

                if (gt33cnt gt 0) then begin
                   ram_dir[lt_indx[indx_gt33]] = (ram_dir[lt_indx[indx_gt33]] - $
                              l_setup.l_num_spinsec)
                endif
              endif

          endif else begin
             if (l_setup.l_angle_zero eq 'M') then begin
                moon = (*l_oa.lena_moon_sectPtr)
                ;---------------------------------------------------------
                ; subtract from the s/c position the amount moon moved
                ; to be centered at zero spin angle
                ;---------------------------------------------------------
                gt_indx = where (moon ge l_setup.l_spin_ang_zero, gtcnt)

                if (gtcnt gt 0) then begin
                   moon_dist = moon[gt_indx] - l_setup.l_spin_ang_zero
                   ram_dir[gt_indx] = ram_markers[gt_indx] - moon_dist
                endif

                ;---------------------------------------------------------
                ; position of sun when readjusted, if sun sector is at
                ; 33 will cross the 44, 0 boundary
                ;---------------------------------------------------------
                lt_indx = where (moon lt l_setup.l_spin_ang_zero, ltcnt)
                if (ltcnt gt 0) then begin
                   moon_dist = l_setup.l_spin_ang_zero - moon[lt_indx]
                   indx_gt33 = where (ram_dir[lt_indx] ge $
                          (sectors - l_setup.l_spin_ang_zero), gt33cnt)
                   ram_dir[lt_indx] = ram_markers[lt_indx] + moon_dist

                   if (gt33cnt gt 0) then begin
                      ram_dir[lt_indx[indx_gt33]] = (ram_dir[lt_indx[indx_gt33]] - $
                                  l_setup.l_num_spinsec)
                   endif
                endif
             endif
          endelse
     endelse


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure: l_CalcRamDirection
;
;  Description: Routine which calls the procedures to calulate the RAM direction. 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *      I        pointer to OA data
;  l_setup               struct        I        general setup parameters
;  img_pos               long[]        I        plot position
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               04/23/03       v1.0.11           Original implementation
;-------------------------------------------------------------------------------


PRO l_CalcRamDirection, l_oa, l_setup, img_pos

    nspins = n_elements ((*l_oa.tme_ptr))
    l_HndlMissOA, (*l_oa.tme_ptr), $
                  hmd_oa_tme, $
                  nspins, $
                  (*l_oa.lena_ram_sectPtr), $
                  ram_markers, $
                  l_setup.l_start_dtime, $
                  l_setup.l_stop_dtime, $
                  1

    l_DetermineRamDir, ram_markers, ram_dir, l_setup, l_oa

    l_DrawRamDirection, hmd_oa_tme, $
                        ram_dir, $
                        img_pos


RETURN
END

