
;-------------------------------------------------------------------------------
;
;  Unit Name : l_OARtns.pro
;
;  Purpose   : File contains all routines for reading, writing, and plotting
;              the orbit and attitude data.
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;   ELH                               v1.0           Original implementation
;
;
;  File Revision Number:   %I%
;  File Last Modified  :   %E%  %T%
;
;-------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
;  Function   :  l_OARead
;
;  Description: 
;      Read the orbit and attitude data using the UDF-DLM
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  SUCCESS                       successful completion
;  FAILURE                       failure to complete
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *     I/O       OA data parameters
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  etime                 long[]        I        UDF end time [yr,doy,hh,mm,ss]
;  l_pltpos              struct        I        generic plot parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  l_LenaPlot.pro
;      ERROR_TYPES     SUCCESS         int         G          successful completion
;                      FAILURE         int         G          unsuccessful completion
;                      WARNING         int         G          warning message
;                      INFO            int         G          informational message
;
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH               05/08/00        v1.0             Original implementation
;  ELH                 05/04         v1.0.11          Convert input string to
;                                                     lowercase
;-------------------------------------------------------------------------------

FUNCTION l_OARead, l_oa, btime, etime, l_pltpos
COMMON ERROR_TYPES, SUCCESS, FAILURE, WARNING, INFO

    inst    = 'OA'
    expr    = 'ORBIT'
    vinst   = 'IMOORBIT'

    ptr = ptr_new ()

    l_oa = {             $
         tme_ptr  : ptr, $
         startUdfTimePtr: ptr, $
         stopUdfTimePtr : ptr, $

         mag_field: ptr, $

         lena_sun_sectPtr: ptr, $
         lena_moon_sectPtr: ptr, $
         lena_ram_sectPtr: ptr, $
         lena_mag_fieldPtr: ptr, $

         lsh_ptr  : ptr, $    

         gci_x_solar_pos_ptr: ptr, $
         gci_y_solar_pos_ptr: ptr, $
         gci_z_solar_pos_ptr: ptr, $

         gci_x_lunar_pos_ptr: ptr, $
         gci_y_lunar_pos_ptr: ptr, $
         gci_z_lunar_pos_ptr: ptr, $

         gsm_x_pos_ptr: ptr, $
         gsm_y_pos_ptr: ptr, $
         gsm_z_pos_ptr: ptr, $
         gsm_x_vel_ptr: ptr, $
         gsm_y_vel_ptr: ptr, $
         gsm_z_vel_ptr: ptr, $

         gci_x_pos_ptr: ptr, $
         gci_x_vel_ptr: ptr, $
         gci_y_pos_ptr: ptr, $
         gci_y_vel_ptr: ptr, $
         gci_z_pos_ptr: ptr, $
         gci_z_vel_ptr: ptr, $

         gse_x_pos_ptr: ptr, $
         gse_x_vel_ptr: ptr, $
         gse_y_pos_ptr: ptr, $
         gse_y_vel_ptr: ptr, $
         gse_z_pos_ptr: ptr, $
         gse_z_vel_ptr: ptr $
     
   }

   numRecs = 0L
   data_source = ' '


   start_tme = btime
   stop_tme = etime


   ;------------------------------------------------
   ; pass1, determine number of records to read
   ;------------------------------------------------
   numRecs = l_UdfGetNumRec (start_tme, stop_tme,  $
                             inst, expr, vinst, data_source)


   if (numRecs le 0) then return, FAILURE

   l_pltpos.data_source = data_source


   sctme  = dblarr(numRecs)
   lshell = fltarr(numRecs)

   gci_x_solar_pos = fltarr(numRecs)
   gci_y_solar_pos = fltarr(numRecs)
   gci_z_solar_pos = fltarr(numRecs)

   gci_x_lunar_pos = fltarr(numRecs)
   gci_y_lunar_pos = fltarr(numRecs)
   gci_z_lunar_pos = fltarr(numRecs)

   gsm_x_pos  = fltarr(numRecs)
   gsm_y_pos  = fltarr(numRecs)
   gsm_z_pos  = fltarr(numRecs)

   gsm_x_vel  = fltarr(numRecs)
   gsm_y_vel  = fltarr(numRecs)
   gsm_z_vel  = fltarr(numRecs)

   gci_x_pos  = fltarr(numRecs)
   gci_y_pos  = fltarr(numRecs)
   gci_z_pos  = fltarr(numRecs)

   gci_x_vel  = fltarr(numRecs)
   gci_y_vel  = fltarr(numRecs)
   gci_z_vel  = fltarr(numRecs)

   gse_x_pos  = fltarr(numRecs)
   gse_y_pos  = fltarr(numRecs)
   gse_z_pos  = fltarr(numRecs)

   gse_x_vel  = fltarr(numRecs)
   gse_y_vel  = fltarr(numRecs)
   gse_z_vel  = fltarr(numRecs)

   start_array = replicate ({UDF_TIMES_STRUCT}, numRecs)
   stop_array = replicate ({UDF_TIMES_STRUCT}, numRecs)


   ;------------------------------------------------
   ; read the OA data from the UDF DB
   ;------------------------------------------------
   fh = l_UdfGetDataHandle (inst, expr, vinst, start_tme, stop_tme, data_source, /GROUP)
   if (fh eq -1) then begin
      udf_close, fh
      RETURN, FAILURE
   endif

   indx = 0L


   while not udf_eof (fh) do begin

       d = udf_read (fh)
       bt = l_GetTime (d.btime, doy, hr, min, sec)

       sctme [indx] = bt

       start_array[indx].year =  d.btime.year
       start_array[indx].doy  = d.btime.doy
       start_array[indx].hour = d.btime.hour
       start_array[indx].min  = d.btime.min
       start_array[indx].sec  = d.btime.sec
       start_array[indx].msec = d.btime.msec

       stop_array[indx].year =  d.etime.year
       stop_array[indx].doy  = d.etime.doy
       stop_array[indx].hour = d.etime.hour
       stop_array[indx].min  = d.etime.min
       stop_array[indx].sec  = d.etime.sec
       stop_array[indx].msec = d.etime.msec

       gci_x_solar_pos[indx] = d.gci_solar_position[0]
       gci_y_solar_pos[indx] = d.gci_solar_position[1]
       gci_z_solar_pos[indx] = d.gci_solar_position[2]

       gci_x_lunar_pos[indx] = d.gci_lunar_position[0]
       gci_y_lunar_pos[indx] = d.gci_lunar_position[1]
       gci_z_lunar_pos[indx] = d.gci_lunar_position[2]

       gsm_x_pos[indx]  = d.gsm_satellite_position[0]
       gsm_y_pos[indx]  = d.gsm_satellite_position[1]
       gsm_z_pos[indx]  = d.gsm_satellite_position[2]

       gsm_x_vel[indx]  = d.gsm_satellite_velocity[0]
       gsm_y_vel[indx]  = d.gsm_satellite_velocity[1]
       gsm_z_vel[indx]  = d.gsm_satellite_velocity[2]

       gci_x_pos[indx]  = d.gci_satellite_position[0]
       gci_y_pos[indx]  = d.gci_satellite_position[1]
       gci_z_pos[indx]  = d.gci_satellite_position[2]

       gci_x_vel[indx]  = d.gci_satellite_velocity[0]
       gci_y_vel[indx]  = d.gci_satellite_velocity[1]
       gci_z_vel[indx]  = d.gci_satellite_velocity[2]

       gse_x_pos[indx]  = d.gse_satellite_position[0]
       gse_y_pos[indx]  = d.gse_satellite_position[1]
       gse_z_pos[indx]  = d.gse_satellite_position[2]

       gse_x_vel[indx]  = d.gse_satellite_velocity[0]
       gse_y_vel[indx]  = d.gse_satellite_velocity[1]
       gse_z_vel[indx]  = d.gse_satellite_velocity[2]

       lshell[indx] = d.l_shell[0]

       indx = indx + 1

   endwhile

   udf_close, fh


   ptr = ptr_new(sctme, /NO_COPY)
   l_oa.tme_ptr = ptr

   ptr = ptr_new(lshell, /NO_COPY)
   l_oa.lsh_ptr = ptr

   ;------------------------------
   ;  GCI Solar Position coords
   ;------------------------------
   ptr = ptr_new(gci_x_solar_pos, /NO_COPY)
   l_oa.gci_x_solar_pos_ptr = ptr

   ptr = ptr_new(gci_y_solar_pos, /NO_COPY)
   l_oa.gci_y_solar_pos_ptr = ptr

   ptr = ptr_new(gci_z_solar_pos, /NO_COPY)
   l_oa.gci_z_solar_pos_ptr = ptr

   ;------------------------------
   ;  GCI Lunar Position coords
   ;------------------------------
   ptr = ptr_new(gci_x_lunar_pos, /NO_COPY)
   l_oa.gci_x_lunar_pos_ptr = ptr

   ptr = ptr_new(gci_y_lunar_pos, /NO_COPY)
   l_oa.gci_y_lunar_pos_ptr = ptr

   ptr = ptr_new(gci_z_lunar_pos, /NO_COPY)
   l_oa.gci_z_lunar_pos_ptr = ptr

   ;------------------------------
   ;   GCM(GSM) X,Y,Z coords
   ;------------------------------
   ptr = ptr_new(gsm_x_pos, /NO_COPY)
   l_oa.gsm_x_pos_ptr = ptr

   ptr = ptr_new(gsm_x_vel, /NO_COPY)
   l_oa.gsm_x_vel_ptr = ptr

   ptr = ptr_new(gsm_y_pos, /NO_COPY)
   l_oa.gsm_y_pos_ptr = ptr

   ptr = ptr_new(gsm_y_vel, /NO_COPY)
   l_oa.gsm_y_vel_ptr = ptr

   ptr = ptr_new(gsm_z_pos, /NO_COPY)
   l_oa.gsm_z_pos_ptr = ptr

   ptr = ptr_new(gsm_z_vel, /NO_COPY)
   l_oa.gsm_z_vel_ptr = ptr

   ;------------------------------
   ;   GCI X,Y,Z coords
   ;------------------------------
   ptr = ptr_new(gci_x_pos, /NO_COPY)
   l_oa.gci_x_pos_ptr = ptr

   ptr = ptr_new(gci_x_vel, /NO_COPY)
   l_oa.gci_x_vel_ptr = ptr

   ptr = ptr_new(gci_y_pos, /NO_COPY)
   l_oa.gci_y_pos_ptr = ptr

   ptr = ptr_new(gci_y_vel, /NO_COPY)
   l_oa.gci_y_vel_ptr = ptr

   ptr = ptr_new(gci_z_pos, /NO_COPY)
   l_oa.gci_z_pos_ptr = ptr

   ptr = ptr_new(gci_z_vel, /NO_COPY)
   l_oa.gci_z_vel_ptr = ptr


   ;------------------------------
   ;   GSE X,Y,Z coords
   ;------------------------------
   ptr = ptr_new(gse_x_pos, /NO_COPY)
   l_oa.gse_x_pos_ptr = ptr

   ptr = ptr_new(gse_x_vel, /NO_COPY)
   l_oa.gse_x_vel_ptr = ptr

   ptr = ptr_new(gse_y_pos, /NO_COPY)
   l_oa.gse_y_pos_ptr = ptr

   ptr = ptr_new(gse_y_vel, /NO_COPY)
   l_oa.gse_y_vel_ptr = ptr

   ptr = ptr_new(gse_z_pos, /NO_COPY)
   l_oa.gse_z_pos_ptr = ptr

   ptr = ptr_new(gse_z_vel, /NO_COPY)
   l_oa.gse_z_vel_ptr = ptr


   ptr = ptr_new (start_array, /NO_COPY)
   l_oa.startUdfTimePtr = ptr

   ptr = ptr_new (stop_array, /NO_COPY)
   l_oa.stopUdfTimePtr = ptr


   RETURN, SUCCESS
END


;-------------------------------------------------------------------------------
;  Procedure: l_OATerm
;
;  Description: 
;      Free pointers associated with OA data 
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct*       I        OA data parameters
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------

PRO l_OATerm, l_oa

   if (ptr_valid (l_oa.tme_ptr)) then begin
       ptr_free, l_oa.tme_ptr
   endif

   if (ptr_valid (l_oa.mag_field)) then begin
       ptr_free, l_oa.mag_field
   endif

   if (ptr_valid (l_oa.lena_sun_sectPtr)) then begin
       ptr_free, l_oa.lena_sun_sectPtr
   endif

   if (ptr_valid (l_oa.lena_moon_sectPtr)) then begin
       ptr_free, l_oa.lena_moon_sectPtr
   endif

   if (ptr_valid (l_oa.lena_ram_sectPtr)) then begin
       ptr_free, l_oa.lena_ram_sectPtr
   endif

   if (ptr_valid (l_oa.lena_mag_fieldPtr)) then begin
       ptr_free, l_oa.lena_mag_fieldPtr
   endif

   if (ptr_valid (l_oa.lsh_ptr)) then begin
       ptr_free, l_oa.lsh_ptr
   endif

   ;--------------------------------------
   ;  free gci solar position pointers 
   ;--------------------------------------
   if (ptr_valid (l_oa.gci_x_solar_pos_ptr)) then begin
       ptr_free, l_oa.gci_x_solar_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_y_solar_pos_ptr)) then begin
       ptr_free, l_oa.gci_y_solar_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_z_solar_pos_ptr)) then begin
       ptr_free, l_oa.gci_z_solar_pos_ptr
   endif

   ;--------------------------------------
   ;  free gci lunar position pointers 
   ;--------------------------------------
   if (ptr_valid (l_oa.gci_x_lunar_pos_ptr)) then begin
       ptr_free, l_oa.gci_x_lunar_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_y_lunar_pos_ptr)) then begin
       ptr_free, l_oa.gci_y_lunar_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_z_lunar_pos_ptr)) then begin
       ptr_free, l_oa.gci_z_lunar_pos_ptr
   endif

   ;--------------------------------------
   ;  free gsm pointers
   ;--------------------------------------
   if (ptr_valid (l_oa.gsm_x_pos_ptr)) then begin
       ptr_free, l_oa.gsm_x_pos_ptr
   endif

   if (ptr_valid (l_oa.gsm_x_vel_ptr)) then begin
       ptr_free, l_oa.gsm_x_vel_ptr
   endif

   if (ptr_valid (l_oa.gsm_y_pos_ptr)) then begin
       ptr_free, l_oa.gsm_y_pos_ptr
   endif

   if (ptr_valid (l_oa.gsm_y_vel_ptr)) then begin
       ptr_free, l_oa.gsm_y_vel_ptr
   endif

   if (ptr_valid (l_oa.gsm_z_pos_ptr)) then begin
       ptr_free, l_oa.gsm_z_pos_ptr
   endif

   if (ptr_valid (l_oa.gsm_z_vel_ptr)) then begin
       ptr_free, l_oa.gsm_z_vel_ptr
   endif

   ;--------------------------------------
   ;  free gci pointers
   ;--------------------------------------

   if (ptr_valid (l_oa.gci_x_pos_ptr)) then begin
       ptr_free, l_oa.gci_x_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_x_vel_ptr)) then begin
       ptr_free, l_oa.gci_x_vel_ptr
   endif

   if (ptr_valid (l_oa.gci_y_pos_ptr)) then begin
       ptr_free, l_oa.gci_y_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_y_vel_ptr)) then begin
       ptr_free, l_oa.gci_y_vel_ptr
   endif

   if (ptr_valid (l_oa.gci_z_pos_ptr)) then begin
       ptr_free, l_oa.gci_z_pos_ptr
   endif

   if (ptr_valid (l_oa.gci_z_vel_ptr)) then begin
       ptr_free, l_oa.gci_z_vel_ptr
   endif

   ;--------------------------------------
   ;  free gse pointers
   ;--------------------------------------

   if (ptr_valid (l_oa.gse_x_pos_ptr)) then begin
       ptr_free, l_oa.gse_x_pos_ptr
   endif

   if (ptr_valid (l_oa.gse_x_vel_ptr)) then begin
       ptr_free, l_oa.gse_x_vel_ptr
   endif

   if (ptr_valid (l_oa.gse_y_pos_ptr)) then begin
       ptr_free, l_oa.gse_y_pos_ptr
   endif

   if (ptr_valid (l_oa.gse_y_vel_ptr)) then begin
       ptr_free, l_oa.gse_y_vel_ptr
   endif

   if (ptr_valid (l_oa.gse_z_pos_ptr)) then begin
       ptr_free, l_oa.gse_z_pos_ptr
   endif

   if (ptr_valid (l_oa.gse_z_vel_ptr)) then begin
       ptr_free, l_oa.gse_z_vel_ptr
   endif

   ;--------------------------------------
   ;  UDF Time pointers
   ;--------------------------------------
   if (ptr_valid (l_oa.startUdfTimePtr)) then begin
       ptr_free, l_oa.startUdfTimePtr
   endif

   if (ptr_valid (l_oa.stopUdfTimePtr)) then begin
       ptr_free, l_oa.stopUdfTimePtr
   endif


RETURN
END

;-------------------------------------------------------------------------------
;  Procedure: l_OAPlot
;
;  Description: 
;      Write the OA data label under each major time tick on the X-axis.
;
;  Return Value:  type =
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct*       I        OA data parameters
;  l_pltpos              struct        I        generic plot parameters
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH                               v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_OAPlot, l_oa, l_pltpos, btime

   ;------------------------------------------------------------
   ;  setup label positions 
   ;------------------------------------------------------------
   ntcks   = l_pltpos.nticks 
   lb_xpmn = l_pltpos.xpltmn
   lb_xpmx = l_pltpos.xpltmx
   lb_ypmn = l_pltpos.ypltmn
   lb_ypmx = l_pltpos.ypltmx

   lb_xdel = 0.03
   lb_ydel = 0.018

   lb_xpos_base = lb_xpmn
   lb_ypos_base = lb_ypmn


   ;------------------------------------------------------------
   ;  add documentation 
   ;------------------------------------------------------------
   lb_xpos = 0.01
   lb_ypos = lb_ypos_base

   xyouts, lb_xpos, lb_ypos, '!5Lshell', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'RE', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'MLT', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'MLat', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'InvLat', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel


   ;----------------------------------------
   ; determine length of x-axis 
   ;----------------------------------------
   x_axis_len = lb_xpmx-lb_xpmn
   xtck_delta = x_axis_len/ntcks

   ;------------------------------------------------------------
   ; the begin/end time of actual data on plot in seconds 
   ;------------------------------------------------------------
   tmin = l_pltpos.min_time_axis_len
   tmax = l_pltpos.max_time_axis_len

   ;------------------------------------------------------------
   ; find the length of the time based on begin/end time of data
   ;------------------------------------------------------------
   tdel = tmax - tmin

   ;------------------------------------------------------------
   ; find the size of the plot based on the x axis size and
   ; size of the drawable area
   ;------------------------------------------------------------
   xdel = l_pltpos.axis_size/l_pltpos.xin
   xpltmn = 0.92/l_pltpos.xin


   ;------------------------------------------------------------
   ; find the index first time label that will be shown on the plots
   ; that will correspond to the OA data
   ;------------------------------------------------------------
   oa_num = 0L
   first_time_label = l_pltpos.first_time_label

   indx_of_first_label = ((first_time_label - (*l_oa.tme_ptr)[0])/120.)
   oaindx = indx_of_first_label
   if (indx_of_first_label lt 0) then oaindx = 0


   oa_delta = 0.0 
   oa_num = N_ELEMENTS((*l_oa.gsm_x_pos_ptr)[*]) - 1     

   ;------------------------------------------------------------ 
   ; take the delta between each major tick mark, convert it to 
   ; seconds and then find the number of spins there are in 
   ; that interval 
   ;------------------------------------------------------------ 
   oa_delta = (l_pltpos.xtick_delta*60.)/120.

   min_x_pos = lb_xpmn - 0.001
   max_x_pos = lb_xpmx + 0.001

   ;------------------------------------------------------------ 
   ; draw the labels on the plot
   ;------------------------------------------------------------ 
   for ii = 0L, ntcks do begin

      lb_ypos = lb_ypos_base

      hr = l_pltpos.xtick_locs[ii]
      lb_xpos = double((hr - tmin)/ tdel * xdel + xpltmn)


      ;------------------------------------------------------------ 
      ; make sure the labels will not exceed the x axis length
      ;------------------------------------------------------------ 

      if (lb_xpos ge min_x_pos) AND (lb_xpos le max_x_pos) then begin


         label=strtrim(string(format='(f10.2)', (*l_oa.lsh_ptr)[oaindx]),2)
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=l_pltpos.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel

         ;------------------------------------------------------------ 
         ; find the distance between the satellite and the earth
         ;------------------------------------------------------------ 
         rxsm = ((*l_oa.gsm_x_pos_ptr)[oaindx])/6371.0
         rysm = ((*l_oa.gsm_y_pos_ptr)[oaindx])/6371.0
         rzsm = ((*l_oa.gsm_z_pos_ptr)[oaindx])/6371.0

         val = sqrt(rxsm*rxsm+rysm*rysm+rzsm*rzsm)

         label=strtrim(string(format='(f10.2)', val),2)
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=l_pltpos.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel


         time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oaindx], btime[0])
         RECALC, time_array[0], time_array[1], time_array[2], $
                 time_array[3], time_array[4]

         ;------------------------------------------------
         ; convert the gsm coordinates to sm
         ;------------------------------------------------
         SMGSM, xsm, ysm, zsm, $
                (*l_oa.gsm_x_pos_ptr)[oaindx], $
                (*l_oa.gsm_y_pos_ptr)[oaindx], $
                (*l_oa.gsm_z_pos_ptr)[oaindx], $
                -1 

         rsm = sqrt(xsm*xsm+ysm*ysm+zsm*zsm)


         ;------------------------------------------------------------ 
         ; calculate mlt in hours 
         ;------------------------------------------------------------ 
         val = atan(ysm,xsm)*12./!pi + 12.
         label=strtrim(string(format='(f10.2)', val),2)
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=l_pltpos.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel

         ;------------------------------------------------------------ 
         ; calculate mlat in degrees 
         ;------------------------------------------------------------ 
         val = asin(zsm/rsm)*180./!pi 
         label=strtrim(string(format='(f10.2)', val),2)
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=l_pltpos.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel

         ;------------------------------------------------------------ 
         ; calculate the invariant latitude in degrees
         ; multiply by 57.3 will give degrees (180/pi)
         ;------------------------------------------------------------ 
         val = (acos(sqrt(1./((*l_oa.lsh_ptr)[oaindx])))) * 180./!pi
         label=strtrim(string(format='(f10.2)', val),2)
         xyouts, lb_xpos, lb_ypos, label, /normal, $
                 size=l_pltpos.sm_font, alignment=0.5
         lb_ypos = lb_ypos - lb_ydel


         ;------------------------------------------------------------ 
         ; increment index to get next OA element
         ; check if oaindx exceeds the total number of elements
         ; available in the array, if so, usually this occurs for
         ; a 24 hour period, the last spin on the plot matches the
         ; last element in the oa array, it is correct to use
         ; the last element
         ;------------------------------------------------------------ 
         oaindx = oaindx + oa_delta
         if (oaindx gt oa_num) then oaindx = oa_num

      endif



   endfor

   ;------------------------------------------------------------
   ; add units to orbit data 
   ;------------------------------------------------------------
   lb_xpos = lb_xpos_base + (ntcks*xtck_delta) + 0.05
   lb_ypos = lb_ypos_base


   xyouts, lb_xpos, lb_ypos, ' ', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'Re', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'hrs', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'degs', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel

   xyouts, lb_xpos, lb_ypos, 'degs', size=l_pltpos.sm_font, /NORMAL
   lb_ypos = lb_ypos - lb_ydel


   l_pltpos.doc_ymn = lb_ypos
   l_pltpos.doc_ymx = lb_ypmx - l_pltpos.yspace


RETURN
END


;-------------------------------------------------------------------------------
;  Procedure : l_OAEarthSize
;
;  Description:  Draw lines to indicate the size of the earth 
;
;  Return Value:  type = int
;  Value                         Description
;  -------------------------     ------------------------------
;  None
;
;  Argument List:
;  Name                  Type         Use       Description
;  -----------------     ------       ---       ---------------------------
;  l_oa                  struct *      I        OA data parameters
;  btime                 long[]        I        UDF begin time [yr,doy,hh,mm,ss]
;  theta_top             float[]       O        top earth line
;  theta_bot             float[]       O        bottom earth line  
;  nspins                long          I        total number of spins
;  oa_time               float[]       O        OA UDF time in seconds
;
;
;  External Variables:
;  Source              Name            Type        Use        Description
;  ---------------     ---------       ----        ----       --------------------
;  None
;
;  Development History:
;  Author             Date           Build            Description of Change
;  --------------   ---------       --------         -------------------------
;  ELH              6/15/00          v1.0             Original implementation
;-------------------------------------------------------------------------------


PRO l_OAEarthSize, l_oa, btime, theta_top, theta_bot, nspins, oa_time


   ;----------------------------------------
   ; number of synced oa data
   ;----------------------------------------
   oa_num = N_ELEMENTS((*l_oa.gsm_x_pos_ptr)[*])    
   N_ORBIT_PTS = oa_num 
   nspins = N_ORBIT_PTS

   oa_delta = 0.0

   ;----------------------------------------
   ; float to get closest OA/LENA match
   ;----------------------------------------
   oa_delta = oa_num/N_ORBIT_PTS

   oaindx = 0 
   oa_time = dblarr(N_ORBIT_PTS)
   theta_top = fltarr(N_ORBIT_PTS)
   theta_bot = fltarr(N_ORBIT_PTS)
   Re_e = 6378.
   Re = 1.

   for ii = 0,N_ORBIT_PTS-1 do begin

      time_array = ConvSec2Arr ((*l_oa.tme_ptr)[oaindx], btime[0])
      RECALC, time_array[0], time_array[1], time_array[2], $
              time_array[3], time_array[4]

      ;------------------------------------------------
      ; convert the gsm coordinates to sm
      ;------------------------------------------------
      SMGSM, xsm, ysm, zsm, $
             (*l_oa.gsm_x_pos_ptr)[oaindx], $
             (*l_oa.gsm_y_pos_ptr)[oaindx], $
             (*l_oa.gsm_z_pos_ptr)[oaindx], $
             -1 

      xsm = xsm / Re_e 
      ysm = ysm / Re_e
      zsm = zsm / Re_e

      rsm = sqrt(xsm*xsm+ysm*ysm+zsm*zsm)

      theta_top[ii] = (asin (1.0/rsm)) * (180./!pi)
      theta_bot[ii] = theta_top[ii] * (-1)

      oa_time[ii] = (*l_oa.tme_ptr)[oaindx]

      oaindx = ROUND(ii*oa_delta)
      tmestr = met2utcstr (oa_time[ii], btime[0], 0L)

   endfor


RETURN
END
